/* This file is an image processing operation for GEGL
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Copyright (C) 2013 Andreas Fischer (andreas.fischer@student.kit.edu)
 *
 * Based on the GIMP apply-canvas filter
 * Copyright (C) 1997 Karl-Johan Andersson (t96kja@student.tdb.uu.se)
 */

#include "config.h"
#include <glib/gi18n-lib.h>

#ifdef GEGL_PROPERTIES

enum_start (gegl_texturize_canvas_direction)
  enum_value (GEGL_TEXTURIZE_CANVAS_DIRECTION_TOP_RIGHT,    "top-right",
              N_("Top-right"))
  enum_value (GEGL_TEXTURIZE_CANVAS_DIRECTION_TOP_LEFT,     "top-left",
              N_("Top-left"))
  enum_value (GEGL_TEXTURIZE_CANVAS_DIRECTION_BOTTOM_LEFT,  "bottom-left",
              N_("Bottom-left"))
  enum_value (GEGL_TEXTURIZE_CANVAS_DIRECTION_BOTTOM_RIGHT, "bottom-right",
              N_("Bottom-right"))
enum_end (GeglTexturizeCanvasDirection)

property_enum (direction, _("Direction"),
    GeglTexturizeCanvasDirection, gegl_texturize_canvas_direction,
    GEGL_TEXTURIZE_CANVAS_DIRECTION_TOP_RIGHT)
    description(_("Position of the light source which lightens the canvas: "
               "Top-right, Top-left, Bottom-left or Bottom-right"))

property_int (depth, _("Depth"), 4)
    description (_("Apparent depth of the rendered canvas effect; "
                "from 1 (very flat) to 50 (very deep)"))
    value_range (1, 50)

#else

#define GEGL_OP_POINT_FILTER
#define GEGL_OP_NAME     texturize_canvas
#define GEGL_OP_C_SOURCE texturize-canvas.c

#include "gegl-op.h"

/* This array contains 16384 floats interpreted as a 128x128 texture */
static const gfloat sdata [128 * 128] =
{
+0.000000000000000, +0.019607843137255, -0.011764705882353, -0.050980392156863,
+0.050980392156863, +0.039215686274510, -0.043137254901961, +0.023529411764706,
-0.043137254901961, -0.054901960784314, +0.113725490196078, +0.168627450980392,
-0.070588235294118, -0.156862745098039, +0.094117647058824, +0.015686274509804,
-0.125490196078431, +0.101960784313725, +0.152941176470588, -0.133333333333333,
-0.062745098039216, +0.098039215686275, -0.086274509803922, +0.047058823529412,
+0.007843137254902, -0.156862745098039, +0.035294117647059, +0.180392156862745,
-0.050980392156863, -0.184313725490196, +0.066666666666667, +0.074509803921569,
-0.082352941176471, +0.011764705882353, +0.200000000000000, -0.086274509803922,
-0.180392156862745, +0.058823529411765, +0.000000000000000, -0.058823529411765,
+0.019607843137255, +0.031372549019608, +0.000000000000000, +0.105882352941176,
-0.019607843137255, -0.117647058823529, +0.011764705882353, -0.019607843137255,
-0.015686274509804, +0.094117647058824, +0.000000000000000, -0.078431372549020,
+0.058823529411765, +0.031372549019608, -0.070588235294118, +0.019607843137255,
+0.200000000000000, -0.050980392156863, -0.200000000000000, +0.062745098039216,
+0.090196078431373, -0.082352941176471, -0.011764705882353, +0.113725490196078,
-0.070588235294118, -0.086274509803922, +0.062745098039216, +0.062745098039216,
-0.011764705882353, +0.031372549019608, -0.007843137254902, -0.121568627450980,
+0.027450980392157, +0.105882352941176, -0.062745098039216, -0.050980392156863,
+0.117647058823529, -0.035294117647059, -0.125490196078431, +0.105882352941176,
+0.082352941176471, +0.011764705882353, +0.105882352941176, -0.043137254901961,
-0.117647058823529, +0.070588235294118, +0.031372549019608, -0.149019607843137,
-0.031372549019608, +0.023529411764706, -0.027450980392157, -0.027450980392157,
-0.011764705882353, +0.031372549019608, -0.007843137254902, +0.074509803921569,
+0.074509803921569, -0.117647058823529, -0.047058823529412, +0.164705882352941,
-0.062745098039216, -0.125490196078431, +0.035294117647059, -0.039215686274510,
-0.039215686274510, +0.039215686274510, +0.050980392156863, +0.007843137254902,
-0.019607843137255, -0.031372549019608, -0.015686274509804, +0.105882352941176,
+0.007843137254902, -0.113725490196078, +0.039215686274510, +0.047058823529412,
-0.050980392156863, -0.031372549019608, +0.000000000000000, +0.160784313725490,
+0.000000000000000, -0.168627450980392, +0.019607843137255, +0.011764705882353,
+0.023529411764706, +0.015686274509804, -0.031372549019608, +0.000000000000000,
-0.039215686274510, -0.019607843137255, -0.094117647058824, -0.184313725490196,
-0.035294117647059, -0.027450980392157, -0.152941176470588, -0.011764705882353,
-0.066666666666667, -0.125490196078431, +0.094117647058824, +0.160784313725490,
-0.098039215686275, -0.258823529411765, -0.047058823529412, -0.066666666666667,
-0.184313725490196, +0.050980392156863, +0.113725490196078, -0.188235294117647,
-0.121568627450980, +0.094117647058824, -0.109803921568627, +0.007843137254902,
+0.011764705882353, -0.227450980392157, -0.027450980392157, +0.180392156862745,
-0.050980392156863, -0.192156862745098, +0.023529411764706, -0.050980392156863,
-0.203921568627451, -0.074509803921569, +0.113725490196078, -0.117647058823529,
-0.207843137254902, -0.043137254901961, -0.168627450980392, -0.196078431372549,
-0.031372549019608, -0.043137254901961, -0.105882352941176, +0.082352941176471,
-0.062745098039216, -0.200000000000000, -0.058823529411765, -0.164705882352941,
-0.160784313725490, +0.047058823529412, -0.086274509803922, -0.219607843137255,
-0.031372549019608, -0.070588235294118, -0.207843137254902, -0.113725490196078,
+0.156862745098039, -0.031372549019608, -0.239215686274510, +0.023529411764706,
+0.105882352941176, -0.043137254901961, +0.007843137254902, +0.160784313725490,
-0.078431372549020, -0.196078431372549, +0.031372549019608, +0.066666666666667,
+0.043137254901961, +0.074509803921569, -0.078431372549020, -0.184313725490196,
-0.011764705882353, +0.066666666666667, -0.125490196078431, -0.086274509803922,
+0.078431372549020, -0.070588235294118, -0.086274509803922, +0.035294117647059,
-0.031372549019608, -0.074509803921569, +0.094117647058824, +0.027450980392157,
-0.074509803921569, +0.105882352941176, +0.113725490196078, -0.078431372549020,
-0.031372549019608, +0.023529411764706, -0.047058823529412, -0.054901960784314,
-0.039215686274510, -0.058823529411765, -0.109803921568627, +0.050980392156863,
+0.031372549019608, -0.188235294117647, -0.082352941176471, +0.105882352941176,
-0.109803921568627, -0.149019607843137, +0.003921568627451, -0.121568627450980,
-0.094117647058824, +0.007843137254902, -0.019607843137255, -0.050980392156863,
-0.035294117647059, -0.086274509803922, -0.117647058823529, +0.062745098039216,
-0.074509803921569, -0.266666666666667, -0.031372549019608, +0.015686274509804,
-0.168627450980392, -0.156862745098039, -0.113725490196078, +0.019607843137255,
-0.094117647058824, -0.200000000000000, -0.011764705882353, -0.066666666666667,
-0.031372549019608, +0.043137254901961, -0.066666666666667, -0.149019607843137,
-0.015686274509804, +0.043137254901961, +0.050980392156863, -0.043137254901961,
+0.047058823529412, +0.050980392156863, -0.117647058823529, +0.054901960784314,
+0.007843137254902, -0.125490196078431, +0.062745098039216, +0.164705882352941,
+0.043137254901961, -0.023529411764706, +0.070588235294118, -0.003921568627451,
-0.156862745098039, +0.066666666666667, +0.266666666666667, +0.019607843137255,
-0.054901960784314, +0.168627450980392, +0.043137254901961, +0.054901960784314,
+0.113725490196078, -0.125490196078431, -0.019607843137255, +0.149019607843137,
-0.066666666666667, -0.152941176470588, +0.125490196078431, +0.070588235294118,
-0.149019607843137, -0.086274509803922, +0.043137254901961, -0.062745098039216,
-0.062745098039216, +0.062745098039216, -0.039215686274510, -0.176470588235294,
-0.011764705882353, +0.035294117647059, -0.164705882352941, +0.074509803921569,
+0.058823529411765, -0.117647058823529, +0.094117647058824, +0.015686274509804,
-0.117647058823529, +0.137254901960784, +0.094117647058824, -0.152941176470588,
+0.043137254901961, +0.109803921568627, -0.050980392156863, -0.070588235294118,
+0.168627450980392, +0.047058823529412, -0.227450980392157, -0.031372549019608,
+0.058823529411765, -0.141176470588235, -0.125490196078431, +0.176470588235294,
+0.062745098039216, -0.152941176470588, +0.015686274509804, -0.031372549019608,
-0.043137254901961, +0.152941176470588, -0.023529411764706, -0.117647058823529,
+0.043137254901961, +0.039215686274510, -0.125490196078431, -0.023529411764706,
+0.113725490196078, -0.101960784313725, -0.015686274509804, +0.117647058823529,
+0.015686274509804, +0.007843137254902, +0.184313725490196, +0.062745098039216,
-0.086274509803922, +0.117647058823529, +0.137254901960784, -0.105882352941176,
-0.090196078431373, +0.023529411764706, -0.125490196078431, -0.117647058823529,
-0.015686274509804, -0.066666666666667, -0.113725490196078, +0.105882352941176,
+0.074509803921569, -0.105882352941176, +0.015686274509804, +0.082352941176471,
-0.082352941176471, -0.050980392156863, +0.078431372549020, -0.094117647058824,
-0.082352941176471, +0.058823529411765, -0.031372549019608, -0.035294117647059,
+0.094117647058824, -0.007843137254902, -0.211764705882353, +0.070588235294118,
+0.137254901960784, -0.203921568627451, -0.043137254901961, +0.101960784313725,
-0.137254901960784, -0.160784313725490, -0.074509803921569, -0.007843137254902,
-0.105882352941176, -0.145098039215686, +0.047058823529412, -0.074509803921569,
-0.109803921568627, +0.062745098039216, +0.007843137254902, -0.125490196078431,
-0.109803921568627, -0.078431372549020, -0.094117647058824, -0.070588235294118,
+0.074509803921569, -0.031372549019608, -0.176470588235294, +0.015686274509804,
-0.043137254901961, -0.117647058823529, +0.050980392156863, +0.047058823529412,
-0.074509803921569, +0.019607843137255, +0.117647058823529, +0.000000000000000,
-0.156862745098039, -0.050980392156863, +0.164705882352941, -0.007843137254902,
-0.113725490196078, +0.023529411764706, -0.011764705882353, +0.050980392156863,
+0.094117647058824, -0.101960784313725, +0.003921568627451, +0.192156862745098,
-0.039215686274510, -0.113725490196078, +0.149019607843137, +0.145098039215686,
+0.027450980392157, +0.113725490196078, +0.062745098039216, -0.082352941176471,
+0.050980392156863, +0.145098039215686, +0.062745098039216, -0.105882352941176,
-0.031372549019608, +0.117647058823529, -0.003921568627451, +0.113725490196078,
+0.109803921568627, -0.074509803921569, +0.070588235294118, +0.023529411764706,
-0.129411764705882, +0.086274509803922, +0.180392156862745, -0.023529411764706,
+0.062745098039216, +0.149019607843137, +0.043137254901961, +0.023529411764706,
+0.176470588235294, +0.113725490196078, -0.086274509803922, +0.000000000000000,
+0.094117647058824, -0.050980392156863, -0.117647058823529, +0.101960784313725,
+0.074509803921569, -0.062745098039216, +0.101960784313725, +0.101960784313725,
+0.031372549019608, +0.211764705882353, -0.023529411764706, -0.180392156862745,
+0.113725490196078, +0.125490196078431, -0.031372549019608, +0.058823529411765,
+0.168627450980392, -0.011764705882353, -0.019607843137255, +0.168627450980392,
+0.086274509803922, -0.058823529411765, +0.137254901960784, +0.007843137254902,
-0.200000000000000, +0.000000000000000, +0.125490196078431, -0.027450980392157,
-0.047058823529412, +0.168627450980392, +0.007843137254902, -0.129411764705882,
+0.047058823529412, +0.054901960784314, -0.011764705882353, +0.160784313725490,
+0.094117647058824, -0.082352941176471, +0.090196078431373, +0.149019607843137,
-0.023529411764706, +0.011764705882353, +0.145098039215686, -0.011764705882353,
-0.074509803921569, +0.105882352941176, +0.043137254901961, -0.043137254901961,
+0.094117647058824, +0.117647058823529, -0.086274509803922, +0.031372549019608,
+0.196078431372549, -0.117647058823529, -0.062745098039216, +0.250980392156863,
+0.137254901960784, +0.062745098039216, +0.188235294117647, +0.184313725490196,
-0.062745098039216, -0.105882352941176, +0.141176470588235, +0.050980392156863,
+0.000000000000000, +0.180392156862745, +0.200000000000000, +0.137254901960784,
+0.086274509803922, +0.098039215686275, -0.011764705882353, -0.019607843137255,
+0.160784313725490, +0.031372549019608, -0.094117647058824, +0.062745098039216,
-0.090196078431373, -0.137254901960784, +0.137254901960784, +0.152941176470588,
-0.160784313725490, -0.078431372549020, +0.149019607843137, +0.031372549019608,
-0.180392156862745, -0.125490196078431, +0.105882352941176, -0.062745098039216,
-0.137254901960784, -0.015686274509804, -0.094117647058824, -0.015686274509804,
+0.015686274509804, -0.196078431372549, -0.137254901960784, +0.086274509803922,
-0.074509803921569, -0.192156862745098, +0.050980392156863, +0.035294117647059,
-0.078431372549020, +0.125490196078431, +0.090196078431373, -0.207843137254902,
-0.066666666666667, +0.105882352941176, +0.000000000000000, -0.129411764705882,
-0.137254901960784, -0.137254901960784, -0.105882352941176, +0.074509803921569,
+0.023529411764706, -0.149019607843137, +0.011764705882353, +0.019607843137255,
-0.184313725490196, -0.031372549019608, +0.062745098039216, -0.149019607843137,
-0.043137254901961, +0.019607843137255, -0.176470588235294, -0.094117647058824,
+0.039215686274510, -0.086274509803922, -0.117647058823529, +0.000000000000000,
+0.000000000000000, -0.086274509803922, -0.062745098039216, -0.015686274509804,
-0.149019607843137, -0.235294117647059, -0.117647058823529, -0.082352941176471,
-0.117647058823529, +0.050980392156863, -0.062745098039216, -0.254901960784314,
-0.086274509803922, -0.074509803921569, -0.184313725490196, -0.078431372549020,
-0.031372549019608, -0.113725490196078, -0.200000000000000, -0.129411764705882,
-0.117647058823529, -0.145098039215686, +0.086274509803922, -0.043137254901961,
-0.262745098039216, -0.149019607843137, -0.062745098039216, -0.145098039215686,
-0.211764705882353, -0.050980392156863, -0.007843137254902, -0.125490196078431,
-0.109803921568627, -0.117647058823529, -0.121568627450980, -0.035294117647059,
-0.074509803921569, -0.172549019607843, -0.117647058823529, +0.023529411764706,
-0.113725490196078, -0.125490196078431, +0.015686274509804, -0.086274509803922,
-0.149019607843137, -0.039215686274510, -0.094117647058824, -0.074509803921569,
+0.050980392156863, -0.031372549019608, -0.160784313725490, -0.031372549019608,
+0.188235294117647, -0.074509803921569, -0.207843137254902, +0.000000000000000,
-0.031372549019608, -0.054901960784314, +0.117647058823529, +0.188235294117647,
-0.086274509803922, -0.200000000000000, -0.043137254901961, -0.149019607843137,
-0.137254901960784, +0.074509803921569, +0.027450980392157, -0.003921568627451,
+0.129411764705882, +0.149019607843137, +0.082352941176471, +0.062745098039216,
+0.200000000000000, +0.149019607843137, +0.078431372549020, +0.231372549019608,
+0.113725490196078, -0.058823529411765, +0.109803921568627, +0.266666666666667,
-0.050980392156863, -0.117647058823529, +0.219607843137255, +0.262745098039216,
+0.043137254901961, -0.003921568627451, +0.176470588235294, +0.094117647058824,
+0.039215686274510, +0.133333333333333, +0.066666666666667, +0.109803921568627,
+0.168627450980392, +0.015686274509804, -0.011764705882353, +0.105882352941176,
-0.027450980392157, -0.109803921568627, +0.117647058823529, +0.121568627450980,
-0.035294117647059, +0.109803921568627, +0.200000000000000, -0.137254901960784,
-0.098039215686275, +0.176470588235294, +0.164705882352941, +0.117647058823529,
+0.121568627450980, -0.117647058823529, -0.211764705882353, +0.090196078431373,
+0.074509803921569, -0.105882352941176, +0.050980392156863, +0.188235294117647,
+0.019607843137255, +0.047058823529412, +0.117647058823529, -0.188235294117647,
-0.078431372549020, +0.137254901960784, -0.137254901960784, -0.047058823529412,
+0.168627450980392, -0.007843137254902, -0.035294117647059, +0.125490196078431,
+0.058823529411765, -0.188235294117647, -0.086274509803922, +0.105882352941176,
+0.039215686274510, -0.090196078431373, -0.031372549019608, -0.039215686274510,
-0.074509803921569, +0.105882352941176, +0.117647058823529, +0.019607843137255,
+0.054901960784314, +0.011764705882353, -0.129411764705882, -0.027450980392157,
+0.015686274509804, +0.015686274509804, +0.050980392156863, -0.015686274509804,
-0.098039215686275, -0.023529411764706, +0.156862745098039, +0.039215686274510,
-0.098039215686275, -0.003921568627451, +0.054901960784314, -0.050980392156863,
-0.078431372549020, +0.039215686274510, +0.054901960784314, +0.035294117647059,
+0.000000000000000, -0.117647058823529, -0.023529411764706, +0.098039215686275,
+0.070588235294118, -0.011764705882353, -0.078431372549020, +0.105882352941176,
+0.039215686274510, -0.019607843137255, +0.121568627450980, +0.019607843137255,
-0.043137254901961, +0.078431372549020, -0.007843137254902, +0.000000000000000,
+0.188235294117647, +0.074509803921569, -0.200000000000000, -0.109803921568627,
+0.227450980392157, +0.062745098039216, -0.054901960784314, +0.047058823529412,
-0.101960784313725, -0.141176470588235, +0.050980392156863, +0.141176470588235,
-0.074509803921569, -0.200000000000000, -0.105882352941176, -0.152941176470588,
-0.113725490196078, +0.117647058823529, -0.011764705882353, -0.098039215686275,
-0.062745098039216, -0.043137254901961, -0.172549019607843, -0.188235294117647,
-0.027450980392157, -0.019607843137255, -0.050980392156863, +0.062745098039216,
-0.011764705882353, -0.180392156862745, -0.137254901960784, +0.070588235294118,
-0.090196078431373, -0.200000000000000, -0.007843137254902, -0.015686274509804,
-0.117647058823529, -0.078431372549020, -0.043137254901961, -0.137254901960784,
-0.082352941176471, +0.031372549019608, -0.003921568627451, +0.050980392156863,
+0.066666666666667, -0.054901960784314, +0.054901960784314, +0.207843137254902,
-0.027450980392157, -0.149019607843137, -0.011764705882353, -0.050980392156863,
-0.082352941176471, +0.023529411764706, +0.066666666666667, -0.149019607843137,
-0.105882352941176, +0.082352941176471, +0.019607843137255, +0.078431372549020,
+0.262745098039216, +0.027450980392157, -0.200000000000000, +0.105882352941176,
+0.090196078431373, -0.113725490196078, -0.039215686274510, +0.054901960784314,
-0.019607843137255, +0.000000000000000, +0.211764705882353, -0.031372549019608,
-0.082352941176471, +0.219607843137255, +0.027450980392157, -0.011764705882353,
+0.180392156862745, +0.019607843137255, -0.062745098039216, +0.149019607843137,
+0.168627450980392, -0.023529411764706, -0.007843137254902, +0.168627450980392,
+0.149019607843137, +0.031372549019608, +0.149019607843137, +0.156862745098039,
+0.082352941176471, +0.188235294117647, +0.141176470588235, +0.113725490196078,
+0.211764705882353, +0.203921568627451, +0.078431372549020, +0.168627450980392,
+0.200000000000000, +0.074509803921569, +0.192156862745098, +0.219607843137255,
+0.035294117647059, +0.035294117647059, +0.219607843137255, +0.141176470588235,
-0.011764705882353, +0.125490196078431, +0.286274509803922, +0.141176470588235,
+0.000000000000000, +0.141176470588235, +0.133333333333333, +0.137254901960784,
+0.254901960784314, +0.094117647058824, +0.023529411764706, +0.152941176470588,
+0.156862745098039, +0.062745098039216, +0.019607843137255, +0.133333333333333,
+0.121568627450980, +0.078431372549020, +0.172549019607843, +0.023529411764706,
-0.050980392156863, +0.125490196078431, +0.094117647058824, -0.047058823529412,
+0.094117647058824, +0.243137254901961, -0.062745098039216, -0.129411764705882,
+0.152941176470588, -0.011764705882353, -0.039215686274510, +0.160784313725490,
+0.000000000000000, -0.086274509803922, +0.109803921568627, +0.243137254901961,
+0.141176470588235, +0.094117647058824, +0.027450980392157, -0.078431372549020,
-0.019607843137255, +0.184313725490196, +0.149019607843137, +0.125490196078431,
-0.058823529411765, +0.066666666666667, -0.074509803921569, -0.129411764705882,
-0.023529411764706, -0.074509803921569, -0.113725490196078, +0.047058823529412,
+0.000000000000000, -0.086274509803922, -0.058823529411765, +0.058823529411765,
-0.062745098039216, -0.090196078431373, +0.023529411764706, -0.105882352941176,
-0.207843137254902, -0.047058823529412, +0.003921568627451, -0.211764705882353,
-0.203921568627451, -0.031372549019608, -0.125490196078431, -0.043137254901961,
+0.078431372549020, -0.207843137254902, -0.129411764705882, +0.168627450980392,
-0.031372549019608, -0.196078431372549, -0.074509803921569, -0.192156862745098,
-0.145098039215686, +0.082352941176471, +0.019607843137255, -0.223529411764706,
-0.172549019607843, -0.003921568627451, -0.188235294117647, -0.149019607843137,
+0.145098039215686, -0.011764705882353, -0.274509803921569, -0.019607843137255,
+0.000000000000000, -0.188235294117647, -0.098039215686275, -0.074509803921569,
-0.141176470588235, -0.094117647058824, +0.098039215686275, -0.070588235294118,
-0.117647058823529, +0.109803921568627, -0.019607843137255, -0.074509803921569,
+0.086274509803922, -0.101960784313725, -0.250980392156863, +0.066666666666667,
+0.086274509803922, -0.121568627450980, -0.101960784313725, +0.054901960784314,
-0.031372549019608, -0.188235294117647, -0.047058823529412, -0.082352941176471,
-0.129411764705882, -0.035294117647059, -0.149019607843137, -0.149019607843137,
-0.035294117647059, -0.078431372549020, -0.105882352941176, +0.000000000000000,
+0.113725490196078, -0.086274509803922, -0.105882352941176, +0.035294117647059,
-0.090196078431373, -0.176470588235294, +0.066666666666667, +0.078431372549020,
-0.172549019607843, -0.094117647058824, +0.062745098039216, +0.011764705882353,
-0.192156862745098, -0.113725490196078, -0.082352941176471, -0.133333333333333,
+0.054901960784314, -0.011764705882353, -0.188235294117647, -0.062745098039216,
-0.062745098039216, -0.207843137254902, -0.101960784313725, -0.011764705882353,
-0.105882352941176, -0.066666666666667, +0.078431372549020, -0.113725490196078,
-0.168627450980392, +0.003921568627451, +0.000000000000000, -0.168627450980392,
-0.101960784313725, +0.137254901960784, -0.105882352941176, -0.145098039215686,
+0.101960784313725, -0.101960784313725, -0.215686274509804, -0.074509803921569,
-0.168627450980392, -0.215686274509804, +0.019607843137255, +0.121568627450980,
+0.039215686274510, +0.156862745098039, +0.117647058823529, -0.180392156862745,
-0.125490196078431, +0.007843137254902, -0.066666666666667, +0.043137254901961,
+0.039215686274510, +0.168627450980392, -0.007843137254902, -0.027450980392157,
+0.117647058823529, -0.003921568627451, -0.070588235294118, +0.125490196078431,
+0.094117647058824, +0.062745098039216, +0.145098039215686, +0.180392156862745,
+0.015686274509804, +0.035294117647059, +0.156862745098039, +0.031372549019608,
-0.078431372549020, +0.050980392156863, +0.243137254901961, +0.074509803921569,
-0.058823529411765, +0.074509803921569, -0.003921568627451, -0.003921568627451,
+0.215686274509804, +0.031372549019608, -0.074509803921569, +0.188235294117647,
+0.105882352941176, -0.023529411764706, +0.172549019607843, +0.003921568627451,
-0.086274509803922, +0.200000000000000, +0.133333333333333, -0.113725490196078,
-0.050980392156863, +0.137254901960784, -0.105882352941176, -0.117647058823529,
+0.219607843137255, +0.156862745098039, -0.066666666666667, +0.074509803921569,
+0.054901960784314, -0.133333333333333, -0.007843137254902, +0.047058823529412,
+0.000000000000000, +0.047058823529412, +0.164705882352941, -0.098039215686275,
-0.160784313725490, +0.031372549019608, -0.149019607843137, -0.168627450980392,
+0.168627450980392, +0.098039215686275, -0.156862745098039, +0.109803921568627,
+0.168627450980392, -0.137254901960784, -0.184313725490196, +0.105882352941176,
+0.101960784313725, -0.058823529411765, +0.062745098039216, -0.047058823529412,
-0.074509803921569, +0.090196078431373, -0.074509803921569, -0.074509803921569,
+0.039215686274510, -0.137254901960784, -0.192156862745098, -0.094117647058824,
+0.125490196078431, -0.003921568627451, -0.156862745098039, +0.043137254901961,
+0.023529411764706, -0.105882352941176, +0.094117647058824, +0.109803921568627,
-0.117647058823529, -0.031372549019608, +0.062745098039216, +0.082352941176471,
-0.062745098039216, -0.031372549019608, +0.007843137254902, -0.098039215686275,
+0.066666666666667, +0.062745098039216, -0.078431372549020, +0.094117647058824,
+0.050980392156863, -0.239215686274510, -0.117647058823529, +0.078431372549020,
-0.094117647058824, -0.062745098039216, +0.168627450980392, -0.050980392156863,
-0.141176470588235, +0.082352941176471, +0.113725490196078, -0.098039215686275,
-0.047058823529412, +0.133333333333333, -0.098039215686275, -0.062745098039216,
+0.137254901960784, +0.035294117647059, +0.035294117647059, +0.082352941176471,
-0.094117647058824, -0.262745098039216, -0.043137254901961, +0.070588235294118,
-0.152941176470588, -0.050980392156863, +0.192156862745098, +0.000000000000000,
-0.027450980392157, +0.070588235294118, -0.137254901960784, -0.035294117647059,
-0.003921568627451, +0.094117647058824, -0.117647058823529, -0.129411764705882,
+0.137254901960784, -0.050980392156863, -0.125490196078431, +0.082352941176471,
-0.019607843137255, -0.160784313725490, -0.027450980392157, +0.039215686274510,
-0.145098039215686, -0.058823529411765, +0.176470588235294, +0.109803921568627,
-0.137254901960784, -0.105882352941176, +0.090196078431373, -0.011764705882353,
-0.062745098039216, -0.015686274509804, -0.054901960784314, -0.062745098039216,
-0.031372549019608, -0.074509803921569, -0.160784313725490, +0.098039215686275,
+0.105882352941176, -0.074509803921569, +0.192156862745098, +0.050980392156863,
-0.200000000000000, +0.105882352941176, +0.101960784313725, -0.117647058823529,
+0.019607843137255, +0.203921568627451, -0.074509803921569, -0.243137254901961,
-0.015686274509804, +0.023529411764706, +0.000000000000000, +0.180392156862745,
+0.054901960784314, -0.117647058823529, +0.082352941176471, +0.098039215686275,
-0.007843137254902, +0.043137254901961, +0.207843137254902, -0.047058823529412,
-0.133333333333333, +0.176470588235294, +0.039215686274510, -0.145098039215686,
+0.098039215686275, +0.145098039215686, -0.090196078431373, +0.031372549019608,
+0.160784313725490, +0.007843137254902, -0.066666666666667, +0.121568627450980,
+0.156862745098039, +0.062745098039216, +0.176470588235294, +0.094117647058824,
-0.019607843137255, +0.149019607843137, +0.050980392156863, +0.023529411764706,
+0.211764705882353, +0.023529411764706, -0.098039215686275, -0.007843137254902,
+0.164705882352941, +0.082352941176471, -0.125490196078431, +0.078431372549020,
+0.156862745098039, -0.050980392156863, +0.086274509803922, +0.145098039215686,
-0.031372549019608, +0.050980392156863, +0.098039215686275, +0.094117647058824,
+0.050980392156863, +0.086274509803922, +0.035294117647059, -0.054901960784314,
+0.098039215686275, +0.050980392156863, -0.031372549019608, +0.180392156862745,
+0.125490196078431, -0.149019607843137, -0.137254901960784, +0.074509803921569,
+0.011764705882353, +0.003921568627451, +0.149019607843137, -0.074509803921569,
-0.227450980392157, +0.050980392156863, +0.168627450980392, -0.031372549019608,
+0.050980392156863, +0.160784313725490, -0.105882352941176, -0.039215686274510,
+0.164705882352941, +0.066666666666667, +0.121568627450980, +0.231372549019608,
+0.062745098039216, -0.200000000000000, +0.000000000000000, +0.156862745098039,
-0.086274509803922, -0.113725490196078, +0.031372549019608, +0.000000000000000,
+0.062745098039216, +0.192156862745098, -0.011764705882353, -0.039215686274510,
+0.031372549019608, +0.149019607843137, +0.039215686274510, -0.058823529411765,
+0.196078431372549, +0.066666666666667, -0.058823529411765, +0.149019607843137,
+0.180392156862745, -0.047058823529412, +0.007843137254902, +0.125490196078431,
-0.023529411764706, -0.062745098039216, +0.129411764705882, +0.172549019607843,
-0.105882352941176, -0.105882352941176, +0.094117647058824, +0.000000000000000,
+0.027450980392157, +0.031372549019608, -0.094117647058824, +0.027450980392157,
+0.050980392156863, -0.090196078431373, -0.192156862745098, +0.043137254901961,
+0.015686274509804, -0.211764705882353, +0.121568627450980, +0.129411764705882,
-0.192156862745098, +0.062745098039216, +0.125490196078431, -0.188235294117647,
-0.062745098039216, +0.176470588235294, +0.043137254901961, -0.145098039215686,
-0.062745098039216, -0.125490196078431, -0.223529411764706, +0.090196078431373,
+0.101960784313725, -0.125490196078431, +0.003921568627451, -0.035294117647059,
-0.184313725490196, -0.054901960784314, +0.105882352941176, -0.066666666666667,
-0.109803921568627, +0.129411764705882, +0.039215686274510, -0.098039215686275,
+0.058823529411765, +0.050980392156863, -0.141176470588235, -0.043137254901961,
+0.074509803921569, -0.105882352941176, -0.133333333333333, +0.019607843137255,
-0.023529411764706, -0.168627450980392, -0.070588235294118, -0.035294117647059,
-0.168627450980392, -0.043137254901961, -0.082352941176471, -0.113725490196078,
+0.176470588235294, +0.031372549019608, -0.215686274509804, -0.062745098039216,
+0.039215686274510, -0.011764705882353, -0.156862745098039, -0.054901960784314,
+0.066666666666667, -0.105882352941176, -0.050980392156863, -0.011764705882353,
-0.137254901960784, -0.078431372549020, -0.078431372549020, -0.180392156862745,
-0.082352941176471, +0.078431372549020, -0.098039215686275, -0.156862745098039,
+0.054901960784314, -0.082352941176471, -0.133333333333333, +0.168627450980392,
+0.113725490196078, -0.094117647058824, -0.121568627450980, -0.050980392156863,
-0.117647058823529, -0.145098039215686, +0.015686274509804, -0.031372549019608,
-0.164705882352941, +0.003921568627451, +0.074509803921569, -0.094117647058824,
+0.066666666666667, +0.145098039215686, -0.133333333333333, -0.117647058823529,
+0.054901960784314, -0.074509803921569, -0.164705882352941, -0.007843137254902,
-0.019607843137255, -0.184313725490196, +0.019607843137255, +0.117647058823529,
-0.149019607843137, -0.066666666666667, -0.003921568627451, -0.168627450980392,
-0.074509803921569, +0.137254901960784, -0.011764705882353, -0.058823529411765,
-0.019607843137255, +0.031372549019608, -0.050980392156863, -0.098039215686275,
+0.098039215686275, +0.070588235294118, -0.078431372549020, +0.023529411764706,
+0.058823529411765, -0.070588235294118, -0.027450980392157, +0.117647058823529,
+0.035294117647059, -0.094117647058824, +0.101960784313725, +0.211764705882353,
-0.011764705882353, -0.050980392156863, +0.121568627450980, +0.043137254901961,
+0.039215686274510, +0.176470588235294, +0.062745098039216, +0.086274509803922,
+0.239215686274510, +0.058823529411765, -0.062745098039216, +0.180392156862745,
+0.125490196078431, -0.101960784313725, +0.125490196078431, +0.254901960784314,
-0.027450980392157, +0.066666666666667, +0.180392156862745, -0.105882352941176,
-0.035294117647059, +0.192156862745098, +0.215686274509804, +0.070588235294118,
+0.145098039215686, +0.184313725490196, -0.062745098039216, +0.082352941176471,
+0.156862745098039, -0.094117647058824, +0.035294117647059, +0.117647058823529,
-0.043137254901961, +0.062745098039216, +0.074509803921569, -0.113725490196078,
-0.054901960784314, +0.113725490196078, -0.062745098039216, -0.145098039215686,
+0.113725490196078, +0.113725490196078, -0.070588235294118, +0.019607843137255,
+0.168627450980392, -0.101960784313725, -0.184313725490196, +0.062745098039216,
+0.043137254901961, -0.094117647058824, +0.019607843137255, +0.074509803921569,
-0.011764705882353, +0.101960784313725, +0.066666666666667, -0.054901960784314,
+0.137254901960784, +0.137254901960784, -0.117647058823529, +0.019607843137255,
+0.160784313725490, +0.098039215686275, -0.003921568627451, +0.035294117647059,
+0.133333333333333, +0.062745098039216, +0.121568627450980, +0.113725490196078,
+0.035294117647059, +0.105882352941176, +0.149019607843137, -0.031372549019608,
-0.035294117647059, +0.200000000000000, +0.105882352941176, -0.015686274509804,
+0.125490196078431, +0.039215686274510, -0.047058823529412, +0.145098039215686,
+0.117647058823529, +0.027450980392157, +0.058823529411765, +0.113725490196078,
-0.019607843137255, -0.101960784313725, +0.094117647058824, +0.094117647058824,
-0.019607843137255, +0.094117647058824, +0.125490196078431, -0.039215686274510,
+0.101960784313725, +0.141176470588235, -0.066666666666667, -0.062745098039216,
+0.105882352941176, +0.101960784313725, +0.019607843137255, +0.121568627450980,
+0.090196078431373, -0.074509803921569, +0.094117647058824, +0.243137254901961,
-0.027450980392157, +0.027450980392157, +0.219607843137255, +0.019607843137255,
-0.047058823529412, +0.184313725490196, +0.117647058823529, +0.121568627450980,
-0.062745098039216, -0.011764705882353, -0.109803921568627, -0.207843137254902,
-0.066666666666667, -0.043137254901961, -0.219607843137255, -0.105882352941176,
-0.098039215686275, -0.192156862745098, -0.192156862745098, -0.054901960784314,
-0.098039215686275, -0.223529411764706, -0.074509803921569, -0.011764705882353,
-0.125490196078431, -0.152941176470588, +0.007843137254902, -0.082352941176471,
-0.239215686274510, -0.133333333333333, -0.125490196078431, -0.117647058823529,
+0.082352941176471, -0.062745098039216, -0.176470588235294, +0.086274509803922,
+0.050980392156863, -0.129411764705882, -0.086274509803922, +0.000000000000000,
-0.086274509803922, +0.003921568627451, +0.086274509803922, -0.160784313725490,
-0.168627450980392, -0.003921568627451, +0.074509803921569, +0.011764705882353,
+0.000000000000000, +0.050980392156863, -0.074509803921569, -0.019607843137255,
+0.062745098039216, -0.125490196078431, -0.031372549019608, +0.121568627450980,
-0.023529411764706, +0.121568627450980, +0.109803921568627, -0.149019607843137,
-0.047058823529412, +0.176470588235294, +0.000000000000000, -0.172549019607843,
+0.078431372549020, +0.113725490196078, -0.094117647058824, -0.019607843137255,
+0.168627450980392, -0.023529411764706, -0.117647058823529, +0.156862745098039,
+0.141176470588235, +0.031372549019608, +0.133333333333333, +0.121568627450980,
+0.043137254901961, +0.129411764705882, +0.125490196078431, +0.023529411764706,
+0.047058823529412, +0.086274509803922, -0.054901960784314, -0.035294117647059,
+0.137254901960784, +0.058823529411765, -0.086274509803922, -0.015686274509804,
+0.031372549019608, -0.043137254901961, +0.078431372549020, +0.090196078431373,
-0.027450980392157, +0.094117647058824, +0.168627450980392, +0.015686274509804,
-0.027450980392157, +0.090196078431373, +0.019607843137255, +0.007843137254902,
+0.086274509803922, +0.050980392156863, +0.039215686274510, +0.094117647058824,
-0.011764705882353, -0.094117647058824, +0.031372549019608, +0.188235294117647,
+0.062745098039216, -0.043137254901961, +0.098039215686275, -0.058823529411765,
-0.145098039215686, +0.039215686274510, +0.035294117647059, -0.062745098039216,
+0.027450980392157, -0.003921568627451, -0.160784313725490, -0.066666666666667,
+0.101960784313725, +0.082352941176471, +0.047058823529412, +0.145098039215686,
+0.039215686274510, -0.152941176470588, -0.058823529411765, +0.113725490196078,
-0.023529411764706, -0.133333333333333, +0.007843137254902, -0.023529411764706,
-0.058823529411765, +0.054901960784314, -0.031372549019608, -0.007843137254902,
+0.098039215686275, +0.172549019607843, +0.113725490196078, -0.035294117647059,
+0.062745098039216, +0.156862745098039, +0.015686274509804, +0.086274509803922,
+0.188235294117647, +0.117647058823529, +0.031372549019608, +0.113725490196078,
+0.066666666666667, +0.015686274509804, +0.031372549019608, -0.039215686274510,
-0.054901960784314, -0.043137254901961, +0.141176470588235, +0.137254901960784,
-0.031372549019608, -0.031372549019608, -0.039215686274510, -0.031372549019608,
+0.133333333333333, -0.039215686274510, -0.188235294117647, +0.019607843137255,
-0.031372549019608, -0.105882352941176, -0.039215686274510, -0.078431372549020,
-0.200000000000000, -0.066666666666667, +0.031372549019608, -0.125490196078431,
-0.156862745098039, -0.137254901960784, -0.180392156862745, -0.058823529411765,
+0.007843137254902, -0.149019607843137, -0.211764705882353, -0.070588235294118,
+0.031372549019608, -0.137254901960784, -0.105882352941176, +0.050980392156863,
-0.109803921568627, -0.086274509803922, -0.023529411764706, -0.211764705882353,
-0.172549019607843, -0.023529411764706, -0.101960784313725, -0.239215686274510,
-0.039215686274510, +0.043137254901961, -0.188235294117647, -0.070588235294118,
+0.058823529411765, -0.156862745098039, -0.188235294117647, +0.074509803921569,
+0.000000000000000, -0.160784313725490, -0.039215686274510, -0.090196078431373,
-0.196078431372549, -0.086274509803922, -0.109803921568627, -0.062745098039216,
-0.019607843137255, -0.094117647058824, -0.156862745098039, -0.219607843137255,
-0.035294117647059, -0.054901960784314, -0.231372549019608, -0.196078431372549,
-0.141176470588235, -0.227450980392157, -0.050980392156863, -0.011764705882353,
-0.262745098039216, -0.125490196078431, -0.043137254901961, -0.211764705882353,
-0.168627450980392, -0.019607843137255, -0.168627450980392, -0.239215686274510,
-0.168627450980392, -0.207843137254902, -0.109803921568627, +0.113725490196078,
-0.015686274509804, -0.235294117647059, -0.098039215686275, +0.050980392156863,
-0.101960784313725, -0.168627450980392, +0.043137254901961, -0.113725490196078,
-0.200000000000000, -0.011764705882353, -0.094117647058824, -0.117647058823529,
+0.031372549019608, +0.027450980392157, -0.145098039215686, -0.105882352941176,
-0.003921568627451, -0.168627450980392, -0.188235294117647, +0.035294117647059,
-0.027450980392157, -0.250980392156863, -0.203921568627451, -0.105882352941176,
-0.129411764705882, -0.172549019607843, -0.086274509803922, -0.156862745098039,
-0.117647058823529, -0.003921568627451, -0.145098039215686, -0.058823529411765,
-0.054901960784314, +0.027450980392157, -0.023529411764706, -0.086274509803922,
+0.039215686274510, +0.039215686274510, -0.050980392156863, +0.011764705882353,
+0.117647058823529, +0.043137254901961, +0.011764705882353, +0.062745098039216,
-0.043137254901961, +0.019607843137255, +0.156862745098039, +0.094117647058824,
-0.023529411764706, -0.086274509803922, +0.062745098039216, +0.149019607843137,
+0.074509803921569, +0.133333333333333, +0.156862745098039, +0.098039215686275,
+0.223529411764706, +0.062745098039216, -0.090196078431373, +0.145098039215686,
+0.137254901960784, +0.054901960784314, +0.188235294117647, +0.219607843137255,
+0.039215686274510, +0.086274509803922, +0.149019607843137, +0.011764705882353,
+0.082352941176471, +0.168627450980392, -0.086274509803922, -0.082352941176471,
+0.152941176470588, +0.039215686274510, -0.086274509803922, +0.070588235294118,
+0.156862745098039, -0.023529411764706, -0.015686274509804, +0.184313725490196,
+0.164705882352941, +0.094117647058824, +0.062745098039216, -0.101960784313725,
-0.054901960784314, +0.082352941176471, +0.011764705882353, -0.141176470588235,
+0.000000000000000, +0.168627450980392, -0.003921568627451, +0.078431372549020,
+0.160784313725490, -0.066666666666667, -0.203921568627451, +0.047058823529412,
+0.050980392156863, -0.160784313725490, +0.027450980392157, +0.058823529411765,
-0.105882352941176, +0.031372549019608, -0.070588235294118, -0.043137254901961,
+0.145098039215686, +0.043137254901961, +0.027450980392157, -0.031372549019608,
+0.121568627450980, +0.207843137254902, +0.066666666666667, +0.062745098039216,
+0.066666666666667, -0.047058823529412, +0.117647058823529, +0.207843137254902,
-0.074509803921569, +0.031372549019608, +0.160784313725490, -0.031372549019608,
-0.070588235294118, +0.149019607843137, +0.125490196078431, -0.035294117647059,
-0.074509803921569, -0.188235294117647, -0.207843137254902, +0.117647058823529,
+0.125490196078431, -0.086274509803922, +0.043137254901961, +0.105882352941176,
+0.019607843137255, +0.000000000000000, +0.188235294117647, +0.074509803921569,
-0.039215686274510, +0.129411764705882, +0.098039215686275, +0.086274509803922,
+0.149019607843137, +0.184313725490196, +0.074509803921569, +0.094117647058824,
+0.196078431372549, +0.047058823529412, -0.031372549019608, +0.137254901960784,
+0.200000000000000, +0.062745098039216, +0.082352941176471, +0.137254901960784,
+0.066666666666667, +0.035294117647059, +0.168627450980392, +0.039215686274510,
-0.019607843137255, +0.156862745098039, +0.066666666666667, +0.125490196078431,
-0.184313725490196, -0.090196078431373, -0.133333333333333, -0.188235294117647,
-0.019607843137255, -0.074509803921569, -0.254901960784314, -0.211764705882353,
-0.011764705882353, -0.105882352941176, -0.180392156862745, -0.098039215686275,
-0.188235294117647, -0.200000000000000, -0.082352941176471, -0.054901960784314,
-0.156862745098039, -0.200000000000000, -0.152941176470588, -0.094117647058824,
-0.172549019607843, -0.125490196078431, -0.062745098039216, -0.105882352941176,
+0.023529411764706, -0.066666666666667, -0.211764705882353, -0.019607843137255,
+0.035294117647059, -0.129411764705882, -0.117647058823529, +0.000000000000000,
-0.047058823529412, +0.000000000000000, +0.137254901960784, -0.047058823529412,
-0.078431372549020, +0.094117647058824, -0.039215686274510, -0.149019607843137,
+0.113725490196078, +0.105882352941176, -0.117647058823529, +0.019607843137255,
+0.062745098039216, -0.094117647058824, -0.078431372549020, +0.082352941176471,
+0.137254901960784, +0.188235294117647, +0.211764705882353, -0.039215686274510,
-0.027450980392157, +0.243137254901961, +0.180392156862745, -0.003921568627451,
+0.047058823529412, +0.062745098039216, -0.090196078431373, +0.058823529411765,
+0.211764705882353, +0.086274509803922, -0.054901960784314, +0.086274509803922,
+0.066666666666667, -0.192156862745098, -0.031372549019608, +0.105882352941176,
-0.011764705882353, +0.129411764705882, +0.039215686274510, -0.082352941176471,
+0.164705882352941, +0.152941176470588, +0.000000000000000, -0.050980392156863,
+0.082352941176471, +0.137254901960784, +0.023529411764706, +0.019607843137255,
-0.007843137254902, -0.117647058823529, -0.003921568627451, +0.054901960784314,
-0.086274509803922, -0.015686274509804, +0.086274509803922, -0.031372549019608,
-0.070588235294118, +0.090196078431373, +0.133333333333333, +0.164705882352941,
+0.231372549019608, +0.027450980392157, -0.223529411764706, -0.062745098039216,
+0.007843137254902, -0.054901960784314, +0.098039215686275, +0.133333333333333,
+0.043137254901961, -0.035294117647059, +0.031372549019608, -0.023529411764706,
-0.094117647058824, +0.082352941176471, +0.105882352941176, +0.027450980392157,
+0.000000000000000, +0.023529411764706, +0.000000000000000, +0.082352941176471,
+0.141176470588235, +0.003921568627451, +0.000000000000000, +0.019607843137255,
+0.000000000000000, -0.031372549019608, +0.011764705882353, +0.137254901960784,
+0.011764705882353, -0.047058823529412, +0.125490196078431, +0.062745098039216,
-0.113725490196078, -0.101960784313725, -0.094117647058824, -0.074509803921569,
+0.125490196078431, +0.145098039215686, +0.109803921568627, +0.015686274509804,
+0.149019607843137, +0.164705882352941, -0.043137254901961, -0.058823529411765,
+0.164705882352941, +0.203921568627451, +0.113725490196078, +0.125490196078431,
+0.011764705882353, -0.062745098039216, +0.054901960784314, +0.129411764705882,
+0.000000000000000, -0.003921568627451, +0.062745098039216, +0.019607843137255,
-0.082352941176471, -0.043137254901961, -0.066666666666667, -0.168627450980392,
-0.039215686274510, +0.027450980392157, -0.070588235294118, -0.023529411764706,
+0.066666666666667, -0.035294117647059, -0.054901960784314, +0.015686274509804,
-0.066666666666667, -0.129411764705882, +0.082352941176471, -0.019607843137255,
-0.176470588235294, -0.019607843137255, -0.031372549019608, -0.184313725490196,
+0.003921568627451, +0.062745098039216, -0.192156862745098, -0.062745098039216,
-0.023529411764706, -0.168627450980392, -0.184313725490196, -0.054901960784314,
-0.074509803921569, -0.043137254901961, +0.137254901960784, -0.094117647058824,
-0.227450980392157, +0.074509803921569, +0.023529411764706, -0.117647058823529,
+0.062745098039216, +0.007843137254902, -0.231372549019608, -0.094117647058824,
+0.086274509803922, -0.070588235294118, -0.117647058823529, +0.101960784313725,
+0.062745098039216, -0.200000000000000, -0.121568627450980, -0.054901960784314,
-0.200000000000000, -0.121568627450980, -0.078431372549020, -0.215686274509804,
-0.074509803921569, +0.113725490196078, -0.101960784313725, -0.200000000000000,
-0.007843137254902, -0.003921568627451, -0.101960784313725, -0.027450980392157,
-0.011764705882353, -0.133333333333333, -0.109803921568627, -0.121568627450980,
-0.160784313725490, -0.062745098039216, -0.015686274509804, -0.180392156862745,
-0.207843137254902, -0.062745098039216, -0.141176470588235, -0.113725490196078,
+0.121568627450980, +0.050980392156863, -0.129411764705882, -0.086274509803922,
-0.094117647058824, -0.156862745098039, -0.113725490196078, +0.019607843137255,
-0.023529411764706, -0.180392156862745, -0.125490196078431, -0.098039215686275,
-0.168627450980392, -0.043137254901961, +0.031372549019608, -0.137254901960784,
-0.215686274509804, -0.176470588235294, -0.188235294117647, -0.094117647058824,
+0.023529411764706, -0.125490196078431, -0.031372549019608, +0.050980392156863,
-0.105882352941176, -0.180392156862745, -0.188235294117647, -0.019607843137255,
-0.145098039215686, -0.215686274509804, -0.003921568627451, +0.011764705882353,
-0.113725490196078, -0.180392156862745, -0.164705882352941, -0.164705882352941,
+0.094117647058824, +0.078431372549020, +0.098039215686275, +0.105882352941176,
+0.188235294117647, +0.172549019607843, -0.062745098039216, -0.035294117647059,
+0.094117647058824, +0.168627450980392, +0.219607843137255, +0.254901960784314,
+0.125490196078431, +0.035294117647059, +0.168627450980392, +0.294117647058824,
+0.149019607843137, +0.082352941176471, +0.231372549019608, +0.180392156862745,
+0.047058823529412, +0.125490196078431, +0.172549019607843, +0.054901960784314,
+0.074509803921569, +0.168627450980392, +0.141176470588235, +0.133333333333333,
+0.164705882352941, +0.074509803921569, +0.094117647058824, +0.188235294117647,
+0.160784313725490, +0.050980392156863, +0.176470588235294, +0.109803921568627,
-0.015686274509804, +0.203921568627451, +0.168627450980392, -0.039215686274510,
+0.105882352941176, +0.117647058823529, -0.125490196078431, +0.039215686274510,
+0.149019607843137, +0.050980392156863, -0.043137254901961, +0.019607843137255,
+0.047058823529412, +0.047058823529412, +0.219607843137255, +0.007843137254902,
-0.188235294117647, +0.105882352941176, +0.066666666666667, -0.098039215686275,
+0.156862745098039, +0.168627450980392, -0.117647058823529, -0.050980392156863,
+0.184313725490196, -0.019607843137255, -0.137254901960784, +0.160784313725490,
+0.231372549019608, +0.035294117647059, +0.027450980392157, +0.035294117647059,
-0.027450980392157, -0.019607843137255, +0.023529411764706, -0.066666666666667,
-0.074509803921569, +0.180392156862745, +0.062745098039216, -0.078431372549020,
+0.149019607843137, +0.090196078431373, -0.035294117647059, +0.113725490196078,
+0.180392156862745, +0.137254901960784, +0.160784313725490, +0.043137254901961,
-0.050980392156863, +0.113725490196078, +0.239215686274510, +0.070588235294118,
-0.019607843137255, +0.137254901960784, +0.066666666666667, +0.007843137254902,
+0.168627450980392, +0.180392156862745, +0.113725490196078, +0.105882352941176,
+0.070588235294118, +0.019607843137255, -0.070588235294118, +0.098039215686275,
+0.168627450980392, +0.054901960784314, +0.117647058823529, +0.094117647058824,
-0.027450980392157, -0.015686274509804, +0.145098039215686, +0.141176470588235,
+0.054901960784314, +0.054901960784314, +0.000000000000000, +0.003921568627451,
+0.168627450980392, +0.086274509803922, +0.098039215686275, +0.254901960784314,
+0.172549019607843, +0.062745098039216, +0.031372549019608, +0.168627450980392,
+0.000000000000000, -0.121568627450980, +0.117647058823529, +0.168627450980392,
+0.074509803921569, +0.109803921568627, +0.168627450980392, +0.133333333333333,
-0.062745098039216, -0.054901960784314, -0.200000000000000, -0.168627450980392,
-0.003921568627451, +0.074509803921569, -0.168627450980392, -0.125490196078431,
-0.070588235294118, -0.196078431372549, -0.141176470588235, -0.047058823529412,
-0.109803921568627, -0.113725490196078, +0.019607843137255, +0.078431372549020,
-0.015686274509804, -0.062745098039216, -0.035294117647059, -0.086274509803922,
-0.031372549019608, +0.074509803921569, +0.101960784313725, +0.043137254901961,
-0.015686274509804, -0.078431372549020, -0.078431372549020, +0.062745098039216,
+0.054901960784314, -0.094117647058824, -0.070588235294118, -0.074509803921569,
-0.054901960784314, +0.031372549019608, +0.207843137254902, +0.156862745098039,
-0.050980392156863, +0.070588235294118, +0.043137254901961, -0.047058823529412,
+0.137254901960784, +0.129411764705882, -0.129411764705882, -0.039215686274510,
+0.035294117647059, -0.047058823529412, +0.000000000000000, +0.054901960784314,
+0.031372549019608, +0.039215686274510, +0.239215686274510, +0.031372549019608,
-0.172549019607843, +0.050980392156863, -0.011764705882353, -0.098039215686275,
+0.133333333333333, +0.101960784313725, -0.149019607843137, -0.070588235294118,
+0.192156862745098, +0.090196078431373, -0.086274509803922, +0.086274509803922,
+0.121568627450980, +0.086274509803922, +0.168627450980392, +0.137254901960784,
+0.168627450980392, +0.192156862745098, +0.082352941176471, -0.007843137254902,
+0.019607843137255, +0.164705882352941, +0.062745098039216, -0.125490196078431,
+0.078431372549020, +0.094117647058824, -0.105882352941176, -0.047058823529412,
+0.007843137254902, +0.043137254901961, +0.196078431372549, +0.121568627450980,
-0.078431372549020, +0.003921568627451, +0.200000000000000, +0.129411764705882,
-0.003921568627451, +0.086274509803922, +0.082352941176471, +0.011764705882353,
+0.141176470588235, +0.141176470588235, +0.086274509803922, +0.125490196078431,
+0.050980392156863, -0.050980392156863, -0.105882352941176, +0.058823529411765,
+0.133333333333333, +0.086274509803922, +0.152941176470588, -0.011764705882353,
-0.094117647058824, -0.007843137254902, +0.054901960784314, +0.105882352941176,
+0.074509803921569, +0.000000000000000, -0.074509803921569, -0.043137254901961,
+0.043137254901961, +0.011764705882353, +0.039215686274510, +0.137254901960784,
+0.050980392156863, -0.074509803921569, +0.011764705882353, +0.215686274509804,
+0.058823529411765, -0.105882352941176, +0.125490196078431, +0.176470588235294,
+0.023529411764706, +0.054901960784314, +0.121568627450980, +0.043137254901961,
+0.007843137254902, +0.105882352941176, -0.121568627450980, -0.192156862745098,
-0.007843137254902, +0.164705882352941, -0.011764705882353, +0.003921568627451,
+0.047058823529412, -0.105882352941176, -0.054901960784314, -0.003921568627451,
-0.200000000000000, -0.239215686274510, +0.011764705882353, +0.058823529411765,
-0.109803921568627, -0.027450980392157, +0.035294117647059, -0.192156862745098,
-0.105882352941176, +0.070588235294118, -0.078431372549020, -0.082352941176471,
+0.000000000000000, -0.152941176470588, -0.176470588235294, +0.027450980392157,
+0.007843137254902, -0.074509803921569, +0.078431372549020, -0.058823529411765,
-0.262745098039216, -0.133333333333333, +0.094117647058824, +0.086274509803922,
-0.105882352941176, -0.117647058823529, -0.200000000000000, -0.231372549019608,
-0.015686274509804, +0.101960784313725, -0.137254901960784, -0.090196078431373,
-0.003921568627451, -0.231372549019608, -0.211764705882353, -0.125490196078431,
-0.196078431372549, -0.133333333333333, +0.184313725490196, +0.003921568627451,
-0.180392156862745, +0.058823529411765, -0.109803921568627, -0.274509803921569,
+0.000000000000000, +0.011764705882353, -0.227450980392157, -0.117647058823529,
+0.050980392156863, -0.062745098039216, -0.203921568627451, -0.062745098039216,
-0.098039215686275, -0.156862745098039, +0.066666666666667, -0.007843137254902,
-0.074509803921569, +0.050980392156863, -0.101960784313725, -0.231372549019608,
-0.094117647058824, -0.027450980392157, -0.062745098039216, -0.149019607843137,
-0.043137254901961, +0.050980392156863, -0.090196078431373, -0.137254901960784,
-0.176470588235294, -0.203921568627451, -0.050980392156863, +0.011764705882353,
-0.215686274509804, -0.176470588235294, -0.015686274509804, -0.066666666666667,
-0.200000000000000, -0.098039215686275, -0.074509803921569, -0.219607843137255,
-0.050980392156863, -0.043137254901961, -0.176470588235294, -0.074509803921569,
-0.074509803921569, -0.192156862745098, -0.207843137254902, -0.129411764705882,
-0.160784313725490, -0.152941176470588, +0.078431372549020, -0.054901960784314,
-0.262745098039216, -0.160784313725490, -0.149019607843137, -0.168627450980392,
-0.137254901960784, -0.133333333333333, -0.231372549019608, -0.227450980392157,
-0.176470588235294, -0.250980392156863, -0.129411764705882, +0.031372549019608,
-0.086274509803922, -0.278431372549020, -0.121568627450980, +0.098039215686275,
-0.050980392156863, -0.239215686274510, -0.039215686274510, +0.003921568627451,
-0.129411764705882, -0.082352941176471, -0.074509803921569, -0.168627450980392,
-0.074509803921569, +0.109803921568627, +0.086274509803922, -0.094117647058824,
-0.011764705882353, +0.074509803921569, -0.054901960784314, +0.074509803921569,
+0.117647058823529, -0.015686274509804, +0.109803921568627, +0.184313725490196,
-0.070588235294118, -0.254901960784314, -0.031372549019608, +0.094117647058824,
-0.058823529411765, +0.047058823529412, +0.247058823529412, -0.074509803921569,
-0.152941176470588, +0.133333333333333, +0.043137254901961, +0.039215686274510,
+0.192156862745098, +0.054901960784314, -0.003921568627451, +0.074509803921569,
-0.031372549019608, -0.043137254901961, +0.200000000000000, +0.168627450980392,
-0.149019607843137, -0.149019607843137, +0.074509803921569, +0.133333333333333,
+0.062745098039216, +0.098039215686275, +0.039215686274510, -0.070588235294118,
+0.031372549019608, +0.149019607843137, -0.050980392156863, +0.023529411764706,
+0.223529411764706, -0.031372549019608, -0.074509803921569, +0.054901960784314,
-0.039215686274510, -0.070588235294118, +0.203921568627451, +0.113725490196078,
-0.078431372549020, +0.196078431372549, +0.200000000000000, -0.047058823529412,
+0.082352941176471, +0.141176470588235, -0.168627450980392, -0.082352941176471,
+0.145098039215686, -0.003921568627451, -0.125490196078431, +0.074509803921569,
+0.047058823529412, -0.188235294117647, +0.003921568627451, -0.019607843137255,
-0.152941176470588, +0.090196078431373, +0.074509803921569, -0.109803921568627,
+0.003921568627451, +0.105882352941176, +0.070588235294118, +0.031372549019608,
+0.070588235294118, +0.090196078431373, +0.035294117647059, +0.074509803921569,
+0.062745098039216, -0.007843137254902, +0.070588235294118, +0.117647058823529,
-0.109803921568627, -0.062745098039216, +0.109803921568627, +0.058823529411765,
-0.050980392156863, +0.082352941176471, +0.152941176470588, -0.047058823529412,
+0.074509803921569, +0.156862745098039, +0.031372549019608, +0.082352941176471,
+0.082352941176471, -0.039215686274510, +0.015686274509804, +0.074509803921569,
-0.066666666666667, -0.113725490196078, +0.160784313725490, +0.219607843137255,
+0.007843137254902, -0.007843137254902, +0.043137254901961, +0.062745098039216,
+0.094117647058824, +0.188235294117647, +0.054901960784314, -0.066666666666667,
+0.031372549019608, -0.062745098039216, -0.023529411764706, +0.207843137254902,
+0.133333333333333, -0.152941176470588, -0.047058823529412, +0.184313725490196,
+0.066666666666667, -0.125490196078431, +0.043137254901961, +0.086274509803922,
-0.011764705882353, +0.086274509803922, +0.078431372549020, +0.054901960784314,
+0.039215686274510, +0.082352941176471, +0.188235294117647, -0.050980392156863,
-0.043137254901961, +0.003921568627451, -0.160784313725490, +0.082352941176471,
+0.180392156862745, -0.086274509803922, -0.007843137254902, +0.188235294117647,
+0.007843137254902, -0.227450980392157, -0.160784313725490, -0.094117647058824,
-0.223529411764706, -0.094117647058824, +0.196078431372549, -0.031372549019608,
-0.243137254901961, -0.019607843137255, -0.039215686274510, -0.105882352941176,
-0.015686274509804, -0.098039215686275, -0.019607843137255, +0.094117647058824,
-0.105882352941176, -0.168627450980392, +0.074509803921569, +0.156862745098039,
-0.117647058823529, -0.196078431372549, +0.054901960784314, +0.086274509803922,
-0.003921568627451, +0.105882352941176, +0.066666666666667, -0.062745098039216,
+0.070588235294118, +0.152941176470588, -0.078431372549020, -0.023529411764706,
+0.188235294117647, -0.011764705882353, -0.027450980392157, +0.137254901960784,
-0.007843137254902, -0.094117647058824, +0.105882352941176, +0.054901960784314,
-0.101960784313725, +0.070588235294118, +0.160784313725490, +0.050980392156863,
+0.113725490196078, +0.192156862745098, -0.117647058823529, -0.098039215686275,
+0.239215686274510, +0.043137254901961, -0.125490196078431, +0.117647058823529,
+0.094117647058824, -0.113725490196078, +0.105882352941176, +0.101960784313725,
-0.113725490196078, +0.035294117647059, +0.035294117647059, -0.137254901960784,
-0.094117647058824, +0.094117647058824, +0.043137254901961, -0.058823529411765,
+0.117647058823529, +0.074509803921569, -0.062745098039216, +0.101960784313725,
+0.160784313725490, +0.070588235294118, +0.121568627450980, +0.117647058823529,
-0.105882352941176, -0.086274509803922, +0.078431372549020, +0.007843137254902,
-0.074509803921569, +0.058823529411765, +0.133333333333333, -0.035294117647059,
-0.023529411764706, +0.117647058823529, +0.098039215686275, +0.207843137254902,
+0.156862745098039, -0.121568627450980, -0.094117647058824, +0.117647058823529,
+0.086274509803922, -0.043137254901961, +0.031372549019608, +0.078431372549020,
+0.062745098039216, +0.145098039215686, +0.133333333333333, +0.050980392156863,
+0.062745098039216, +0.219607843137255, +0.137254901960784, -0.070588235294118,
+0.050980392156863, +0.019607843137255, -0.047058823529412, +0.192156862745098,
+0.168627450980392, -0.105882352941176, -0.121568627450980, +0.105882352941176,
+0.098039215686275, -0.098039215686275, +0.043137254901961, +0.054901960784314,
+0.019607843137255, +0.149019607843137, +0.047058823529412, +0.031372549019608,
+0.101960784313725, +0.031372549019608, +0.137254901960784, +0.086274509803922,
+0.090196078431373, +0.168627450980392, +0.000000000000000, +0.074509803921569,
+0.207843137254902, +0.031372549019608, +0.011764705882353, +0.219607843137255,
+0.200000000000000, +0.027450980392157, +0.023529411764706, +0.094117647058824,
-0.113725490196078, -0.125490196078431, +0.192156862745098, +0.149019607843137,
-0.117647058823529, -0.007843137254902, +0.062745098039216, +0.000000000000000,
+0.007843137254902, -0.105882352941176, -0.058823529411765, +0.129411764705882,
+0.000000000000000, -0.180392156862745, +0.019607843137255, +0.141176470588235,
-0.058823529411765, -0.164705882352941, +0.066666666666667, +0.015686274509804,
-0.168627450980392, +0.039215686274510, -0.011764705882353, -0.254901960784314,
-0.050980392156863, +0.082352941176471, -0.156862745098039, -0.054901960784314,
+0.101960784313725, -0.129411764705882, -0.168627450980392, +0.050980392156863,
-0.101960784313725, -0.207843137254902, +0.050980392156863, -0.023529411764706,
-0.219607843137255, -0.047058823529412, -0.023529411764706, -0.145098039215686,
+0.000000000000000, +0.094117647058824, -0.137254901960784, -0.156862745098039,
+0.188235294117647, +0.031372549019608, -0.219607843137255, +0.043137254901961,
+0.003921568627451, -0.215686274509804, +0.035294117647059, +0.101960784313725,
-0.113725490196078, +0.007843137254902, +0.058823529411765, -0.058823529411765,
-0.113725490196078, +0.003921568627451, -0.078431372549020, -0.250980392156863,
+0.082352941176471, +0.133333333333333, -0.152941176470588, -0.074509803921569,
+0.000000000000000, -0.086274509803922, +0.011764705882353, +0.023529411764706,
-0.094117647058824, -0.109803921568627, -0.074509803921569, -0.180392156862745,
-0.223529411764706, -0.105882352941176, -0.062745098039216, -0.168627450980392,
-0.200000000000000, -0.082352941176471, -0.133333333333333, +0.031372549019608,
+0.117647058823529, -0.074509803921569, -0.137254901960784, -0.058823529411765,
-0.011764705882353, -0.113725490196078, -0.066666666666667, -0.105882352941176,
-0.231372549019608, -0.023529411764706, +0.039215686274510, -0.200000000000000,
-0.133333333333333, +0.168627450980392, +0.141176470588235, -0.145098039215686,
-0.137254901960784, -0.078431372549020, -0.168627450980392, +0.039215686274510,
+0.101960784313725, -0.086274509803922, -0.117647058823529, +0.058823529411765,
+0.043137254901961, -0.200000000000000, -0.035294117647059, -0.023529411764706,
-0.062745098039216, +0.149019607843137, +0.031372549019608, -0.050980392156863,
+0.000000000000000, -0.027450980392157, -0.035294117647059, -0.086274509803922,
-0.109803921568627, -0.035294117647059, -0.094117647058824, -0.113725490196078,
-0.062745098039216, -0.054901960784314, -0.101960784313725, +0.011764705882353,
+0.180392156862745, +0.125490196078431, +0.137254901960784, +0.211764705882353,
+0.078431372549020, -0.058823529411765, +0.105882352941176, +0.086274509803922,
-0.070588235294118, +0.039215686274510, +0.066666666666667, +0.074509803921569,
+0.105882352941176, -0.019607843137255, -0.105882352941176, +0.094117647058824,
+0.184313725490196, +0.019607843137255, +0.101960784313725, +0.192156862745098,
+0.054901960784314, -0.035294117647059, +0.094117647058824, +0.011764705882353,
-0.192156862745098, +0.082352941176471, +0.149019607843137, -0.176470588235294,
+0.007843137254902, +0.184313725490196, -0.031372549019608, +0.086274509803922,
+0.196078431372549, -0.086274509803922, -0.117647058823529, +0.200000000000000,
+0.101960784313725, -0.109803921568627, +0.125490196078431, +0.168627450980392,
-0.058823529411765, +0.070588235294118, +0.125490196078431, -0.015686274509804,
+0.125490196078431, +0.180392156862745, -0.062745098039216, -0.137254901960784,
+0.164705882352941, +0.192156862745098, -0.050980392156863, +0.113725490196078,
+0.098039215686275, -0.188235294117647, +0.019607843137255, +0.152941176470588,
-0.050980392156863, +0.050980392156863, +0.215686274509804, +0.152941176470588,
+0.074509803921569, +0.164705882352941, +0.113725490196078, -0.117647058823529,
+0.043137254901961, +0.176470588235294, -0.003921568627451, +0.023529411764706,
+0.101960784313725, +0.031372549019608, +0.121568627450980, +0.149019607843137,
+0.121568627450980, +0.164705882352941, +0.200000000000000, +0.066666666666667,
-0.011764705882353, +0.094117647058824, +0.133333333333333, +0.050980392156863,
+0.000000000000000, +0.019607843137255, -0.105882352941176, +0.054901960784314,
+0.219607843137255, +0.105882352941176, +0.047058823529412, +0.043137254901961,
+0.074509803921569, -0.039215686274510, +0.074509803921569, +0.180392156862745,
-0.043137254901961, +0.043137254901961, +0.168627450980392, -0.133333333333333,
-0.149019607843137, +0.149019607843137, +0.243137254901961, +0.129411764705882,
+0.019607843137255, +0.074509803921569, -0.031372549019608, +0.035294117647059,
+0.192156862745098, +0.011764705882353, -0.019607843137255, +0.188235294117647,
+0.176470588235294, -0.066666666666667, +0.047058823529412, +0.094117647058824,
+0.007843137254902, +0.192156862745098, +0.203921568627451, +0.129411764705882,
-0.031372549019608, -0.031372549019608, +0.000000000000000, -0.125490196078431,
-0.164705882352941, -0.105882352941176, -0.105882352941176, -0.105882352941176,
-0.145098039215686, -0.125490196078431, -0.149019607843137, -0.113725490196078,
-0.031372549019608, -0.145098039215686, -0.062745098039216, -0.031372549019608,
-0.125490196078431, -0.168627450980392, +0.003921568627451, -0.062745098039216,
-0.180392156862745, -0.043137254901961, -0.145098039215686, -0.152941176470588,
-0.050980392156863, -0.117647058823529, -0.176470588235294, -0.039215686274510,
-0.019607843137255, -0.129411764705882, -0.066666666666667, +0.039215686274510,
-0.094117647058824, -0.050980392156863, +0.156862745098039, +0.039215686274510,
-0.231372549019608, +0.000000000000000, +0.145098039215686, -0.176470588235294,
-0.019607843137255, +0.156862745098039, -0.054901960784314, +0.027450980392157,
+0.164705882352941, -0.133333333333333, -0.188235294117647, +0.101960784313725,
+0.086274509803922, -0.047058823529412, +0.007843137254902, +0.007843137254902,
-0.074509803921569, +0.011764705882353, +0.054901960784314, -0.031372549019608,
+0.066666666666667, +0.137254901960784, -0.058823529411765, -0.125490196078431,
+0.023529411764706, +0.047058823529412, -0.074509803921569, +0.094117647058824,
+0.125490196078431, -0.160784313725490, -0.011764705882353, +0.156862745098039,
-0.101960784313725, -0.156862745098039, -0.023529411764706, -0.117647058823529,
-0.043137254901961, +0.098039215686275, +0.070588235294118, -0.062745098039216,
-0.019607843137255, +0.000000000000000, -0.117647058823529, -0.015686274509804,
+0.031372549019608, -0.094117647058824, +0.007843137254902, +0.054901960784314,
-0.062745098039216, +0.039215686274510, +0.188235294117647, +0.054901960784314,
-0.070588235294118, +0.098039215686275, +0.105882352941176, +0.007843137254902,
+0.105882352941176, +0.125490196078431, -0.129411764705882, -0.070588235294118,
+0.133333333333333, -0.039215686274510, -0.129411764705882, +0.039215686274510,
+0.184313725490196, -0.011764705882353, +0.000000000000000, +0.094117647058824,
-0.023529411764706, +0.094117647058824, +0.215686274509804, +0.015686274509804,
-0.149019607843137, -0.066666666666667, +0.027450980392157, +0.094117647058824,
+0.058823529411765, +0.058823529411765, -0.047058823529412, -0.031372549019608,
+0.113725490196078, -0.070588235294118, -0.145098039215686, +0.078431372549020,
+0.117647058823529, -0.090196078431373, -0.054901960784314, -0.031372549019608,
-0.094117647058824, +0.039215686274510, +0.082352941176471, +0.011764705882353,
-0.031372549019608, -0.011764705882353, +0.176470588235294, +0.039215686274510,
+0.043137254901961, +0.094117647058824, +0.070588235294118, +0.141176470588235,
+0.145098039215686, +0.105882352941176, +0.031372549019608, +0.074509803921569,
+0.109803921568627, -0.066666666666667, +0.047058823529412, +0.113725490196078,
-0.023529411764706, -0.113725490196078, +0.094117647058824, +0.031372549019608,
-0.086274509803922, +0.149019607843137, +0.113725490196078, +0.066666666666667,
+0.180392156862745, +0.000000000000000, -0.137254901960784, +0.043137254901961,
-0.007843137254902, -0.141176470588235, -0.129411764705882, +0.043137254901961,
-0.090196078431373, -0.145098039215686, +0.203921568627451, +0.192156862745098,
-0.105882352941176, +0.070588235294118, +0.156862745098039, -0.184313725490196,
-0.101960784313725, +0.047058823529412, -0.164705882352941, -0.062745098039216,
+0.200000000000000, -0.090196078431373, -0.207843137254902, -0.062745098039216,
-0.141176470588235, -0.121568627450980, -0.019607843137255, -0.125490196078431,
-0.231372549019608, -0.149019607843137, -0.137254901960784, -0.200000000000000,
-0.058823529411765, +0.082352941176471, -0.113725490196078, -0.098039215686275,
-0.011764705882353, -0.137254901960784, -0.211764705882353, -0.043137254901961,
+0.027450980392157, -0.196078431372549, -0.070588235294118, +0.137254901960784,
-0.078431372549020, -0.133333333333333, +0.007843137254902, -0.200000000000000,
-0.168627450980392, -0.011764705882353, -0.101960784313725, -0.192156862745098,
-0.027450980392157, +0.011764705882353, -0.188235294117647, -0.101960784313725,
-0.043137254901961, -0.219607843137255, -0.078431372549020, +0.074509803921569,
-0.200000000000000, -0.219607843137255, -0.023529411764706, -0.090196078431373,
-0.262745098039216, -0.070588235294118, -0.043137254901961, -0.247058823529412,
+0.000000000000000, +0.152941176470588, -0.149019607843137, -0.176470588235294,
+0.066666666666667, -0.074509803921569, -0.196078431372549, -0.035294117647059,
+0.066666666666667, -0.141176470588235, -0.090196078431373, -0.074509803921569,
-0.227450980392157, -0.082352941176471, -0.015686274509804, -0.047058823529412,
-0.149019607843137, -0.152941176470588, -0.125490196078431, -0.180392156862745,
-0.133333333333333, -0.070588235294118, -0.098039215686275, -0.058823529411765,
-0.062745098039216, -0.188235294117647, -0.231372549019608, -0.121568627450980,
-0.043137254901961, -0.149019607843137, -0.082352941176471, -0.094117647058824,
-0.239215686274510, -0.082352941176471, -0.050980392156863, -0.141176470588235,
-0.129411764705882, -0.149019607843137, +0.031372549019608, -0.094117647058824,
-0.011764705882353, +0.050980392156863, -0.125490196078431, +0.007843137254902,
+0.145098039215686, +0.039215686274510, -0.074509803921569, +0.023529411764706,
+0.043137254901961, -0.117647058823529, -0.027450980392157, +0.156862745098039,
+0.086274509803922, -0.027450980392157, +0.137254901960784, -0.003921568627451,
-0.149019607843137, +0.066666666666667, +0.090196078431373, +0.074509803921569,
+0.211764705882353, -0.007843137254902, -0.215686274509804, +0.031372549019608,
+0.121568627450980, +0.086274509803922, +0.039215686274510, +0.180392156862745,
+0.133333333333333, -0.086274509803922, +0.125490196078431, +0.200000000000000,
-0.062745098039216, +0.086274509803922, +0.192156862745098, -0.149019607843137,
-0.047058823529412, +0.113725490196078, -0.105882352941176, -0.039215686274510,
+0.270588235294118, +0.058823529411765, -0.094117647058824, +0.074509803921569,
-0.019607843137255, -0.003921568627451, +0.180392156862745, +0.184313725490196,
+0.058823529411765, +0.090196078431373, +0.113725490196078, -0.031372549019608,
+0.007843137254902, +0.152941176470588, -0.094117647058824, -0.043137254901961,
+0.211764705882353, +0.074509803921569, +0.015686274509804, +0.109803921568627,
+0.125490196078431, -0.031372549019608, +0.031372549019608, +0.211764705882353,
+0.066666666666667, +0.047058823529412, +0.219607843137255, +0.011764705882353,
-0.090196078431373, +0.098039215686275, +0.113725490196078, -0.031372549019608,
+0.074509803921569, +0.180392156862745, +0.035294117647059, +0.082352941176471,
+0.141176470588235, -0.019607843137255, +0.000000000000000, +0.164705882352941,
-0.066666666666667, -0.152941176470588, +0.074509803921569, +0.121568627450980,
-0.058823529411765, +0.066666666666667, +0.109803921568627, -0.192156862745098,
+0.000000000000000, +0.219607843137255, -0.011764705882353, -0.039215686274510,
+0.200000000000000, +0.031372549019608, -0.082352941176471, +0.062745098039216,
+0.019607843137255, -0.125490196078431, +0.070588235294118, +0.098039215686275,
-0.098039215686275, -0.007843137254902, +0.062745098039216, +0.074509803921569,
-0.019607843137255, +0.031372549019608, +0.086274509803922, -0.094117647058824,
-0.058823529411765, +0.031372549019608, +0.062745098039216, +0.180392156862745,
+0.168627450980392, +0.082352941176471, +0.050980392156863, +0.043137254901961,
+0.000000000000000, -0.058823529411765, +0.101960784313725, +0.156862745098039,
-0.043137254901961, +0.066666666666667, +0.152941176470588, +0.101960784313725,
+0.074509803921569, +0.039215686274510, +0.090196078431373, -0.062745098039216,
+0.000000000000000, +0.168627450980392, -0.035294117647059, -0.082352941176471,
+0.007843137254902, -0.066666666666667, -0.098039215686275, +0.047058823529412,
-0.003921568627451, -0.180392156862745, -0.129411764705882, +0.000000000000000,
-0.105882352941176, -0.117647058823529, +0.149019607843137, +0.054901960784314,
-0.149019607843137, -0.125490196078431, -0.125490196078431, -0.156862745098039,
-0.019607843137255, -0.047058823529412, -0.192156862745098, -0.086274509803922,
-0.019607843137255, -0.082352941176471, -0.098039215686275, -0.007843137254902,
-0.047058823529412, -0.160784313725490, -0.019607843137255, +0.078431372549020,
-0.164705882352941, -0.023529411764706, +0.196078431372549, -0.129411764705882,
-0.011764705882353, +0.156862745098039, -0.137254901960784, -0.117647058823529,
+0.152941176470588, +0.023529411764706, -0.125490196078431, +0.066666666666667,
+0.003921568627451, -0.050980392156863, +0.054901960784314, +0.019607843137255,
+0.011764705882353, +0.133333333333333, +0.180392156862745, +0.000000000000000,
-0.074509803921569, +0.062745098039216, -0.145098039215686, -0.137254901960784,
+0.152941176470588, +0.031372549019608, -0.011764705882353, +0.105882352941176,
+0.023529411764706, -0.121568627450980, -0.062745098039216, +0.090196078431373,
+0.003921568627451, -0.007843137254902, +0.090196078431373, -0.043137254901961,
-0.043137254901961, +0.105882352941176, +0.062745098039216, -0.066666666666667,
-0.039215686274510, +0.039215686274510, -0.023529411764706, +0.023529411764706,
+0.082352941176471, +0.000000000000000, +0.023529411764706, +0.105882352941176,
-0.050980392156863, -0.113725490196078, +0.098039215686275, +0.117647058823529,
-0.027450980392157, +0.121568627450980, +0.137254901960784, -0.168627450980392,
-0.039215686274510, +0.160784313725490, +0.027450980392157, -0.047058823529412,
+0.050980392156863, -0.105882352941176, -0.098039215686275, +0.172549019607843,
+0.019607843137255, -0.188235294117647, +0.035294117647059, +0.117647058823529,
-0.074509803921569, +0.031372549019608, +0.223529411764706, +0.149019607843137,
-0.007843137254902, +0.078431372549020, +0.129411764705882, -0.050980392156863,
-0.066666666666667, -0.011764705882353, -0.019607843137255, +0.090196078431373,
+0.094117647058824, -0.019607843137255, +0.043137254901961, +0.203921568627451,
+0.074509803921569, -0.160784313725490, -0.019607843137255, +0.145098039215686,
+0.011764705882353, -0.023529411764706, +0.015686274509804, +0.043137254901961,
+0.105882352941176, +0.113725490196078, +0.231372549019608, +0.070588235294118,
-0.027450980392157, +0.039215686274510, -0.043137254901961, +0.011764705882353,
+0.031372549019608, -0.098039215686275, -0.082352941176471, +0.156862745098039,
+0.192156862745098, -0.015686274509804, +0.050980392156863, +0.113725490196078,
-0.043137254901961, -0.043137254901961, +0.164705882352941, +0.200000000000000,
+0.113725490196078, +0.066666666666667, +0.062745098039216, +0.007843137254902,
+0.047058823529412, +0.137254901960784, +0.054901960784314, +0.031372549019608,
+0.094117647058824, -0.062745098039216, -0.094117647058824, +0.054901960784314,
-0.027450980392157, -0.152941176470588, -0.027450980392157, +0.098039215686275,
-0.145098039215686, -0.054901960784314, +0.231372549019608, +0.011764705882353,
+0.031372549019608, +0.180392156862745, -0.137254901960784, -0.145098039215686,
+0.137254901960784, +0.031372549019608, -0.164705882352941, +0.003921568627451,
-0.043137254901961, -0.141176470588235, +0.023529411764706, -0.090196078431373,
-0.168627450980392, -0.019607843137255, +0.086274509803922, -0.074509803921569,
-0.074509803921569, +0.129411764705882, -0.078431372549020, -0.215686274509804,
+0.023529411764706, -0.054901960784314, -0.192156862745098, -0.007843137254902,
-0.047058823529412, -0.219607843137255, -0.180392156862745, -0.031372549019608,
-0.145098039215686, -0.074509803921569, +0.062745098039216, -0.160784313725490,
-0.125490196078431, +0.043137254901961, -0.094117647058824, -0.200000000000000,
-0.113725490196078, -0.054901960784314, -0.176470588235294, -0.145098039215686,
+0.003921568627451, -0.019607843137255, +0.074509803921569, +0.168627450980392,
+0.027450980392157, -0.105882352941176, +0.050980392156863, +0.000000000000000,
-0.200000000000000, +0.047058823529412, +0.125490196078431, -0.145098039215686,
-0.086274509803922, +0.015686274509804, -0.066666666666667, +0.000000000000000,
+0.078431372549020, -0.145098039215686, -0.129411764705882, +0.211764705882353,
+0.137254901960784, -0.207843137254902, -0.058823529411765, +0.156862745098039,
-0.078431372549020, -0.101960784313725, +0.133333333333333, +0.000000000000000,
-0.145098039215686, +0.094117647058824, +0.137254901960784, -0.023529411764706,
-0.113725490196078, -0.078431372549020, -0.098039215686275, -0.019607843137255,
+0.000000000000000, -0.168627450980392, -0.141176470588235, +0.098039215686275,
+0.164705882352941, -0.121568627450980, -0.168627450980392, +0.035294117647059,
+0.011764705882353, -0.070588235294118, -0.160784313725490, -0.137254901960784,
-0.090196078431373, -0.164705882352941, +0.000000000000000, -0.090196078431373,
-0.086274509803922, -0.113725490196078, -0.239215686274510, -0.094117647058824,
+0.023529411764706, -0.137254901960784, -0.243137254901961, +0.011764705882353,
+0.133333333333333, -0.062745098039216, +0.070588235294118, +0.078431372549020,
-0.137254901960784, -0.188235294117647, -0.027450980392157, +0.023529411764706,
-0.003921568627451, +0.094117647058824, +0.168627450980392, +0.105882352941176,
+0.054901960784314, +0.090196078431373, +0.039215686274510, +0.054901960784314,
+0.215686274509804, +0.101960784313725, -0.011764705882353, +0.180392156862745,
+0.137254901960784, -0.066666666666667, +0.074509803921569, +0.219607843137255,
-0.007843137254902, -0.019607843137255, +0.152941176470588, +0.000000000000000,
+0.000000000000000, +0.219607843137255, +0.011764705882353, -0.043137254901961,
+0.207843137254902, +0.113725490196078, -0.086274509803922, +0.086274509803922,
+0.125490196078431, +0.007843137254902, +0.168627450980392, +0.062745098039216,
-0.121568627450980, +0.015686274509804, +0.227450980392157, +0.050980392156863,
-0.011764705882353, +0.250980392156863, +0.176470588235294, -0.011764705882353,
+0.145098039215686, +0.125490196078431, -0.094117647058824, +0.121568627450980,
+0.250980392156863, +0.050980392156863, -0.086274509803922, +0.074509803921569,
+0.062745098039216, +0.050980392156863, +0.207843137254902, +0.035294117647059,
-0.019607843137255, +0.200000000000000, +0.117647058823529, -0.047058823529412,
+0.074509803921569, +0.211764705882353, +0.070588235294118, +0.000000000000000,
+0.117647058823529, +0.082352941176471, +0.113725490196078, +0.125490196078431,
-0.039215686274510, -0.117647058823529, +0.125490196078431, +0.117647058823529,
-0.133333333333333, +0.070588235294118, +0.192156862745098, -0.047058823529412,
-0.015686274509804, +0.145098039215686, +0.050980392156863, +0.113725490196078,
+0.254901960784314, +0.031372549019608, -0.082352941176471, +0.152941176470588,
+0.266666666666667, +0.043137254901961, +0.015686274509804, +0.184313725490196,
+0.019607843137255, -0.062745098039216, +0.176470588235294, +0.117647058823529,
-0.082352941176471, +0.133333333333333, +0.180392156862745, +0.082352941176471,
+0.074509803921569, +0.074509803921569, +0.031372549019608, +0.145098039215686,
+0.152941176470588, -0.090196078431373, -0.109803921568627, +0.043137254901961,
+0.133333333333333, -0.039215686274510, -0.098039215686275, +0.101960784313725,
+0.168627450980392, +0.192156862745098, +0.094117647058824, +0.105882352941176,
+0.074509803921569, +0.000000000000000, +0.133333333333333, +0.027450980392157,
+0.109803921568627, +0.192156862745098, +0.054901960784314, +0.000000000000000,
+0.149019607843137, +0.156862745098039, -0.062745098039216, +0.019607843137255,
+0.050980392156863, -0.137254901960784, +0.070588235294118, +0.168627450980392,
-0.066666666666667, -0.223529411764706, -0.011764705882353, -0.011764705882353,
-0.180392156862745, -0.074509803921569, -0.031372549019608, -0.113725490196078,
-0.164705882352941, -0.168627450980392, -0.243137254901961, -0.125490196078431,
+0.062745098039216, -0.043137254901961, -0.211764705882353, +0.000000000000000,
+0.070588235294118, -0.203921568627451, -0.094117647058824, +0.090196078431373,
-0.094117647058824, -0.117647058823529, +0.019607843137255, -0.137254901960784,
-0.113725490196078, +0.074509803921569, -0.094117647058824, -0.137254901960784,
+0.129411764705882, +0.062745098039216, -0.149019607843137, +0.000000000000000,
-0.023529411764706, -0.101960784313725, +0.105882352941176, +0.019607843137255,
-0.231372549019608, -0.113725490196078, +0.168627450980392, +0.007843137254902,
-0.164705882352941, +0.023529411764706, +0.066666666666667, +0.023529411764706,
+0.133333333333333, +0.145098039215686, -0.058823529411765, -0.023529411764706,
+0.113725490196078, +0.125490196078431, -0.003921568627451, -0.023529411764706,
+0.082352941176471, +0.070588235294118, +0.031372549019608, -0.039215686274510,
-0.086274509803922, +0.121568627450980, +0.156862745098039, -0.094117647058824,
-0.015686274509804, +0.156862745098039, +0.000000000000000, -0.137254901960784,
-0.047058823529412, -0.031372549019608, +0.070588235294118, +0.082352941176471,
-0.152941176470588, -0.168627450980392, +0.113725490196078, +0.086274509803922,
-0.137254901960784, +0.007843137254902, +0.121568627450980, -0.094117647058824,
-0.066666666666667, +0.101960784313725, +0.000000000000000, -0.078431372549020,
-0.023529411764706, -0.105882352941176, -0.133333333333333, +0.015686274509804,
+0.011764705882353, -0.074509803921569, -0.031372549019608, +0.043137254901961,
-0.094117647058824, -0.188235294117647, +0.019607843137255, +0.066666666666667,
-0.054901960784314, +0.062745098039216, +0.023529411764706, -0.137254901960784,
-0.050980392156863, -0.054901960784314, -0.141176470588235, +0.086274509803922,
+0.192156862745098, -0.117647058823529, -0.215686274509804, +0.035294117647059,
+0.039215686274510, -0.192156862745098, -0.149019607843137, -0.035294117647059,
-0.011764705882353, +0.125490196078431, +0.027450980392157, -0.047058823529412,
+0.062745098039216, +0.066666666666667, +0.200000000000000, +0.070588235294118,
+0.066666666666667, +0.168627450980392, +0.066666666666667, -0.062745098039216,
-0.015686274509804, +0.058823529411765, +0.007843137254902, +0.125490196078431,
+0.074509803921569, -0.184313725490196, -0.023529411764706, +0.262745098039216,
+0.098039215686275, -0.105882352941176, +0.121568627450980, +0.109803921568627,
-0.043137254901961, +0.074509803921569, +0.113725490196078, +0.062745098039216,
+0.070588235294118, +0.101960784313725, -0.058823529411765, -0.031372549019608,
+0.149019607843137, +0.062745098039216, -0.188235294117647, -0.035294117647059,
+0.192156862745098, -0.094117647058824, -0.129411764705882, +0.082352941176471,
-0.054901960784314, -0.066666666666667, +0.121568627450980, -0.043137254901961,
-0.035294117647059, +0.031372549019608, -0.192156862745098, -0.262745098039216,
+0.054901960784314, +0.062745098039216, -0.196078431372549, +0.023529411764706,
-0.011764705882353, -0.200000000000000, +0.043137254901961, +0.074509803921569,
-0.200000000000000, -0.149019607843137, +0.117647058823529, +0.000000000000000,
-0.200000000000000, -0.160784313725490, -0.231372549019608, -0.137254901960784,
+0.031372549019608, -0.023529411764706, -0.105882352941176, -0.054901960784314,
-0.031372549019608, -0.039215686274510, +0.027450980392157, -0.086274509803922,
-0.164705882352941, +0.015686274509804, -0.019607843137255, -0.141176470588235,
-0.129411764705882, +0.062745098039216, +0.129411764705882, -0.160784313725490,
-0.168627450980392, +0.054901960784314, -0.062745098039216, -0.262745098039216,
-0.164705882352941, -0.152941176470588, -0.015686274509804, +0.192156862745098,
+0.000000000000000, -0.070588235294118, +0.101960784313725, +0.007843137254902,
-0.196078431372549, -0.031372549019608, +0.062745098039216, -0.176470588235294,
-0.117647058823529, +0.019607843137255, -0.031372549019608, -0.015686274509804,
-0.035294117647059, -0.207843137254902, -0.188235294117647, +0.050980392156863,
-0.043137254901961, -0.196078431372549, -0.047058823529412, +0.094117647058824,
-0.027450980392157, -0.137254901960784, -0.007843137254902, -0.078431372549020,
-0.160784313725490, +0.031372549019608, +0.035294117647059, -0.168627450980392,
-0.054901960784314, +0.027450980392157, -0.145098039215686, +0.011764705882353,
+0.254901960784314, +0.062745098039216, -0.149019607843137, +0.137254901960784,
+0.223529411764706, -0.082352941176471, -0.062745098039216, +0.000000000000000,
-0.101960784313725, +0.074509803921569, +0.074509803921569, -0.113725490196078,
-0.101960784313725, -0.125490196078431, +0.054901960784314, -0.105882352941176,
-0.105882352941176, +0.086274509803922, -0.031372549019608, -0.164705882352941,
-0.200000000000000, -0.247058823529412, -0.200000000000000, +0.094117647058824,
+0.094117647058824, -0.219607843137255, -0.200000000000000, +0.101960784313725,
-0.023529411764706, -0.180392156862745, +0.031372549019608, -0.035294117647059,
-0.164705882352941, +0.007843137254902, +0.117647058823529, +0.070588235294118,
+0.074509803921569, +0.066666666666667, -0.043137254901961, -0.058823529411765,
+0.101960784313725, +0.145098039215686, -0.082352941176471, -0.031372549019608,
+0.164705882352941, -0.054901960784314, -0.070588235294118, +0.145098039215686,
+0.050980392156863, +0.039215686274510, +0.145098039215686, +0.007843137254902,
+0.090196078431373, +0.254901960784314, +0.035294117647059, -0.176470588235294,
+0.074509803921569, +0.180392156862745, -0.137254901960784, -0.003921568627451,
+0.137254901960784, -0.058823529411765, +0.101960784313725, +0.223529411764706,
+0.015686274509804, +0.007843137254902, +0.192156862745098, +0.196078431372549,
+0.109803921568627, +0.109803921568627, -0.082352941176471, -0.082352941176471,
+0.121568627450980, +0.031372549019608, -0.149019607843137, -0.019607843137255,
+0.125490196078431, -0.098039215686275, +0.011764705882353, +0.137254901960784,
-0.070588235294118, +0.074509803921569, +0.200000000000000, +0.023529411764706,
-0.039215686274510, +0.125490196078431, +0.231372549019608, +0.031372549019608,
-0.043137254901961, +0.145098039215686, +0.090196078431373, -0.054901960784314,
+0.070588235294118, -0.031372549019608, -0.062745098039216, +0.086274509803922,
-0.058823529411765, -0.023529411764706, +0.200000000000000, +0.047058823529412,
-0.164705882352941, +0.066666666666667, +0.176470588235294, -0.149019607843137,
-0.129411764705882, +0.074509803921569, +0.047058823529412, +0.117647058823529,
+0.184313725490196, -0.015686274509804, -0.090196078431373, +0.152941176470588,
+0.086274509803922, -0.101960784313725, +0.015686274509804, +0.207843137254902,
+0.125490196078431, +0.027450980392157, +0.172549019607843, +0.039215686274510,
-0.074509803921569, +0.133333333333333, +0.164705882352941, -0.019607843137255,
+0.043137254901961, +0.156862745098039, -0.007843137254902, -0.007843137254902,
+0.196078431372549, +0.113725490196078, -0.109803921568627, +0.062745098039216,
+0.168627450980392, -0.019607843137255, +0.031372549019608, +0.113725490196078,
-0.066666666666667, +0.007843137254902, +0.196078431372549, +0.047058823529412,
+0.007843137254902, -0.101960784313725, +0.133333333333333, -0.039215686274510,
-0.086274509803922, +0.200000000000000, +0.160784313725490, +0.066666666666667,
+0.086274509803922, -0.047058823529412, -0.200000000000000, +0.062745098039216,
+0.152941176470588, -0.062745098039216, -0.070588235294118, +0.117647058823529,
-0.050980392156863, -0.121568627450980, +0.062745098039216, +0.007843137254902,
-0.105882352941176, -0.043137254901961, -0.019607843137255, -0.129411764705882,
-0.054901960784314, -0.062745098039216, -0.149019607843137, -0.196078431372549,
-0.039215686274510, +0.043137254901961, -0.129411764705882, +0.011764705882353,
+0.094117647058824, -0.180392156862745, -0.109803921568627, +0.023529411764706,
-0.145098039215686, -0.098039215686275, -0.039215686274510, -0.168627450980392,
-0.125490196078431, +0.164705882352941, +0.098039215686275, -0.180392156862745,
-0.062745098039216, +0.066666666666667, -0.137254901960784, -0.117647058823529,
+0.039215686274510, -0.098039215686275, +0.015686274509804, +0.078431372549020,
-0.117647058823529, -0.094117647058824, +0.078431372549020, +0.113725490196078,
+0.145098039215686, +0.176470588235294, -0.043137254901961, -0.168627450980392,
+0.062745098039216, +0.149019607843137, -0.152941176470588, -0.152941176470588,
+0.074509803921569, -0.137254901960784, -0.156862745098039, -0.011764705882353,
-0.145098039215686, -0.125490196078431, -0.031372549019608, -0.133333333333333,
-0.215686274509804, -0.098039215686275, +0.000000000000000, -0.066666666666667,
-0.007843137254902, +0.105882352941176, -0.050980392156863, -0.117647058823529,
+0.121568627450980, +0.019607843137255, -0.074509803921569, +0.035294117647059,
-0.203921568627451, -0.207843137254902, +0.176470588235294, +0.070588235294118,
-0.223529411764706, +0.031372549019608, +0.168627450980392, -0.125490196078431,
-0.188235294117647, -0.058823529411765, -0.105882352941176, -0.054901960784314,
+0.027450980392157, -0.098039215686275, -0.168627450980392, +0.062745098039216,
+0.019607843137255, -0.200000000000000, -0.082352941176471, +0.019607843137255,
-0.117647058823529, -0.133333333333333, +0.113725490196078, +0.011764705882353,
-0.137254901960784, +0.015686274509804, +0.000000000000000, -0.168627450980392,
-0.117647058823529, -0.062745098039216, -0.203921568627451, -0.192156862745098,
+0.043137254901961, +0.027450980392157, -0.227450980392157, -0.094117647058824,
-0.023529411764706, -0.211764705882353, -0.109803921568627, +0.011764705882353,
-0.145098039215686, -0.168627450980392, -0.007843137254902, -0.054901960784314,
+0.113725490196078, +0.023529411764706, +0.207843137254902, +0.105882352941176,
-0.031372549019608, +0.094117647058824, +0.054901960784314, +0.023529411764706,
+0.192156862745098, +0.152941176470588, -0.066666666666667, +0.105882352941176,
+0.215686274509804, +0.098039215686275, +0.109803921568627, +0.231372549019608,
-0.015686274509804, -0.050980392156863, +0.196078431372549, +0.117647058823529,
+0.000000000000000, +0.086274509803922, +0.156862745098039, +0.035294117647059,
+0.117647058823529, +0.200000000000000, +0.168627450980392, +0.031372549019608,
+0.090196078431373, +0.058823529411765, -0.141176470588235, +0.094117647058824,
+0.219607843137255, -0.035294117647059, -0.007843137254902, +0.105882352941176,
-0.137254901960784, -0.054901960784314, +0.121568627450980, +0.007843137254902,
-0.109803921568627, +0.094117647058824, +0.109803921568627, -0.129411764705882,
+0.000000000000000, +0.113725490196078, -0.058823529411765, -0.027450980392157,
+0.094117647058824, -0.090196078431373, +0.000000000000000, +0.050980392156863,
-0.152941176470588, -0.145098039215686, +0.000000000000000, -0.082352941176471,
-0.145098039215686, -0.027450980392157, -0.039215686274510, -0.125490196078431,
+0.015686274509804, +0.117647058823529, -0.145098039215686, -0.156862745098039,
+0.043137254901961, -0.019607843137255, -0.058823529411765, -0.078431372549020,
-0.172549019607843, -0.101960784313725, -0.054901960784314, -0.113725490196078,
-0.121568627450980, -0.074509803921569, -0.109803921568627, -0.262745098039216,
-0.035294117647059, +0.200000000000000, -0.027450980392157, -0.145098039215686,
+0.082352941176471, -0.011764705882353, -0.043137254901961, +0.180392156862745,
-0.027450980392157, -0.184313725490196, +0.121568627450980, +0.113725490196078,
-0.243137254901961, -0.011764705882353, +0.231372549019608, +0.090196078431373,
+0.062745098039216, +0.094117647058824, -0.090196078431373, -0.078431372549020,
+0.094117647058824, +0.007843137254902, -0.121568627450980, +0.074509803921569,
+0.137254901960784, -0.156862745098039, -0.094117647058824, +0.043137254901961,
-0.125490196078431, -0.149019607843137, +0.011764705882353, -0.066666666666667,
-0.141176470588235, +0.019607843137255, +0.054901960784314, -0.066666666666667,
+0.000000000000000, +0.035294117647059, -0.168627450980392, -0.203921568627451,
+0.062745098039216, +0.141176470588235, -0.117647058823529, -0.054901960784314,
+0.090196078431373, -0.141176470588235, -0.070588235294118, +0.125490196078431,
-0.039215686274510, -0.019607843137255, +0.105882352941176, +0.019607843137255,
-0.176470588235294, -0.184313725490196, -0.086274509803922, -0.105882352941176,
-0.043137254901961, +0.062745098039216, -0.121568627450980, -0.168627450980392,
+0.074509803921569, +0.066666666666667, -0.145098039215686, -0.101960784313725,
-0.050980392156863, -0.113725490196078, -0.047058823529412, +0.152941176470588,
-0.043137254901961, -0.211764705882353, +0.035294117647059, +0.043137254901961,
-0.176470588235294, -0.113725490196078, +0.035294117647059, -0.027450980392157,
-0.011764705882353, +0.054901960784314, +0.113725490196078, +0.094117647058824,
+0.137254901960784, +0.000000000000000, -0.137254901960784, +0.000000000000000,
+0.050980392156863, +0.007843137254902, +0.094117647058824, +0.219607843137255,
-0.007843137254902, -0.007843137254902, +0.180392156862745, +0.105882352941176,
+0.031372549019608, +0.172549019607843, +0.219607843137255, +0.043137254901961,
+0.133333333333333, +0.200000000000000, +0.031372549019608, +0.047058823529412,
+0.098039215686275, +0.000000000000000, +0.125490196078431, +0.149019607843137,
-0.047058823529412, -0.062745098039216, +0.113725490196078, +0.125490196078431,
+0.023529411764706, +0.082352941176471, +0.062745098039216, -0.039215686274510,
+0.137254901960784, +0.266666666666667, +0.015686274509804, -0.015686274509804,
+0.117647058823529, +0.098039215686275, +0.207843137254902, +0.180392156862745,
+0.019607843137255, +0.137254901960784, +0.184313725490196, +0.062745098039216,
+0.054901960784314, +0.196078431372549, +0.207843137254902, -0.035294117647059,
+0.019607843137255, +0.239215686274510, +0.023529411764706, -0.078431372549020,
+0.164705882352941, +0.094117647058824, +0.043137254901961, +0.188235294117647,
+0.019607843137255, -0.105882352941176, +0.152941176470588, +0.219607843137255,
-0.070588235294118, +0.000000000000000, +0.145098039215686, +0.019607843137255,
+0.050980392156863, +0.254901960784314, +0.137254901960784, -0.015686274509804,
+0.168627450980392, +0.137254901960784, -0.062745098039216, -0.027450980392157,
+0.172549019607843, +0.090196078431373, +0.058823529411765, +0.200000000000000,
+0.031372549019608, +0.023529411764706, +0.184313725490196, +0.094117647058824,
+0.058823529411765, +0.168627450980392, +0.133333333333333, +0.007843137254902,
+0.098039215686275, +0.254901960784314, +0.070588235294118, -0.066666666666667,
+0.137254901960784, +0.184313725490196, +0.039215686274510, +0.078431372549020,
+0.223529411764706, +0.074509803921569, +0.035294117647059, +0.254901960784314,
+0.062745098039216, +0.039215686274510, +0.274509803921569, +0.176470588235294,
-0.043137254901961, -0.058823529411765, -0.007843137254902, -0.133333333333333,
-0.019607843137255, +0.239215686274510, +0.054901960784314, -0.086274509803922,
+0.105882352941176, +0.105882352941176, -0.086274509803922, -0.105882352941176,
-0.098039215686275, -0.156862745098039, -0.058823529411765, +0.113725490196078,
+0.007843137254902, -0.200000000000000, -0.039215686274510, +0.074509803921569,
-0.176470588235294, -0.172549019607843, -0.062745098039216, -0.125490196078431,
-0.137254901960784, -0.180392156862745, -0.227450980392157, -0.031372549019608,
+0.098039215686275, -0.109803921568627, -0.180392156862745, -0.129411764705882,
-0.231372549019608, -0.250980392156863, -0.086274509803922, +0.066666666666667,
-0.094117647058824, -0.121568627450980, +0.031372549019608, -0.133333333333333,
-0.160784313725490, +0.015686274509804, +0.023529411764706, -0.015686274509804,
+0.062745098039216, -0.015686274509804, -0.137254901960784, -0.015686274509804,
-0.098039215686275, -0.211764705882353, +0.000000000000000, +0.027450980392157,
-0.184313725490196, -0.168627450980392, -0.027450980392157, +0.050980392156863,
+0.066666666666667, +0.129411764705882, -0.011764705882353, -0.192156862745098,
-0.011764705882353, +0.164705882352941, +0.000000000000000, -0.015686274509804,
+0.023529411764706, -0.145098039215686, +0.058823529411765, +0.192156862745098,
+0.011764705882353, +0.090196078431373, +0.125490196078431, -0.082352941176471,
-0.188235294117647, -0.050980392156863, +0.023529411764706, -0.129411764705882,
-0.117647058823529, +0.141176470588235, -0.015686274509804, -0.219607843137255,
+0.000000000000000, -0.015686274509804, -0.050980392156863, +0.078431372549020,
-0.086274509803922, -0.262745098039216, +0.000000000000000, +0.141176470588235,
-0.050980392156863, -0.023529411764706, +0.000000000000000, -0.188235294117647,
-0.160784313725490, +0.054901960784314, +0.062745098039216, -0.050980392156863,
+0.062745098039216, +0.035294117647059, -0.137254901960784, -0.180392156862745,
-0.039215686274510, +0.007843137254902, -0.031372549019608, +0.023529411764706,
-0.145098039215686, -0.125490196078431, +0.098039215686275, -0.007843137254902,
-0.082352941176471, +0.035294117647059, -0.050980392156863, -0.192156862745098,
-0.094117647058824, +0.149019607843137, +0.050980392156863, -0.137254901960784,
+0.058823529411765, +0.019607843137255, -0.117647058823529, -0.003921568627451,
+0.050980392156863, -0.043137254901961, -0.109803921568627, +0.086274509803922,
-0.023529411764706, -0.117647058823529, +0.070588235294118, -0.035294117647059,
+0.031372549019608, -0.011764705882353, +0.156862745098039, +0.101960784313725,
+0.011764705882353, +0.121568627450980, +0.007843137254902, -0.113725490196078,
+0.066666666666667, +0.192156862745098, +0.074509803921569, +0.101960784313725,
+0.141176470588235, +0.019607843137255, +0.094117647058824, +0.223529411764706,
+0.160784313725490, +0.011764705882353, +0.121568627450980, +0.239215686274510,
+0.031372549019608, +0.031372549019608, +0.125490196078431, +0.070588235294118,
+0.098039215686275, +0.054901960784314, -0.156862745098039, +0.007843137254902,
+0.211764705882353, +0.074509803921569, +0.047058823529412, +0.109803921568627,
+0.023529411764706, -0.117647058823529, -0.101960784313725, +0.027450980392157,
-0.070588235294118, -0.082352941176471, +0.137254901960784, -0.066666666666667,
-0.184313725490196, -0.011764705882353, -0.145098039215686, -0.156862745098039,
+0.113725490196078, +0.078431372549020, -0.094117647058824, +0.090196078431373,
+0.086274509803922, -0.094117647058824, +0.031372549019608, +0.090196078431373,
-0.105882352941176, -0.086274509803922, -0.015686274509804, -0.137254901960784,
-0.168627450980392, +0.082352941176471, +0.047058823529412, -0.152941176470588,
-0.043137254901961, +0.019607843137255, -0.094117647058824, +0.011764705882353,
+0.074509803921569, -0.207843137254902, -0.094117647058824, +0.035294117647059,
-0.125490196078431, +0.019607843137255, +0.137254901960784, -0.003921568627451,
-0.066666666666667, -0.031372549019608, -0.054901960784314, -0.168627450980392,
-0.160784313725490, +0.149019607843137, +0.082352941176471, -0.200000000000000,
-0.062745098039216, -0.062745098039216, -0.047058823529412, +0.054901960784314,
-0.062745098039216, -0.235294117647059, -0.078431372549020, +0.047058823529412,
-0.149019607843137, +0.011764705882353, +0.168627450980392, -0.113725490196078,
-0.094117647058824, +0.011764705882353, -0.156862745098039, -0.125490196078431,
+0.125490196078431, +0.113725490196078, -0.050980392156863, -0.086274509803922,
-0.098039215686275, -0.168627450980392, -0.117647058823529, +0.000000000000000,
-0.137254901960784, -0.215686274509804, +0.011764705882353, -0.054901960784314,
-0.168627450980392, +0.062745098039216, +0.058823529411765, -0.172549019607843,
-0.141176470588235, +0.027450980392157, -0.058823529411765, -0.184313725490196,
-0.023529411764706, -0.047058823529412, -0.254901960784314, -0.101960784313725,
+0.007843137254902, -0.098039215686275, -0.094117647058824, +0.003921568627451,
-0.105882352941176, -0.066666666666667, +0.074509803921569, -0.086274509803922,
-0.121568627450980, -0.239215686274510, -0.074509803921569, -0.031372549019608,
-0.156862745098039, -0.125490196078431, -0.168627450980392, -0.223529411764706,
-0.062745098039216, +0.050980392156863, -0.086274509803922, -0.031372549019608,
+0.003921568627451, -0.152941176470588, -0.105882352941176, -0.019607843137255,
-0.074509803921569, -0.074509803921569, +0.043137254901961, +0.070588235294118,
-0.050980392156863, -0.007843137254902, +0.043137254901961, -0.031372549019608,
+0.105882352941176, +0.188235294117647, -0.070588235294118, -0.145098039215686,
+0.094117647058824, +0.133333333333333, +0.054901960784314, +0.137254901960784,
+0.152941176470588, +0.043137254901961, +0.035294117647059, +0.109803921568627,
+0.007843137254902, -0.031372549019608, +0.211764705882353, +0.133333333333333,
-0.007843137254902, +0.156862745098039, +0.050980392156863, -0.074509803921569,
+0.109803921568627, +0.129411764705882, +0.000000000000000, +0.113725490196078,
+0.200000000000000, +0.094117647058824, +0.129411764705882, +0.141176470588235,
-0.054901960784314, -0.003921568627451, +0.141176470588235, +0.035294117647059,
-0.082352941176471, +0.125490196078431, +0.101960784313725, -0.074509803921569,
+0.109803921568627, +0.113725490196078, -0.117647058823529, +0.031372549019608,
+0.243137254901961, +0.011764705882353, +0.007843137254902, +0.039215686274510,
-0.184313725490196, -0.011764705882353, +0.200000000000000, +0.101960784313725,
+0.117647058823529, +0.219607843137255, +0.192156862745098, +0.019607843137255,
-0.066666666666667, +0.168627450980392, +0.113725490196078, -0.121568627450980,
+0.003921568627451, -0.007843137254902, +0.094117647058824, +0.235294117647059,
+0.094117647058824, -0.054901960784314, +0.035294117647059, +0.168627450980392,
-0.039215686274510, +0.050980392156863, +0.282352941176471, +0.039215686274510,
+0.019607843137255, +0.196078431372549, -0.058823529411765, -0.152941176470588,
+0.129411764705882, +0.094117647058824, -0.074509803921569, +0.050980392156863,
+0.050980392156863, -0.058823529411765, +0.062745098039216, +0.227450980392157,
+0.054901960784314, -0.098039215686275, +0.149019607843137, +0.133333333333333,
-0.047058823529412, +0.164705882352941, +0.254901960784314, -0.086274509803922,
-0.113725490196078, +0.078431372549020, -0.019607843137255, +0.000000000000000,
+0.129411764705882, +0.074509803921569, -0.121568627450980, +0.023529411764706,
+0.211764705882353, +0.090196078431373, +0.101960784313725, +0.156862745098039,
-0.015686274509804, -0.023529411764706, +0.164705882352941, +0.105882352941176,
+0.098039215686275, -0.074509803921569, +0.050980392156863, +0.113725490196078,
+0.027450980392157, +0.078431372549020, +0.086274509803922, -0.023529411764706,
+0.050980392156863, +0.105882352941176, -0.090196078431373, -0.050980392156863,
+0.000000000000000, -0.101960784313725, -0.039215686274510, -0.011764705882353,
-0.211764705882353, -0.180392156862745, +0.007843137254902, -0.054901960784314,
-0.156862745098039, -0.019607843137255, +0.011764705882353, -0.215686274509804,
-0.113725490196078, +0.098039215686275, -0.007843137254902, -0.176470588235294,
-0.031372549019608, -0.019607843137255, -0.172549019607843, -0.062745098039216,
+0.000000000000000, -0.117647058823529, -0.105882352941176, +0.050980392156863,
-0.094117647058824, -0.156862745098039, +0.007843137254902, -0.050980392156863,
-0.133333333333333, +0.000000000000000, -0.003921568627451, -0.149019607843137,
-0.011764705882353, -0.047058823529412, -0.227450980392157, -0.149019607843137,
-0.039215686274510, -0.105882352941176, -0.094117647058824, -0.003921568627451,
-0.149019607843137, -0.164705882352941, -0.023529411764706, -0.066666666666667,
-0.098039215686275, +0.066666666666667, -0.003921568627451, -0.180392156862745,
+0.011764705882353, +0.086274509803922, -0.152941176470588, -0.098039215686275,
+0.050980392156863, -0.105882352941176, -0.007843137254902, +0.101960784313725,
-0.160784313725490, -0.160784313725490, -0.031372549019608, -0.117647058823529,
-0.164705882352941, -0.003921568627451, +0.098039215686275, -0.105882352941176,
-0.141176470588235, +0.094117647058824, +0.023529411764706, -0.149019607843137,
-0.101960784313725, -0.176470588235294, -0.047058823529412, +0.117647058823529,
-0.050980392156863, -0.176470588235294, -0.019607843137255, +0.113725490196078,
-0.105882352941176, -0.082352941176471, +0.125490196078431, -0.019607843137255,
-0.086274509803922, +0.105882352941176, -0.027450980392157, -0.125490196078431,
+0.129411764705882, +0.031372549019608, -0.200000000000000, +0.043137254901961,
+0.137254901960784, -0.090196078431373, -0.054901960784314, +0.105882352941176,
-0.047058823529412, -0.200000000000000, +0.035294117647059, +0.047058823529412,
-0.141176470588235, -0.007843137254902, +0.160784313725490, -0.101960784313725,
-0.164705882352941, +0.043137254901961, -0.070588235294118, -0.054901960784314,
+0.164705882352941, +0.074509803921569, -0.168627450980392, -0.054901960784314,
+0.125490196078431, -0.027450980392157, -0.035294117647059, +0.074509803921569,
-0.007843137254902, -0.043137254901961, +0.035294117647059, -0.035294117647059,
+0.121568627450980, -0.031372549019608, +0.066666666666667, +0.164705882352941,
+0.149019607843137, +0.145098039215686, +0.074509803921569, -0.019607843137255,
+0.117647058823529, +0.200000000000000, +0.105882352941176, +0.137254901960784,
+0.172549019607843, +0.050980392156863, +0.098039215686275, +0.274509803921569,
+0.047058823529412, -0.078431372549020, +0.156862745098039, +0.129411764705882,
-0.058823529411765, +0.062745098039216, +0.219607843137255, -0.031372549019608,
-0.078431372549020, +0.125490196078431, +0.062745098039216, -0.039215686274510,
+0.156862745098039, +0.168627450980392, +0.019607843137255, +0.101960784313725,
+0.168627450980392, -0.023529411764706, -0.188235294117647, +0.054901960784314,
-0.003921568627451, -0.176470588235294, -0.050980392156863, -0.074509803921569,
-0.239215686274510, -0.113725490196078, +0.039215686274510, -0.180392156862745,
-0.023529411764706, +0.137254901960784, -0.054901960784314, -0.090196078431373,
+0.050980392156863, +0.043137254901961, +0.011764705882353, +0.113725490196078,
+0.058823529411765, +0.003921568627451, +0.086274509803922, -0.070588235294118,
-0.117647058823529, +0.125490196078431, +0.062745098039216, -0.156862745098039,
-0.007843137254902, +0.117647058823529, -0.101960784313725, -0.047058823529412,
+0.035294117647059, -0.156862745098039, -0.082352941176471, +0.113725490196078,
-0.023529411764706, -0.094117647058824, -0.019607843137255, -0.054901960784314,
-0.141176470588235, -0.039215686274510, +0.062745098039216, -0.188235294117647,
-0.129411764705882, +0.156862745098039, +0.094117647058824, -0.047058823529412,
-0.054901960784314, -0.149019607843137, -0.027450980392157, +0.094117647058824,
-0.098039215686275, -0.211764705882353, -0.035294117647059, +0.101960784313725,
-0.176470588235294, -0.168627450980392, +0.086274509803922, -0.027450980392157,
-0.137254901960784, +0.047058823529412, -0.050980392156863, -0.094117647058824,
+0.192156862745098, +0.156862745098039, -0.054901960784314, +0.050980392156863,
+0.180392156862745, -0.050980392156863, -0.074509803921569, +0.090196078431373,
-0.035294117647059, -0.231372549019608, -0.105882352941176, -0.015686274509804,
-0.141176470588235, -0.019607843137255, +0.149019607843137, +0.000000000000000,
-0.094117647058824, +0.074509803921569, -0.007843137254902, -0.101960784313725,
+0.121568627450980, +0.109803921568627, -0.125490196078431, -0.113725490196078,
+0.082352941176471, -0.074509803921569, -0.156862745098039, +0.015686274509804,
-0.054901960784314, -0.039215686274510, +0.058823529411765, -0.117647058823529,
-0.039215686274510, -0.098039215686275, +0.019607843137255, -0.062745098039216,
-0.168627450980392, -0.082352941176471, -0.109803921568627, -0.247058823529412,
-0.054901960784314, -0.003921568627451, -0.078431372549020, +0.000000000000000,
+0.113725490196078, -0.078431372549020, -0.168627450980392, +0.113725490196078,
+0.039215686274510, -0.101960784313725, +0.101960784313725, +0.074509803921569,
-0.164705882352941, -0.137254901960784, +0.125490196078431, +0.094117647058824,
+0.031372549019608, +0.117647058823529, -0.062745098039216, -0.117647058823529,
+0.121568627450980, +0.137254901960784, +0.047058823529412, +0.101960784313725,
+0.250980392156863, +0.172549019607843, -0.050980392156863, +0.113725490196078,
+0.188235294117647, +0.050980392156863, +0.145098039215686, +0.203921568627451,
-0.019607843137255, +0.000000000000000, +0.219607843137255, +0.015686274509804,
+0.003921568627451, +0.145098039215686, +0.054901960784314, +0.050980392156863,
+0.168627450980392, +0.117647058823529, +0.078431372549020, +0.200000000000000,
+0.117647058823529, +0.003921568627451, +0.160784313725490, +0.105882352941176,
-0.062745098039216, +0.164705882352941, +0.133333333333333, -0.043137254901961,
+0.133333333333333, +0.188235294117647, -0.047058823529412, +0.031372549019608,
+0.243137254901961, +0.125490196078431, +0.058823529411765, +0.192156862745098,
+0.125490196078431, +0.027450980392157, +0.176470588235294, +0.137254901960784,
-0.031372549019608, +0.117647058823529, +0.250980392156863, -0.019607843137255,
-0.039215686274510, +0.215686274509804, +0.168627450980392, +0.074509803921569,
+0.129411764705882, +0.043137254901961, +0.117647058823529, +0.266666666666667,
+0.043137254901961, -0.121568627450980, +0.070588235294118, +0.243137254901961,
+0.031372549019608, -0.066666666666667, +0.176470588235294, +0.113725490196078,
-0.039215686274510, +0.149019607843137, +0.094117647058824, -0.011764705882353,
+0.168627450980392, +0.121568627450980, -0.023529411764706, +0.003921568627451,
+0.168627450980392, +0.090196078431373, +0.050980392156863, +0.219607843137255,
+0.168627450980392, +0.019607843137255, +0.047058823529412, +0.121568627450980,
+0.043137254901961, +0.035294117647059, +0.133333333333333, +0.058823529411765,
-0.019607843137255, +0.149019607843137, +0.098039215686275, -0.007843137254902,
+0.137254901960784, +0.113725490196078, -0.047058823529412, -0.011764705882353,
+0.207843137254902, +0.105882352941176, -0.062745098039216, +0.125490196078431,
+0.062745098039216, +0.054901960784314, +0.235294117647059, +0.149019607843137,
+0.019607843137255, -0.078431372549020, +0.125490196078431, +0.101960784313725,
-0.074509803921569, +0.031372549019608, +0.090196078431373, -0.160784313725490,
-0.074509803921569, +0.023529411764706, -0.211764705882353, -0.172549019607843,
+0.137254901960784, +0.000000000000000, -0.176470588235294, +0.039215686274510,
-0.066666666666667, -0.215686274509804, +0.035294117647059, +0.125490196078431,
-0.117647058823529, -0.168627450980392, -0.019607843137255, -0.074509803921569,
-0.050980392156863, +0.086274509803922, -0.105882352941176, -0.152941176470588,
-0.003921568627451, -0.113725490196078, -0.219607843137255, -0.105882352941176,
+0.105882352941176, +0.023529411764706, -0.098039215686275, -0.007843137254902,
-0.019607843137255, -0.015686274509804, +0.082352941176471, +0.031372549019608,
-0.074509803921569, +0.000000000000000, +0.058823529411765, -0.086274509803922,
-0.023529411764706, -0.043137254901961, -0.180392156862745, -0.062745098039216,
+0.035294117647059, -0.141176470588235, -0.172549019607843, -0.007843137254902,
-0.082352941176471, -0.180392156862745, -0.023529411764706, +0.019607843137255,
-0.172549019607843, +0.003921568627451, -0.031372549019608, -0.180392156862745,
+0.035294117647059, +0.058823529411765, -0.156862745098039, -0.113725490196078,
+0.047058823529412, +0.000000000000000, -0.007843137254902, +0.047058823529412,
-0.039215686274510, -0.180392156862745, +0.019607843137255, +0.039215686274510,
-0.176470588235294, +0.015686274509804, +0.137254901960784, -0.078431372549020,
-0.113725490196078, +0.129411764705882, +0.050980392156863, -0.066666666666667,
-0.007843137254902, -0.105882352941176, -0.011764705882353, +0.223529411764706,
-0.007843137254902, -0.247058823529412, -0.047058823529412, +0.031372549019608,
-0.117647058823529, -0.117647058823529, +0.066666666666667, -0.019607843137255,
-0.145098039215686, -0.039215686274510, -0.105882352941176, -0.168627450980392,
-0.031372549019608, -0.086274509803922, -0.164705882352941, -0.105882352941176,
-0.023529411764706, -0.074509803921569, -0.188235294117647, +0.019607843137255,
+0.070588235294118, -0.062745098039216, +0.027450980392157, +0.027450980392157,
-0.062745098039216, -0.098039215686275, -0.011764705882353, -0.074509803921569,
-0.149019607843137, -0.019607843137255, -0.117647058823529, -0.152941176470588,
+0.050980392156863, -0.035294117647059, -0.258823529411765, -0.137254901960784,
+0.027450980392157, -0.043137254901961, -0.180392156862745, -0.050980392156863,
-0.043137254901961, -0.031372549019608, +0.141176470588235, +0.043137254901961,
-0.047058823529412, -0.168627450980392, +0.043137254901961, +0.105882352941176,
+0.000000000000000, +0.094117647058824, +0.176470588235294, +0.011764705882353,
+0.050980392156863, +0.168627450980392, -0.125490196078431, -0.168627450980392,
+0.196078431372549, +0.105882352941176, -0.074509803921569, +0.156862745098039,
+0.050980392156863, -0.149019607843137, +0.078431372549020, +0.243137254901961,
-0.007843137254902, -0.047058823529412, +0.141176470588235, +0.019607843137255,
+0.000000000000000, +0.141176470588235, -0.011764705882353, -0.058823529411765,
+0.109803921568627, -0.003921568627451, -0.207843137254902, -0.101960784313725,
+0.156862745098039, +0.031372549019608, -0.043137254901961, +0.101960784313725,
-0.050980392156863, -0.062745098039216, +0.137254901960784, -0.011764705882353,
-0.141176470588235, +0.070588235294118, +0.090196078431373, -0.094117647058824,
+0.074509803921569, +0.086274509803922, -0.117647058823529, +0.039215686274510,
+0.137254901960784, -0.129411764705882, -0.152941176470588, +0.066666666666667,
-0.039215686274510, -0.074509803921569, +0.035294117647059, +0.007843137254902,
-0.086274509803922, +0.058823529411765, -0.043137254901961, -0.231372549019608,
-0.047058823529412, -0.003921568627451, -0.160784313725490, -0.090196078431373,
+0.027450980392157, -0.129411764705882, -0.043137254901961, +0.066666666666667,
-0.047058823529412, -0.149019607843137, +0.047058823529412, +0.043137254901961,
-0.172549019607843, -0.015686274509804, +0.003921568627451, -0.211764705882353,
-0.156862745098039, +0.086274509803922, -0.058823529411765, -0.141176470588235,
-0.047058823529412, -0.227450980392157, -0.145098039215686, +0.207843137254902,
+0.054901960784314, -0.200000000000000, +0.000000000000000, +0.031372549019608,
-0.196078431372549, -0.145098039215686, +0.098039215686275, +0.019607843137255,
-0.043137254901961, +0.011764705882353, -0.133333333333333, -0.129411764705882,
+0.027450980392157, -0.023529411764706, -0.101960784313725, +0.023529411764706,
+0.062745098039216, -0.015686274509804, -0.149019607843137, +0.011764705882353,
+0.062745098039216, -0.176470588235294, -0.043137254901961, -0.003921568627451,
-0.082352941176471, +0.003921568627451, +0.149019607843137, +0.062745098039216,
-0.015686274509804, +0.050980392156863, -0.094117647058824, -0.090196078431373,
+0.149019607843137, +0.133333333333333, -0.156862745098039, -0.062745098039216,
+0.070588235294118, -0.062745098039216, -0.207843137254902, -0.050980392156863,
-0.086274509803922, -0.156862745098039, +0.082352941176471, -0.086274509803922,
-0.074509803921569, -0.176470588235294, +0.031372549019608, +0.000000000000000,
-0.196078431372549, -0.062745098039216, -0.050980392156863, -0.113725490196078,
+0.082352941176471, +0.227450980392157, -0.090196078431373, -0.247058823529412,
+0.094117647058824, +0.078431372549020, -0.152941176470588, +0.109803921568627,
+0.070588235294118, -0.141176470588235, +0.000000000000000, +0.074509803921569,
-0.145098039215686, -0.211764705882353, -0.019607843137255, -0.031372549019608,
-0.058823529411765, +0.000000000000000, -0.113725490196078, -0.137254901960784,
+0.156862745098039, +0.121568627450980, -0.137254901960784, -0.113725490196078,
+0.027450980392157, -0.019607843137255, -0.050980392156863, +0.125490196078431,
+0.031372549019608, -0.070588235294118, +0.160784313725490, +0.082352941176471,
-0.113725490196078, +0.062745098039216, +0.062745098039216, -0.094117647058824,
+0.086274509803922, +0.105882352941176, -0.062745098039216, +0.105882352941176,
+0.262745098039216, -0.007843137254902, -0.054901960784314, +0.215686274509804,
+0.058823529411765, -0.011764705882353, +0.156862745098039, +0.117647058823529,
+0.082352941176471, +0.211764705882353, +0.164705882352941, +0.027450980392157,
+0.129411764705882, +0.156862745098039, +0.007843137254902, +0.039215686274510,
+0.156862745098039, +0.000000000000000, +0.074509803921569, +0.231372549019608,
+0.000000000000000, -0.094117647058824, +0.176470588235294, +0.149019607843137,
-0.054901960784314, +0.121568627450980, +0.203921568627451, -0.054901960784314,
-0.031372549019608, +0.207843137254902, +0.062745098039216, -0.043137254901961,
+0.133333333333333, -0.035294117647059, -0.156862745098039, +0.188235294117647,
+0.098039215686275, -0.156862745098039, +0.082352941176471, +0.207843137254902,
-0.043137254901961, -0.074509803921569, +0.105882352941176, -0.023529411764706,
+0.000000000000000, +0.211764705882353, +0.133333333333333, +0.101960784313725,
+0.215686274509804, +0.149019607843137, -0.031372549019608, +0.062745098039216,
+0.200000000000000, +0.168627450980392, +0.050980392156863, +0.090196078431373,
+0.094117647058824, -0.094117647058824, +0.039215686274510, +0.117647058823529,
-0.011764705882353, +0.019607843137255, +0.066666666666667, -0.019607843137255,
+0.074509803921569, +0.211764705882353, +0.050980392156863, -0.015686274509804,
+0.160784313725490, +0.203921568627451, -0.027450980392157, +0.000000000000000,
+0.227450980392157, +0.168627450980392, -0.035294117647059, +0.105882352941176,
+0.023529411764706, -0.141176470588235, +0.117647058823529, +0.054901960784314,
+0.137254901960784, -0.007843137254902, +0.145098039215686, +0.168627450980392,
-0.086274509803922, +0.007843137254902, +0.023529411764706, -0.109803921568627,
+0.054901960784314, +0.278431372549020, +0.121568627450980, -0.152941176470588,
+0.066666666666667, +0.152941176470588, -0.078431372549020, +0.098039215686275,
+0.054901960784314, -0.168627450980392, +0.023529411764706, +0.062745098039216,
-0.098039215686275, -0.086274509803922, -0.019607843137255, -0.086274509803922,
-0.113725490196078, -0.031372549019608, -0.074509803921569, -0.133333333333333,
+0.105882352941176, +0.066666666666667, -0.129411764705882, -0.035294117647059,
-0.058823529411765, -0.125490196078431, -0.149019607843137, +0.000000000000000,
-0.054901960784314, -0.219607843137255, +0.019607843137255, +0.031372549019608,
-0.168627450980392, +0.023529411764706, +0.000000000000000, -0.231372549019608,
-0.050980392156863, -0.047058823529412, -0.211764705882353, -0.031372549019608,
+0.200000000000000, -0.027450980392157, -0.109803921568627, +0.105882352941176,
-0.035294117647059, -0.137254901960784, +0.023529411764706, -0.011764705882353,
-0.117647058823529, +0.035294117647059, +0.050980392156863, -0.019607843137255,
+0.113725490196078, +0.086274509803922, -0.098039215686275, -0.054901960784314,
+0.019607843137255, -0.137254901960784, -0.074509803921569, +0.098039215686275,
-0.117647058823529, -0.270588235294118, -0.023529411764706, +0.011764705882353,
-0.156862745098039, +0.000000000000000, +0.192156862745098, -0.050980392156863,
-0.098039215686275, +0.137254901960784, +0.094117647058824, -0.105882352941176,
+0.047058823529412, +0.035294117647059, -0.200000000000000, +0.011764705882353,
-0.003921568627451, -0.243137254901961, -0.043137254901961, +0.149019607843137,
-0.105882352941176, -0.125490196078431, +0.023529411764706, -0.156862745098039,
-0.188235294117647, +0.043137254901961, +0.000000000000000, -0.078431372549020,
+0.074509803921569, +0.043137254901961, -0.137254901960784, -0.149019607843137,
+0.007843137254902, -0.019607843137255, -0.105882352941176, -0.062745098039216,
-0.074509803921569, -0.113725490196078, +0.039215686274510, +0.074509803921569,
-0.141176470588235, -0.160784313725490, -0.109803921568627, -0.231372549019608,
-0.200000000000000, +0.043137254901961, -0.011764705882353, -0.168627450980392,
-0.031372549019608, +0.011764705882353, -0.129411764705882, -0.200000000000000,
-0.011764705882353, +0.015686274509804, -0.101960784313725, +0.003921568627451,
-0.074509803921569, -0.203921568627451, -0.031372549019608, -0.019607843137255,
+0.039215686274510, -0.039215686274510, +0.043137254901961, +0.156862745098039,
+0.039215686274510, +0.047058823529412, +0.054901960784314, -0.086274509803922,
-0.019607843137255, +0.156862745098039, +0.176470588235294, -0.007843137254902,
+0.050980392156863, +0.058823529411765, -0.098039215686275, +0.101960784313725,
+0.188235294117647, +0.000000000000000, +0.156862745098039, +0.192156862745098,
-0.027450980392157, +0.070588235294118, +0.231372549019608, +0.156862745098039,
+0.098039215686275, +0.133333333333333, +0.054901960784314, -0.039215686274510,
+0.137254901960784, +0.168627450980392, +0.039215686274510, +0.164705882352941,
+0.149019607843137, +0.043137254901961, -0.019607843137255, +0.113725490196078,
+0.101960784313725, -0.094117647058824, +0.019607843137255, +0.082352941176471,
-0.054901960784314, +0.129411764705882, +0.160784313725490, -0.164705882352941,
-0.098039215686275, -0.027450980392157, -0.168627450980392, -0.043137254901961,
+0.196078431372549, -0.050980392156863, -0.125490196078431, +0.109803921568627,
-0.043137254901961, -0.176470588235294, +0.011764705882353, -0.023529411764706,
-0.188235294117647, +0.007843137254902, -0.058823529411765, -0.184313725490196,
+0.019607843137255, +0.054901960784314, -0.176470588235294, -0.086274509803922,
+0.094117647058824, -0.035294117647059, -0.043137254901961, +0.054901960784314,
-0.054901960784314, -0.200000000000000, -0.023529411764706, +0.074509803921569,
-0.117647058823529, -0.086274509803922, +0.117647058823529, -0.066666666666667,
-0.168627450980392, +0.031372549019608, +0.078431372549020, -0.070588235294118,
+0.023529411764706, +0.137254901960784, +0.023529411764706, +0.074509803921569,
+0.058823529411765, -0.101960784313725, -0.015686274509804, +0.192156862745098,
-0.062745098039216, -0.047058823529412, +0.207843137254902, +0.011764705882353,
-0.188235294117647, -0.003921568627451, -0.043137254901961, -0.231372549019608,
+0.000000000000000, +0.039215686274510, -0.094117647058824, -0.137254901960784,
-0.031372549019608, +0.043137254901961, +0.035294117647059, +0.098039215686275,
+0.074509803921569, +0.039215686274510, +0.145098039215686, +0.156862745098039,
-0.035294117647059, +0.007843137254902, +0.145098039215686, -0.015686274509804,
-0.231372549019608, -0.019607843137255, +0.019607843137255, -0.176470588235294,
+0.007843137254902, +0.109803921568627, +0.050980392156863, -0.050980392156863,
+0.031372549019608, -0.027450980392157, -0.156862745098039, -0.054901960784314,
-0.058823529411765, -0.082352941176471, +0.054901960784314, +0.031372549019608,
-0.156862745098039, -0.180392156862745, -0.149019607843137, -0.109803921568627,
-0.133333333333333, -0.094117647058824, -0.109803921568627, -0.215686274509804,
-0.125490196078431, -0.082352941176471, -0.117647058823529, -0.188235294117647,
-0.035294117647059, -0.062745098039216, -0.290196078431373, -0.121568627450980,
+0.011764705882353, -0.090196078431373, +0.043137254901961, +0.133333333333333,
-0.149019607843137, -0.129411764705882, +0.062745098039216, -0.039215686274510,
+0.011764705882353, +0.043137254901961, -0.094117647058824, -0.098039215686275,
+0.019607843137255, +0.031372549019608, -0.105882352941176, +0.011764705882353,
+0.062745098039216, -0.082352941176471, -0.058823529411765, +0.117647058823529,
+0.105882352941176, +0.027450980392157, +0.039215686274510, -0.043137254901961,
-0.054901960784314, +0.101960784313725, +0.149019607843137, +0.047058823529412,
+0.031372549019608, -0.023529411764706, -0.172549019607843, -0.105882352941176,
+0.164705882352941, +0.043137254901961, -0.019607843137255, +0.200000000000000,
-0.003921568627451, -0.184313725490196, +0.125490196078431, +0.094117647058824,
-0.078431372549020, +0.176470588235294, +0.035294117647059, -0.125490196078431,
+0.105882352941176, +0.223529411764706, -0.015686274509804, -0.054901960784314,
+0.168627450980392, +0.090196078431373, +0.094117647058824, +0.239215686274510,
+0.164705882352941, +0.019607843137255, +0.074509803921569, +0.113725490196078,
+0.023529411764706, -0.027450980392157, +0.050980392156863, -0.086274509803922,
-0.094117647058824, +0.152941176470588, +0.125490196078431, +0.000000000000000,
-0.050980392156863, +0.043137254901961, +0.188235294117647, +0.239215686274510,
+0.117647058823529, -0.007843137254902, +0.058823529411765, +0.250980392156863,
+0.007843137254902, -0.066666666666667, +0.266666666666667, +0.176470588235294,
-0.027450980392157, +0.149019607843137, +0.192156862745098, -0.062745098039216,
+0.094117647058824, +0.215686274509804, +0.105882352941176, +0.070588235294118,
+0.066666666666667, +0.137254901960784, +0.172549019607843, +0.188235294117647,
+0.062745098039216, -0.035294117647059, +0.125490196078431, +0.247058823529412,
+0.125490196078431, +0.078431372549020, +0.211764705882353, +0.086274509803922,
-0.098039215686275, +0.125490196078431, +0.113725490196078, -0.078431372549020,
+0.098039215686275, +0.156862745098039, +0.074509803921569, +0.062745098039216,
+0.192156862745098, +0.215686274509804, +0.117647058823529, +0.094117647058824,
-0.050980392156863, -0.082352941176471, +0.113725490196078, +0.109803921568627,
+0.043137254901961, +0.023529411764706, +0.082352941176471, +0.062745098039216,
-0.054901960784314, +0.023529411764706, +0.019607843137255, -0.109803921568627,
+0.050980392156863, +0.094117647058824, +0.023529411764706, -0.094117647058824,
+0.070588235294118, +0.125490196078431, -0.168627450980392, -0.168627450980392,
-0.094117647058824, -0.176470588235294, -0.050980392156863, +0.109803921568627,
-0.152941176470588, -0.137254901960784, +0.086274509803922, -0.137254901960784,
-0.047058823529412, +0.082352941176471, -0.141176470588235, -0.098039215686275,
+0.027450980392157, -0.062745098039216, -0.239215686274510, -0.109803921568627,
+0.062745098039216, -0.149019607843137, -0.176470588235294, +0.011764705882353,
-0.094117647058824, -0.050980392156863, +0.121568627450980, -0.145098039215686,
-0.149019607843137, +0.039215686274510, -0.054901960784314, +0.054901960784314,
+0.184313725490196, +0.003921568627451, -0.168627450980392, -0.137254901960784,
-0.019607843137255, -0.101960784313725, -0.054901960784314, +0.160784313725490,
-0.039215686274510, -0.223529411764706, +0.090196078431373, +0.011764705882353,
-0.219607843137255, +0.086274509803922, -0.043137254901961, -0.223529411764706,
-0.039215686274510, +0.074509803921569, -0.082352941176471, -0.137254901960784,
+0.047058823529412, -0.105882352941176, -0.113725490196078, +0.058823529411765,
-0.050980392156863, -0.082352941176471, +0.031372549019608, -0.074509803921569,
-0.062745098039216, +0.101960784313725, +0.054901960784314, -0.172549019607843,
-0.192156862745098, +0.023529411764706, -0.050980392156863, -0.082352941176471,
-0.137254901960784, -0.243137254901961, -0.101960784313725, +0.070588235294118,
-0.054901960784314, -0.180392156862745, -0.035294117647059, +0.137254901960784,
-0.082352941176471, -0.207843137254902, +0.062745098039216, -0.003921568627451,
-0.094117647058824, +0.019607843137255, +0.043137254901961, -0.054901960784314,
+0.003921568627451, +0.007843137254902, -0.125490196078431, -0.023529411764706,
+0.007843137254902, -0.113725490196078, -0.156862745098039, -0.007843137254902,
-0.105882352941176, -0.270588235294118, -0.082352941176471, -0.011764705882353,
-0.098039215686275, -0.129411764705882, +0.035294117647059, -0.050980392156863,
-0.203921568627451, +0.011764705882353, -0.039215686274510, -0.164705882352941,
+0.043137254901961, +0.019607843137255, -0.207843137254902, -0.160784313725490,
+0.027450980392157, +0.054901960784314, +0.043137254901961, +0.160784313725490,
-0.023529411764706, -0.207843137254902, -0.039215686274510, -0.011764705882353,
+0.098039215686275, +0.074509803921569, +0.176470588235294, +0.207843137254902,
+0.047058823529412, +0.125490196078431, +0.062745098039216, -0.149019607843137,
+0.062745098039216, +0.211764705882353, +0.141176470588235, +0.019607843137255,
+0.160784313725490, +0.219607843137255, +0.011764705882353, +0.039215686274510,
+0.145098039215686, +0.039215686274510, +0.054901960784314, +0.231372549019608,
-0.058823529411765, -0.105882352941176, +0.219607843137255, -0.047058823529412,
-0.039215686274510, +0.231372549019608, -0.043137254901961, -0.086274509803922,
+0.196078431372549, +0.109803921568627, -0.058823529411765, +0.027450980392157,
+0.231372549019608, +0.098039215686275, -0.058823529411765, +0.074509803921569,
-0.011764705882353, -0.019607843137255, +0.223529411764706, +0.023529411764706,
-0.023529411764706, +0.196078431372549, -0.007843137254902, -0.019607843137255,
+0.137254901960784, +0.078431372549020, -0.007843137254902, -0.054901960784314,
-0.023529411764706, -0.117647058823529, -0.058823529411765, +0.156862745098039,
-0.023529411764706, -0.121568627450980, +0.160784313725490, +0.113725490196078,
-0.129411764705882, +0.098039215686275, +0.031372549019608, -0.203921568627451,
-0.007843137254902, +0.121568627450980, -0.094117647058824, -0.109803921568627,
+0.113725490196078, -0.090196078431373, -0.125490196078431, +0.011764705882353,
-0.145098039215686, -0.149019607843137, +0.105882352941176, +0.023529411764706,
-0.129411764705882, +0.160784313725490, +0.200000000000000, -0.121568627450980,
-0.200000000000000, +0.043137254901961, +0.011764705882353, -0.050980392156863,
+0.054901960784314, -0.113725490196078, -0.223529411764706, +0.023529411764706,
+0.007843137254902, -0.200000000000000, -0.031372549019608, +0.105882352941176,
-0.070588235294118, -0.094117647058824, +0.109803921568627, -0.094117647058824,
-0.196078431372549, +0.019607843137255, -0.047058823529412, -0.180392156862745,
-0.031372549019608, -0.023529411764706, -0.200000000000000, -0.074509803921569,
+0.082352941176471, -0.015686274509804, -0.149019607843137, +0.082352941176471,
+0.105882352941176, -0.125490196078431, -0.019607843137255, -0.047058823529412,
-0.117647058823529, -0.015686274509804, +0.141176470588235, +0.054901960784314,
-0.070588235294118, +0.047058823529412, -0.105882352941176, -0.192156862745098,
+0.109803921568627, +0.164705882352941, -0.156862745098039, -0.196078431372549,
+0.043137254901961, +0.000000000000000, -0.188235294117647, +0.062745098039216,
+0.145098039215686, -0.101960784313725, -0.133333333333333, -0.160784313725490,
-0.094117647058824, -0.164705882352941, +0.023529411764706, +0.019607843137255,
-0.086274509803922, +0.050980392156863, -0.027450980392157, -0.227450980392157,
-0.125490196078431, +0.031372549019608, -0.094117647058824, -0.215686274509804,
+0.043137254901961, +0.129411764705882, -0.070588235294118, +0.000000000000000,
+0.113725490196078, -0.043137254901961, -0.023529411764706, +0.200000000000000,
-0.043137254901961, -0.200000000000000, +0.105882352941176, -0.074509803921569,
-0.145098039215686, +0.137254901960784, -0.062745098039216, -0.117647058823529,
+0.219607843137255, +0.062745098039216, -0.117647058823529, -0.007843137254902,
+0.015686274509804, +0.003921568627451, +0.031372549019608, +0.039215686274510,
-0.062745098039216, -0.019607843137255, +0.113725490196078, -0.023529411764706,
-0.078431372549020, +0.105882352941176, -0.003921568627451, +0.031372549019608,
+0.105882352941176, -0.043137254901961, -0.043137254901961, -0.050980392156863,
+0.086274509803922, +0.062745098039216, +0.027450980392157, +0.117647058823529,
-0.137254901960784, -0.211764705882353, +0.043137254901961, +0.125490196078431,
+0.031372549019608, +0.172549019607843, +0.203921568627451, -0.035294117647059,
+0.039215686274510, +0.200000000000000, +0.074509803921569, +0.054901960784314,
+0.231372549019608, +0.000000000000000, -0.043137254901961, +0.219607843137255,
+0.082352941176471, -0.070588235294118, +0.125490196078431, +0.101960784313725,
-0.172549019607843, +0.050980392156863, +0.262745098039216, +0.039215686274510,
-0.074509803921569, +0.149019607843137, +0.105882352941176, -0.094117647058824,
+0.121568627450980, +0.200000000000000, -0.007843137254902, +0.113725490196078,
+0.168627450980392, -0.023529411764706, +0.043137254901961, +0.133333333333333,
-0.015686274509804, -0.039215686274510, +0.168627450980392, +0.007843137254902,
-0.176470588235294, +0.113725490196078, +0.101960784313725, -0.129411764705882,
+0.078431372549020, +0.188235294117647, -0.027450980392157, -0.062745098039216,
+0.031372549019608, +0.090196078431373, +0.019607843137255, +0.113725490196078,
+0.145098039215686, -0.043137254901961, +0.152941176470588, +0.137254901960784,
-0.086274509803922, +0.015686274509804, +0.121568627450980, +0.054901960784314,
+0.074509803921569, +0.223529411764706, -0.019607843137255, -0.160784313725490,
+0.113725490196078, +0.168627450980392, -0.050980392156863, -0.117647058823529,
+0.156862745098039, +0.192156862745098, -0.113725490196078, -0.039215686274510,
+0.156862745098039, +0.121568627450980, +0.078431372549020, +0.047058823529412,
+0.047058823529412, -0.090196078431373, +0.054901960784314, +0.043137254901961,
+0.011764705882353, +0.141176470588235, +0.094117647058824, -0.039215686274510,
-0.011764705882353, +0.094117647058824, -0.011764705882353, -0.113725490196078,
+0.094117647058824, +0.156862745098039, -0.168627450980392, -0.156862745098039,
+0.062745098039216, -0.137254901960784, -0.125490196078431, +0.149019607843137,
-0.031372549019608, -0.121568627450980, +0.082352941176471, -0.062745098039216,
-0.086274509803922, +0.105882352941176, -0.062745098039216, -0.105882352941176,
+0.172549019607843, -0.019607843137255, -0.196078431372549, -0.003921568627451,
-0.066666666666667, -0.141176470588235, +0.027450980392157, +0.011764705882353,
-0.215686274509804, -0.074509803921569, +0.074509803921569, -0.074509803921569,
-0.145098039215686, +0.023529411764706, -0.101960784313725, -0.062745098039216,
+0.125490196078431, -0.011764705882353, -0.074509803921569, -0.074509803921569,
+0.074509803921569, -0.011764705882353, +0.019607843137255, +0.137254901960784,
-0.125490196078431, -0.200000000000000, -0.066666666666667, -0.098039215686275,
-0.125490196078431, +0.039215686274510, +0.058823529411765, -0.105882352941176,
-0.074509803921569, +0.035294117647059, -0.062745098039216, -0.086274509803922,
+0.094117647058824, -0.050980392156863, -0.145098039215686, +0.058823529411765,
-0.070588235294118, -0.172549019607843, +0.054901960784314, +0.062745098039216,
-0.211764705882353, -0.062745098039216, +0.176470588235294, -0.031372549019608,
-0.105882352941176, +0.105882352941176, +0.039215686274510, -0.211764705882353,
-0.074509803921569, +0.082352941176471, -0.082352941176471, -0.109803921568627,
-0.082352941176471, -0.098039215686275, -0.054901960784314, +0.007843137254902,
-0.113725490196078, -0.117647058823529, +0.066666666666667, -0.054901960784314,
-0.250980392156863, -0.007843137254902, +0.054901960784314, -0.145098039215686,
-0.039215686274510, -0.023529411764706, -0.125490196078431, -0.125490196078431,
-0.160784313725490, -0.125490196078431, -0.129411764705882, -0.113725490196078,
-0.054901960784314, -0.211764705882353, -0.007843137254902, +0.094117647058824,
-0.215686274509804, -0.113725490196078, +0.003921568627451, -0.160784313725490,
-0.176470588235294, +0.062745098039216, -0.066666666666667, -0.180392156862745,
+0.023529411764706, -0.003921568627451, -0.129411764705882, -0.223529411764706,
+0.031372549019608, +0.137254901960784, -0.145098039215686, -0.137254901960784,
-0.050980392156863, -0.062745098039216, +0.000000000000000, +0.000000000000000,
-0.054901960784314, -0.078431372549020, -0.113725490196078, -0.145098039215686,
-0.074509803921569, +0.074509803921569, +0.062745098039216, -0.086274509803922,
+0.043137254901961, +0.145098039215686, -0.031372549019608, -0.062745098039216,
+0.105882352941176, +0.156862745098039, -0.101960784313725, -0.113725490196078,
+0.176470588235294, +0.011764705882353, -0.050980392156863, +0.160784313725490,
-0.050980392156863, -0.007843137254902, +0.207843137254902, +0.039215686274510,
+0.043137254901961, +0.156862745098039, -0.027450980392157, -0.043137254901961,
+0.219607843137255, +0.027450980392157, -0.141176470588235, +0.098039215686275,
+0.113725490196078, -0.019607843137255, +0.050980392156863, +0.168627450980392,
-0.031372549019608, -0.058823529411765, +0.145098039215686, +0.101960784313725,
-0.003921568627451, +0.156862745098039, +0.082352941176471, -0.019607843137255,
+0.180392156862745, +0.207843137254902, +0.011764705882353, -0.098039215686275,
+0.125490196078431, -0.011764705882353, +0.000000000000000, +0.254901960784314,
+0.058823529411765, -0.047058823529412, +0.105882352941176, -0.039215686274510,
-0.160784313725490, +0.062745098039216, +0.082352941176471, -0.117647058823529,
-0.054901960784314, +0.101960784313725, -0.094117647058824, -0.156862745098039,
+0.066666666666667, +0.031372549019608, -0.058823529411765, -0.019607843137255,
-0.152941176470588, -0.188235294117647, +0.074509803921569, +0.101960784313725,
-0.192156862745098, -0.086274509803922, +0.145098039215686, -0.117647058823529,
-0.164705882352941, +0.149019607843137, +0.105882352941176, -0.207843137254902,
-0.121568627450980, +0.043137254901961, -0.074509803921569, -0.070588235294118,
-0.023529411764706, -0.094117647058824, -0.058823529411765, +0.078431372549020,
-0.094117647058824, -0.023529411764706, +0.211764705882353, +0.101960784313725,
-0.078431372549020, +0.070588235294118, +0.074509803921569, -0.101960784313725,
+0.011764705882353, +0.027450980392157, +0.003921568627451, +0.074509803921569,
+0.062745098039216, -0.003921568627451, -0.019607843137255, -0.050980392156863,
-0.062745098039216, -0.137254901960784, -0.003921568627451, +0.160784313725490,
-0.098039215686275, -0.007843137254902, +0.152941176470588, -0.054901960784314,
-0.094117647058824, +0.086274509803922, -0.054901960784314, -0.113725490196078,
+0.137254901960784, +0.094117647058824, +0.003921568627451, -0.113725490196078,
+0.043137254901961, +0.149019607843137, -0.094117647058824, -0.101960784313725,
-0.074509803921569, -0.211764705882353, -0.117647058823529, -0.066666666666667,
-0.062745098039216, +0.027450980392157, -0.011764705882353, -0.105882352941176,
-0.160784313725490, -0.098039215686275, -0.007843137254902, -0.180392156862745,
-0.031372549019608, +0.101960784313725, -0.152941176470588, -0.203921568627451,
+0.050980392156863, +0.105882352941176, -0.090196078431373, -0.082352941176471,
+0.027450980392157, -0.113725490196078, -0.141176470588235, +0.003921568627451,
-0.168627450980392, -0.219607843137255, -0.070588235294118, -0.141176470588235,
-0.082352941176471, +0.000000000000000, -0.188235294117647, -0.137254901960784,
+0.137254901960784, -0.027450980392157, -0.211764705882353, -0.011764705882353,
+0.035294117647059, -0.086274509803922, -0.066666666666667, +0.035294117647059,
-0.066666666666667, -0.176470588235294, -0.082352941176471, -0.062745098039216,
-0.082352941176471, +0.070588235294118, +0.039215686274510, -0.094117647058824,
-0.011764705882353, +0.035294117647059, -0.074509803921569, -0.101960784313725,
+0.113725490196078, -0.035294117647059, -0.149019607843137, +0.149019607843137,
+0.035294117647059, -0.168627450980392, +0.054901960784314, +0.050980392156863,
-0.117647058823529, +0.094117647058824, +0.164705882352941, -0.066666666666667,
-0.023529411764706, +0.223529411764706, +0.105882352941176, +0.007843137254902,
+0.145098039215686, +0.149019607843137, +0.168627450980392, +0.184313725490196,
-0.031372549019608, -0.117647058823529, +0.160784313725490, +0.231372549019608,
-0.011764705882353, +0.031372549019608, +0.270588235294118, +0.094117647058824,
-0.054901960784314, +0.200000000000000, +0.243137254901961, -0.043137254901961,
+0.011764705882353, +0.231372549019608, +0.176470588235294, +0.141176470588235,
+0.180392156862745, -0.011764705882353, -0.011764705882353, +0.239215686274510,
+0.066666666666667, +0.039215686274510, +0.231372549019608, +0.211764705882353,
+0.070588235294118, +0.145098039215686, +0.133333333333333, -0.031372549019608,
+0.105882352941176, +0.164705882352941, +0.015686274509804, +0.019607843137255,
+0.188235294117647, +0.188235294117647, +0.219607843137255, +0.223529411764706,
+0.105882352941176, +0.050980392156863, +0.094117647058824, +0.211764705882353,
+0.000000000000000, +0.035294117647059, +0.258823529411765, +0.113725490196078,
+0.086274509803922, +0.200000000000000, +0.054901960784314, -0.019607843137255,
+0.196078431372549, +0.176470588235294, +0.043137254901961, +0.000000000000000,
+0.078431372549020, +0.086274509803922, +0.043137254901961, +0.149019607843137,
+0.223529411764706, +0.000000000000000, +0.019607843137255, +0.164705882352941,
+0.137254901960784, +0.160784313725490, +0.211764705882353, +0.156862745098039,
+0.050980392156863, +0.035294117647059, +0.168627450980392, +0.015686274509804,
+0.047058823529412, +0.203921568627451, +0.062745098039216, -0.082352941176471,
+0.117647058823529, +0.188235294117647, -0.039215686274510, +0.050980392156863,
+0.105882352941176, -0.007843137254902, +0.011764705882353, +0.070588235294118,
-0.007843137254902, -0.078431372549020, -0.074509803921569, -0.192156862745098,
-0.066666666666667, +0.137254901960784, -0.062745098039216, -0.039215686274510,
+0.141176470588235, +0.015686274509804, -0.176470588235294, -0.043137254901961,
+0.105882352941176, -0.054901960784314, -0.050980392156863, +0.050980392156863,
-0.035294117647059, -0.105882352941176, -0.035294117647059, -0.078431372549020,
-0.117647058823529, -0.011764705882353, -0.066666666666667, -0.156862745098039,
-0.086274509803922, -0.090196078431373, -0.215686274509804, -0.070588235294118,
+0.133333333333333, -0.054901960784314, -0.192156862745098, +0.062745098039216,
+0.035294117647059, -0.192156862745098, -0.066666666666667, -0.050980392156863,
-0.219607843137255, -0.011764705882353, +0.050980392156863, -0.203921568627451,
-0.082352941176471, +0.137254901960784, -0.015686274509804, -0.129411764705882,
-0.094117647058824, -0.129411764705882, -0.031372549019608, +0.090196078431373,
-0.058823529411765, -0.239215686274510, -0.043137254901961, +0.023529411764706,
-0.086274509803922, -0.011764705882353, +0.125490196078431, +0.000000000000000,
-0.168627450980392, +0.039215686274510, +0.105882352941176, -0.133333333333333,
-0.101960784313725, -0.011764705882353, -0.007843137254902, +0.054901960784314,
+0.031372549019608, -0.149019607843137, -0.105882352941176, +0.074509803921569,
-0.078431372549020, -0.172549019607843, -0.035294117647059, -0.003921568627451,
-0.156862745098039, -0.039215686274510, +0.031372549019608, -0.098039215686275,
+0.011764705882353, +0.047058823529412, -0.196078431372549, -0.200000000000000,
+0.011764705882353, -0.031372549019608, -0.003921568627451, +0.129411764705882,
+0.019607843137255, -0.035294117647059, -0.019607843137255, +0.000000000000000,
-0.125490196078431, -0.105882352941176, +0.113725490196078, -0.015686274509804,
-0.066666666666667, +0.011764705882353, -0.031372549019608, -0.066666666666667,
+0.050980392156863, +0.019607843137255, -0.211764705882353, -0.184313725490196,
+0.031372549019608, -0.043137254901961, -0.105882352941176, +0.066666666666667,
+0.227450980392157, -0.023529411764706, -0.180392156862745, -0.035294117647059,
-0.094117647058824, -0.113725490196078, +0.050980392156863, +0.031372549019608,
-0.094117647058824, +0.000000000000000, +0.070588235294118, -0.094117647058824,
-0.113725490196078, +0.101960784313725, +0.101960784313725, -0.125490196078431,
-0.082352941176471, +0.000000000000000, -0.086274509803922, +0.062745098039216,
+0.117647058823529, +0.039215686274510, +0.156862745098039, +0.172549019607843,
+0.082352941176471, +0.105882352941176, +0.207843137254902, -0.023529411764706,
-0.086274509803922, +0.168627450980392, +0.047058823529412, +0.058823529411765,
+0.211764705882353, +0.113725490196078, -0.137254901960784, -0.050980392156863,
+0.215686274509804, +0.090196078431373, +0.054901960784314, +0.203921568627451,
+0.117647058823529, -0.011764705882353, +0.125490196078431, +0.094117647058824,
-0.019607843137255, +0.101960784313725, +0.054901960784314, -0.011764705882353,
+0.141176470588235, +0.188235294117647, -0.039215686274510, +0.043137254901961,
+0.278431372549020, +0.121568627450980, -0.023529411764706, +0.176470588235294,
+0.211764705882353, +0.043137254901961, +0.082352941176471, +0.113725490196078,
-0.050980392156863, +0.090196078431373, +0.152941176470588, -0.192156862745098,
-0.094117647058824, +0.098039215686275, -0.105882352941176, -0.074509803921569,
+0.031372549019608, -0.082352941176471, -0.062745098039216, +0.078431372549020,
+0.043137254901961, -0.168627450980392, -0.047058823529412, -0.050980392156863,
-0.231372549019608, -0.066666666666667, +0.039215686274510, -0.105882352941176,
-0.207843137254902, -0.054901960784314, +0.000000000000000, -0.203921568627451,
-0.098039215686275, -0.050980392156863, -0.074509803921569, +0.082352941176471,
+0.000000000000000, -0.152941176470588, -0.039215686274510, +0.066666666666667,
-0.074509803921569, -0.176470588235294, -0.062745098039216, -0.019607843137255,
-0.117647058823529, +0.015686274509804, -0.011764705882353, -0.105882352941176,
+0.125490196078431, +0.176470588235294, -0.125490196078431, -0.133333333333333,
+0.035294117647059, -0.105882352941176, -0.152941176470588, +0.043137254901961,
-0.070588235294118, -0.105882352941176, +0.086274509803922, +0.074509803921569,
-0.074509803921569, -0.141176470588235, +0.007843137254902, -0.074509803921569,
-0.074509803921569, -0.054901960784314, -0.164705882352941, -0.125490196078431,
+0.082352941176471, +0.129411764705882, -0.125490196078431, -0.203921568627451,
+0.062745098039216, +0.054901960784314, -0.200000000000000, -0.090196078431373,
+0.098039215686275, -0.058823529411765, -0.223529411764706, -0.188235294117647,
-0.031372549019608, -0.149019607843137, +0.035294117647059, +0.086274509803922,
-0.188235294117647, -0.074509803921569, +0.094117647058824, -0.090196078431373,
-0.262745098039216, -0.070588235294118, +0.050980392156863, -0.105882352941176,
-0.098039215686275, -0.113725490196078, -0.211764705882353, -0.105882352941176,
-0.062745098039216, -0.203921568627451, -0.039215686274510, +0.047058823529412,
-0.121568627450980, -0.105882352941176, +0.145098039215686, +0.003921568627451,
-0.207843137254902, -0.078431372549020, -0.160784313725490, -0.145098039215686,
+0.070588235294118, +0.074509803921569, -0.192156862745098, -0.160784313725490,
+0.003921568627451, -0.094117647058824, -0.129411764705882, +0.094117647058824,
+0.062745098039216, -0.196078431372549, -0.086274509803922, -0.054901960784314,
-0.149019607843137, +0.047058823529412, +0.098039215686275, -0.082352941176471,
+0.019607843137255, +0.086274509803922, -0.105882352941176, -0.121568627450980,
+0.050980392156863, -0.035294117647059, -0.125490196078431, +0.003921568627451,
+0.019607843137255, -0.066666666666667, -0.070588235294118, -0.039215686274510,
-0.047058823529412, +0.105882352941176, +0.160784313725490, -0.149019607843137,
-0.070588235294118, +0.156862745098039, -0.074509803921569, +0.027450980392157,
+0.211764705882353, +0.113725490196078, +0.086274509803922, +0.200000000000000,
+0.145098039215686, -0.105882352941176, +0.058823529411765, +0.094117647058824,
-0.188235294117647, +0.007843137254902, +0.172549019607843, -0.031372549019608,
-0.039215686274510, +0.129411764705882, +0.196078431372549, -0.003921568627451,
+0.000000000000000, +0.031372549019608, -0.007843137254902, +0.192156862745098,
+0.125490196078431, -0.035294117647059, +0.082352941176471, +0.211764705882353,
+0.117647058823529, +0.035294117647059, +0.149019607843137, +0.078431372549020,
-0.003921568627451, +0.180392156862745, +0.086274509803922, -0.043137254901961,
+0.156862745098039, +0.168627450980392, -0.156862745098039, -0.094117647058824,
+0.207843137254902, +0.086274509803922, -0.039215686274510, +0.149019607843137,
+0.074509803921569, -0.070588235294118, +0.031372549019608, +0.082352941176471,
+0.003921568627451, -0.019607843137255, +0.113725490196078, -0.007843137254902,
+0.058823529411765, +0.152941176470588, -0.011764705882353, -0.062745098039216,
+0.117647058823529, +0.239215686274510, +0.039215686274510, -0.098039215686275,
+0.098039215686275, +0.200000000000000, -0.070588235294118, -0.011764705882353,
+0.160784313725490, +0.050980392156863, +0.058823529411765, +0.066666666666667,
+0.125490196078431, -0.023529411764706, +0.094117647058824, +0.219607843137255,
+0.043137254901961, +0.031372549019608, +0.254901960784314, +0.113725490196078,
-0.113725490196078, +0.074509803921569, +0.149019607843137, +0.066666666666667,
+0.160784313725490, +0.168627450980392, +0.054901960784314, +0.078431372549020,
+0.094117647058824, -0.152941176470588, -0.023529411764706, +0.192156862745098,
+0.047058823529412, -0.098039215686275, +0.094117647058824, +0.101960784313725,
-0.011764705882353, +0.050980392156863, -0.070588235294118, -0.078431372549020,
+0.094117647058824, +0.137254901960784, -0.098039215686275, -0.082352941176471,
+0.011764705882353, -0.050980392156863, -0.113725490196078, +0.078431372549020,
+0.156862745098039, -0.098039215686275, -0.054901960784314, +0.047058823529412,
-0.050980392156863, +0.011764705882353, +0.105882352941176, -0.133333333333333,
-0.094117647058824, +0.054901960784314, -0.101960784313725, -0.109803921568627,
-0.023529411764706, -0.137254901960784, -0.168627450980392, -0.054901960784314,
-0.074509803921569, -0.098039215686275, -0.054901960784314, -0.160784313725490,
-0.196078431372549, -0.011764705882353, +0.047058823529412, -0.152941176470588,
-0.117647058823529, +0.105882352941176, -0.129411764705882, -0.125490196078431,
+0.019607843137255, -0.125490196078431, -0.176470588235294, +0.043137254901961,
+0.066666666666667, -0.211764705882353, -0.082352941176471, +0.007843137254902,
-0.231372549019608, -0.066666666666667, +0.176470588235294, -0.082352941176471,
-0.145098039215686, +0.098039215686275, +0.125490196078431, -0.086274509803922,
-0.098039215686275, -0.062745098039216, -0.152941176470588, +0.039215686274510,
+0.031372549019608, -0.137254901960784, -0.011764705882353, +0.094117647058824,
-0.062745098039216, -0.066666666666667, +0.117647058823529, -0.058823529411765,
-0.188235294117647, +0.035294117647059, -0.011764705882353, -0.125490196078431,
+0.125490196078431, +0.156862745098039, -0.141176470588235, -0.105882352941176,
+0.098039215686275, -0.050980392156863, -0.105882352941176, +0.062745098039216,
+0.003921568627451, -0.023529411764706, -0.007843137254902, -0.098039215686275,
-0.200000000000000, -0.074509803921569, +0.149019607843137, -0.054901960784314,
-0.043137254901961, +0.172549019607843, +0.027450980392157, -0.145098039215686,
+0.027450980392157, +0.160784313725490, -0.058823529411765, -0.137254901960784,
+0.050980392156863, +0.129411764705882, -0.113725490196078, -0.050980392156863,
+0.109803921568627, -0.047058823529412, +0.086274509803922, +0.117647058823529,
+0.015686274509804, -0.047058823529412, -0.039215686274510, -0.078431372549020,
-0.098039215686275, -0.074509803921569, +0.172549019607843, +0.043137254901961,
-0.180392156862745, +0.082352941176471, +0.098039215686275, -0.035294117647059,
+0.078431372549020, +0.109803921568627, +0.050980392156863, +0.160784313725490,
+0.227450980392157, +0.000000000000000, +0.019607843137255, +0.137254901960784,
+0.066666666666667, -0.050980392156863, +0.043137254901961, +0.109803921568627,
+0.133333333333333, +0.200000000000000, +0.015686274509804, -0.019607843137255,
+0.168627450980392, +0.188235294117647, +0.043137254901961, +0.109803921568627,
+0.156862745098039, +0.023529411764706, -0.027450980392157, +0.086274509803922,
+0.200000000000000, +0.156862745098039, +0.129411764705882, +0.137254901960784,
+0.094117647058824, +0.105882352941176, +0.215686274509804, +0.035294117647059,
+0.007843137254902, +0.192156862745098, +0.090196078431373, +0.117647058823529,
+0.219607843137255, +0.141176470588235, +0.105882352941176, +0.188235294117647,
+0.121568627450980, +0.043137254901961, +0.192156862745098, +0.078431372549020,
-0.019607843137255, +0.133333333333333, +0.176470588235294, +0.054901960784314,
+0.007843137254902, +0.176470588235294, -0.023529411764706, -0.027450980392157,
+0.160784313725490, +0.000000000000000, -0.176470588235294, +0.031372549019608,
+0.149019607843137, -0.082352941176471, +0.039215686274510, +0.180392156862745,
+0.019607843137255, +0.007843137254902, +0.180392156862745, -0.003921568627451,
-0.200000000000000, +0.035294117647059, +0.117647058823529, -0.035294117647059,
+0.003921568627451, +0.129411764705882, -0.031372549019608, -0.003921568627451,
+0.000000000000000, -0.219607843137255, -0.074509803921569, +0.145098039215686,
-0.019607843137255, -0.058823529411765, +0.145098039215686, -0.015686274509804,
-0.160784313725490, +0.043137254901961, +0.011764705882353, -0.129411764705882,
+0.137254901960784, +0.207843137254902, -0.047058823529412, -0.031372549019608,
+0.109803921568627, -0.113725490196078, -0.188235294117647, +0.054901960784314,
+0.011764705882353, +0.050980392156863, +0.215686274509804, +0.133333333333333,
-0.149019607843137, -0.129411764705882, +0.129411764705882, -0.047058823529412,
-0.113725490196078, +0.113725490196078, +0.078431372549020, -0.121568627450980,
+0.027450980392157, +0.231372549019608, +0.019607843137255, -0.137254901960784,
+0.031372549019608, +0.125490196078431, -0.098039215686275, +0.007843137254902,
+0.184313725490196, -0.031372549019608, +0.031372549019608, +0.015686274509804,
-0.043137254901961, -0.031372549019608, +0.054901960784314, -0.137254901960784,
-0.219607843137255, -0.145098039215686, +0.121568627450980, -0.003921568627451,
-0.184313725490196, +0.003921568627451, -0.031372549019608, -0.219607843137255,
-0.062745098039216, -0.019607843137255, -0.227450980392157, -0.062745098039216,
+0.050980392156863, -0.133333333333333, -0.062745098039216, -0.023529411764706,
-0.137254901960784, -0.200000000000000, -0.184313725490196, -0.156862745098039,
-0.113725490196078, +0.047058823529412, -0.086274509803922, -0.239215686274510,
-0.066666666666667, -0.054901960784314, -0.137254901960784, -0.031372549019608,
+0.117647058823529, -0.090196078431373, -0.152941176470588, -0.050980392156863,
-0.113725490196078, -0.062745098039216, -0.043137254901961, -0.133333333333333,
-0.180392156862745, -0.129411764705882, +0.011764705882353, -0.003921568627451,
+0.000000000000000, +0.031372549019608, -0.137254901960784, -0.117647058823529,
-0.003921568627451, +0.027450980392157, +0.043137254901961, +0.105882352941176,
-0.011764705882353, -0.105882352941176, +0.109803921568627, +0.047058823529412,
-0.031372549019608, +0.137254901960784, +0.149019607843137, +0.074509803921569,
+0.074509803921569, +0.219607843137255, +0.109803921568627, +0.054901960784314,
+0.243137254901961, +0.207843137254902, +0.019607843137255, +0.117647058823529,
+0.250980392156863, +0.074509803921569, +0.070588235294118, +0.086274509803922,
+0.027450980392157, +0.090196078431373, +0.239215686274510, +0.117647058823529,
-0.149019607843137, +0.027450980392157, +0.152941176470588, -0.019607843137255,
-0.098039215686275, +0.031372549019608, +0.082352941176471, +0.152941176470588,
+0.145098039215686, -0.168627450980392, -0.113725490196078, +0.168627450980392,
+0.043137254901961, -0.019607843137255, +0.200000000000000, +0.027450980392157,
-0.188235294117647, +0.047058823529412, +0.117647058823529, -0.023529411764706,
+0.101960784313725, +0.058823529411765, -0.180392156862745, -0.074509803921569,
+0.243137254901961, +0.113725490196078, -0.094117647058824, +0.086274509803922,
+0.027450980392157, -0.019607843137255, +0.078431372549020, +0.082352941176471,
-0.074509803921569, -0.098039215686275, +0.070588235294118, -0.078431372549020,
-0.094117647058824, +0.121568627450980, +0.121568627450980, -0.043137254901961,
+0.019607843137255, +0.141176470588235, +0.007843137254902, -0.156862745098039,
+0.050980392156863, +0.192156862745098, -0.027450980392157, -0.054901960784314,
+0.011764705882353, -0.019607843137255, +0.086274509803922, +0.019607843137255,
+0.031372549019608, +0.023529411764706, +0.250980392156863, +0.188235294117647,
+0.019607843137255, +0.000000000000000, +0.176470588235294, -0.019607843137255,
-0.149019607843137, +0.086274509803922, +0.105882352941176, -0.082352941176471,
+0.043137254901961, +0.207843137254902, -0.058823529411765, -0.019607843137255,
+0.043137254901961, -0.223529411764706, -0.050980392156863, +0.172549019607843,
+0.050980392156863, -0.003921568627451, +0.007843137254902, -0.050980392156863,
-0.145098039215686, +0.050980392156863, +0.105882352941176, -0.156862745098039,
-0.137254901960784, -0.066666666666667, -0.145098039215686, -0.164705882352941,
+0.105882352941176, +0.054901960784314, -0.086274509803922, -0.023529411764706,
-0.164705882352941, -0.156862745098039, -0.035294117647059, -0.050980392156863,
-0.137254901960784, -0.145098039215686, -0.145098039215686, -0.219607843137255,
-0.047058823529412, +0.043137254901961, -0.172549019607843, -0.188235294117647,
-0.156862745098039, -0.188235294117647, -0.141176470588235, -0.023529411764706,
-0.145098039215686, -0.211764705882353, +0.011764705882353, -0.062745098039216,
-0.227450980392157, -0.043137254901961, -0.062745098039216, -0.211764705882353,
-0.141176470588235, -0.070588235294118, -0.117647058823529, -0.176470588235294,
-0.074509803921569, -0.043137254901961, -0.058823529411765, -0.019607843137255,
-0.027450980392157, -0.113725490196078, -0.027450980392157, -0.082352941176471,
-0.239215686274510, -0.082352941176471, +0.062745098039216, +0.019607843137255,
-0.192156862745098, -0.121568627450980, +0.031372549019608, -0.086274509803922,
-0.231372549019608, -0.254901960784314, -0.160784313725490, +0.078431372549020,
+0.129411764705882, -0.200000000000000, -0.192156862745098, +0.039215686274510,
-0.117647058823529, -0.180392156862745, +0.058823529411765, -0.031372549019608,
-0.223529411764706, -0.094117647058824, -0.090196078431373, -0.192156862745098,
+0.031372549019608, +0.054901960784314, -0.231372549019608, -0.176470588235294,
+0.062745098039216, -0.011764705882353, -0.180392156862745, -0.105882352941176,
-0.176470588235294, -0.047058823529412, +0.094117647058824, -0.027450980392157,
-0.149019607843137, -0.105882352941176, +0.050980392156863, -0.152941176470588,
-0.247058823529412, -0.074509803921569, -0.098039215686275, -0.188235294117647,
-0.074509803921569, +0.043137254901961, -0.066666666666667, -0.219607843137255,
-0.031372549019608, +0.031372549019608, -0.105882352941176, -0.125490196078431,
-0.243137254901961, -0.227450980392157, +0.003921568627451, -0.094117647058824,
-0.050980392156863, -0.098039215686275, +0.105882352941176, +0.149019607843137,
-0.050980392156863, -0.070588235294118, +0.168627450980392, -0.015686274509804,
-0.250980392156863, -0.027450980392157, +0.121568627450980, +0.023529411764706,
+0.019607843137255, +0.129411764705882, +0.015686274509804, +0.082352941176471,
+0.211764705882353, -0.086274509803922, -0.066666666666667, +0.176470588235294,
+0.090196078431373, +0.050980392156863, +0.180392156862745, +0.168627450980392,
+0.015686274509804, +0.101960784313725, +0.231372549019608, +0.133333333333333,
+0.113725490196078, +0.109803921568627, +0.031372549019608, -0.031372549019608,
+0.105882352941176, +0.188235294117647, +0.137254901960784, +0.172549019607843,
+0.145098039215686, +0.105882352941176, +0.168627450980392, +0.227450980392157,
+0.137254901960784, +0.082352941176471, +0.074509803921569, -0.121568627450980,
+0.011764705882353, +0.266666666666667, +0.101960784313725, +0.086274509803922,
+0.145098039215686, -0.015686274509804, -0.007843137254902, +0.180392156862745,
-0.007843137254902, -0.125490196078431, +0.105882352941176, +0.031372549019608,
-0.207843137254902, +0.019607843137255, +0.109803921568627, -0.149019607843137,
-0.156862745098039, -0.082352941176471, -0.121568627450980, -0.101960784313725,
-0.011764705882353, -0.117647058823529, -0.062745098039216, +0.019607843137255,
-0.094117647058824, -0.160784313725490, +0.058823529411765, +0.113725490196078,
-0.141176470588235, -0.039215686274510, +0.019607843137255, -0.019607843137255,
-0.058823529411765, +0.019607843137255, +0.172549019607843, +0.070588235294118,
+0.019607843137255, -0.050980392156863, -0.239215686274510, +0.011764705882353,
+0.184313725490196, -0.156862745098039, -0.125490196078431, +0.105882352941176,
-0.094117647058824, -0.129411764705882, +0.137254901960784, +0.062745098039216,
-0.105882352941176, +0.047058823529412, +0.054901960784314, -0.101960784313725,
+0.105882352941176, +0.172549019607843, -0.105882352941176, -0.031372549019608,
+0.105882352941176, -0.031372549019608, -0.066666666666667, +0.074509803921569,
-0.039215686274510, +0.035294117647059, +0.254901960784314, +0.168627450980392,
+0.011764705882353, +0.000000000000000, +0.156862745098039, -0.019607843137255,
-0.156862745098039, +0.007843137254902, -0.054901960784314, -0.125490196078431,
+0.007843137254902, +0.184313725490196, +0.070588235294118, -0.160784313725490,
+0.027450980392157, +0.090196078431373, +0.000000000000000, +0.117647058823529,
+0.000000000000000, -0.125490196078431, +0.101960784313725, +0.031372549019608,
-0.050980392156863, -0.164705882352941, +0.000000000000000, +0.125490196078431,
+0.011764705882353, -0.039215686274510, +0.098039215686275, +0.050980392156863,
-0.074509803921569, -0.031372549019608, -0.082352941176471, -0.137254901960784,
-0.023529411764706, +0.011764705882353, -0.141176470588235, -0.019607843137255,
+0.160784313725490, -0.074509803921569, -0.094117647058824, +0.094117647058824,
+0.000000000000000, -0.184313725490196, -0.031372549019608, -0.007843137254902,
-0.145098039215686, +0.000000000000000, +0.019607843137255, -0.007843137254902,
+0.149019607843137, +0.101960784313725, -0.109803921568627, -0.125490196078431,
-0.023529411764706, +0.007843137254902, +0.050980392156863, +0.035294117647059,
-0.039215686274510, -0.031372549019608, +0.050980392156863, +0.058823529411765,
-0.019607843137255, -0.019607843137255, +0.113725490196078, -0.050980392156863,
-0.086274509803922, +0.141176470588235, -0.011764705882353, -0.011764705882353,
+0.254901960784314, +0.074509803921569, -0.082352941176471, +0.113725490196078,
-0.031372549019608, -0.184313725490196, +0.082352941176471, +0.125490196078431,
-0.125490196078431, +0.000000000000000, +0.168627450980392, +0.074509803921569,
+0.078431372549020, +0.152941176470588, +0.121568627450980, +0.125490196078431,
+0.207843137254902, +0.043137254901961, +0.050980392156863, +0.219607843137255,
+0.070588235294118, -0.137254901960784, -0.007843137254902, +0.043137254901961,
-0.156862745098039, +0.050980392156863, +0.192156862745098, +0.050980392156863,
-0.070588235294118, +0.039215686274510, +0.258823529411765, +0.023529411764706,
+0.000000000000000, +0.180392156862745, -0.117647058823529, -0.019607843137255,
+0.235294117647059, -0.090196078431373, -0.054901960784314, +0.223529411764706,
-0.011764705882353, -0.105882352941176, +0.160784313725490, +0.043137254901961,
-0.188235294117647, +0.015686274509804, +0.074509803921569, -0.043137254901961,
+0.156862745098039, +0.156862745098039, -0.086274509803922, +0.050980392156863,
+0.231372549019608, +0.039215686274510, -0.031372549019608, +0.160784313725490,
+0.062745098039216, +0.039215686274510, +0.188235294117647, +0.129411764705882,
+0.019607843137255, +0.047058823529412, +0.200000000000000, +0.062745098039216,
-0.050980392156863, +0.168627450980392, +0.043137254901961, -0.105882352941176,
+0.027450980392157, +0.188235294117647, +0.066666666666667, -0.137254901960784,
+0.094117647058824, +0.200000000000000, +0.011764705882353, +0.098039215686275,
+0.145098039215686, -0.027450980392157, +0.074509803921569, +0.086274509803922,
+0.145098039215686, -0.015686274509804, +0.047058823529412, +0.125490196078431,
+0.082352941176471, +0.133333333333333, +0.152941176470588, -0.031372549019608,
-0.007843137254902, +0.184313725490196, +0.070588235294118, -0.141176470588235,
+0.011764705882353, +0.188235294117647, -0.082352941176471, -0.082352941176471,
+0.160784313725490, +0.003921568627451, -0.023529411764706, +0.152941176470588,
+0.145098039215686, -0.043137254901961, +0.074509803921569, +0.031372549019608,
-0.152941176470588, +0.082352941176471, +0.050980392156863, -0.105882352941176,
+0.011764705882353, +0.137254901960784, -0.078431372549020, -0.176470588235294,
+0.031372549019608, +0.011764705882353, +0.031372549019608, +0.047058823529412,
-0.164705882352941, -0.262745098039216, -0.019607843137255, +0.086274509803922,
+0.000000000000000, +0.011764705882353, +0.137254901960784, -0.011764705882353,
-0.141176470588235, +0.062745098039216, -0.066666666666667, -0.145098039215686,
+0.188235294117647, +0.074509803921569, -0.105882352941176, +0.066666666666667,
+0.011764705882353, -0.137254901960784, +0.019607843137255, +0.094117647058824,
-0.098039215686275, -0.007843137254902, +0.023529411764706, -0.043137254901961,
+0.086274509803922, +0.062745098039216, -0.105882352941176, -0.062745098039216,
+0.062745098039216, -0.062745098039216, -0.133333333333333, +0.133333333333333,
+0.094117647058824, -0.192156862745098, -0.094117647058824, -0.011764705882353,
-0.192156862745098, -0.082352941176471, +0.082352941176471, -0.023529411764706,
-0.211764705882353, -0.117647058823529, +0.129411764705882, -0.082352941176471,
-0.117647058823529, +0.180392156862745, -0.007843137254902, -0.113725490196078,
+0.074509803921569, -0.129411764705882, -0.105882352941176, +0.223529411764706,
+0.011764705882353, -0.149019607843137, +0.078431372549020, +0.015686274509804,
-0.133333333333333, +0.015686274509804, -0.062745098039216, -0.196078431372549,
+0.035294117647059, +0.000000000000000, -0.176470588235294, -0.011764705882353,
+0.184313725490196, -0.043137254901961, -0.223529411764706, -0.031372549019608,
-0.094117647058824, -0.117647058823529, +0.094117647058824, +0.015686274509804,
-0.180392156862745, -0.058823529411765, +0.168627450980392, +0.007843137254902,
-0.188235294117647, +0.039215686274510, +0.043137254901961, -0.219607843137255,
-0.125490196078431, +0.141176470588235, +0.035294117647059, -0.125490196078431,
+0.050980392156863, +0.113725490196078, -0.152941176470588, -0.137254901960784,
+0.086274509803922, -0.078431372549020, -0.156862745098039, -0.070588235294118,
-0.047058823529412, -0.109803921568627, -0.062745098039216, -0.105882352941176,
-0.219607843137255, +0.023529411764706, +0.196078431372549, -0.117647058823529,
-0.168627450980392, +0.058823529411765, +0.035294117647059, -0.168627450980392,
-0.086274509803922, +0.172549019607843, -0.023529411764706, -0.188235294117647,
-0.007843137254902, -0.054901960784314, -0.098039215686275, +0.019607843137255,
-0.019607843137255, -0.094117647058824, +0.062745098039216, -0.003921568627451,
-0.156862745098039, +0.070588235294118, +0.039215686274510, -0.031372549019608,
+0.019607843137255, +0.137254901960784, +0.098039215686275, -0.047058823529412,
+0.074509803921569, -0.007843137254902, +0.050980392156863, +0.270588235294118,
+0.184313725490196, -0.145098039215686, -0.066666666666667, +0.129411764705882,
-0.011764705882353, +0.000000000000000, +0.200000000000000, +0.129411764705882,
+0.003921568627451, +0.113725490196078, -0.023529411764706, -0.180392156862745,
+0.105882352941176, +0.160784313725490, +0.082352941176471, +0.188235294117647,
+0.113725490196078, +0.019607843137255, +0.121568627450980, +0.168627450980392,
+0.031372549019608, +0.137254901960784, +0.086274509803922, -0.129411764705882,
+0.043137254901961, +0.156862745098039, -0.133333333333333, -0.168627450980392,
+0.070588235294118, -0.011764705882353, -0.168627450980392, +0.094117647058824,
+0.196078431372549, -0.023529411764706, +0.058823529411765, +0.180392156862745,
+0.031372549019608, +0.000000000000000, +0.129411764705882, +0.074509803921569,
-0.152941176470588, -0.031372549019608, +0.180392156862745, +0.019607843137255,
-0.027450980392157, +0.164705882352941, +0.149019607843137, +0.066666666666667,
+0.117647058823529, -0.105882352941176, -0.156862745098039, +0.192156862745098,
+0.094117647058824, -0.019607843137255, +0.188235294117647, +0.019607843137255,
-0.062745098039216, +0.200000000000000, +0.137254901960784, -0.019607843137255,
+0.125490196078431, +0.050980392156863, -0.215686274509804, -0.074509803921569,
+0.223529411764706, +0.101960784313725, -0.070588235294118, +0.078431372549020,
-0.039215686274510, -0.149019607843137, +0.105882352941176, +0.113725490196078,
-0.168627450980392, -0.062745098039216, +0.203921568627451, +0.137254901960784,
-0.011764705882353, +0.058823529411765, +0.180392156862745, -0.007843137254902,
-0.058823529411765, +0.180392156862745, +0.078431372549020, -0.125490196078431,
+0.039215686274510, +0.211764705882353, -0.015686274509804, -0.117647058823529,
+0.125490196078431, +0.070588235294118, -0.070588235294118, +0.007843137254902,
-0.015686274509804, -0.035294117647059, +0.062745098039216, +0.047058823529412,
-0.141176470588235, -0.007843137254902, +0.141176470588235, -0.109803921568627,
-0.094117647058824, +0.082352941176471, +0.000000000000000, -0.133333333333333,
-0.043137254901961, +0.125490196078431, +0.039215686274510, -0.113725490196078,
-0.145098039215686, -0.200000000000000, -0.133333333333333, +0.019607843137255,
-0.145098039215686, -0.223529411764706, -0.062745098039216, -0.133333333333333,
-0.274509803921569, -0.035294117647059, -0.062745098039216, -0.176470588235294,
-0.031372549019608, -0.039215686274510, -0.113725490196078, -0.160784313725490,
-0.074509803921569, -0.196078431372549, -0.160784313725490, +0.050980392156863,
+0.086274509803922, -0.086274509803922, -0.164705882352941, -0.011764705882353,
-0.172549019607843, -0.160784313725490, +0.019607843137255, -0.058823529411765,
-0.031372549019608, +0.031372549019608, -0.113725490196078, -0.274509803921569,
-0.090196078431373, +0.011764705882353, -0.035294117647059, +0.074509803921569,
-0.062745098039216, -0.137254901960784, +0.031372549019608, +0.062745098039216,
-0.043137254901961, +0.074509803921569, +0.074509803921569, -0.149019607843137,
-0.094117647058824, +0.160784313725490, +0.062745098039216, -0.164705882352941,
+0.054901960784314, +0.086274509803922, -0.086274509803922, +0.094117647058824,
+0.176470588235294, +0.031372549019608, +0.070588235294118, +0.121568627450980,
-0.039215686274510, -0.027450980392157, +0.211764705882353, +0.192156862745098,
-0.062745098039216, +0.023529411764706, +0.156862745098039, -0.047058823529412,
-0.125490196078431, +0.003921568627451, +0.011764705882353, +0.074509803921569,
+0.200000000000000, -0.066666666666667, -0.188235294117647, +0.105882352941176,
+0.090196078431373, -0.050980392156863, +0.168627450980392, -0.011764705882353,
-0.200000000000000, +0.070588235294118, +0.098039215686275, -0.050980392156863,
+0.062745098039216, +0.149019607843137, -0.149019607843137, -0.125490196078431,
+0.094117647058824, +0.003921568627451, -0.050980392156863, +0.105882352941176,
-0.019607843137255, -0.156862745098039, +0.062745098039216, +0.109803921568627,
-0.152941176470588, -0.149019607843137, +0.035294117647059, -0.027450980392157,
-0.015686274509804, -0.011764705882353, +0.007843137254902, -0.007843137254902,
-0.011764705882353, +0.078431372549020, -0.031372549019608, -0.211764705882353,
-0.109803921568627, +0.098039215686275, -0.039215686274510, -0.101960784313725,
+0.070588235294118, +0.007843137254902, -0.031372549019608, +0.023529411764706,
+0.035294117647059, +0.019607843137255, +0.149019607843137, +0.188235294117647,
+0.011764705882353, +0.086274509803922, +0.180392156862745, -0.007843137254902,
+0.031372549019608, +0.219607843137255, +0.101960784313725, -0.074509803921569,
+0.062745098039216, +0.184313725490196, +0.137254901960784, +0.168627450980392,
+0.121568627450980, -0.031372549019608, +0.023529411764706, +0.231372549019608,
+0.168627450980392, +0.047058823529412, +0.133333333333333, +0.105882352941176,
-0.090196078431373, +0.066666666666667, +0.117647058823529, -0.113725490196078,
+0.000000000000000, +0.050980392156863, -0.062745098039216, -0.078431372549020,
+0.054901960784314, +0.000000000000000, -0.027450980392157, +0.070588235294118,
+0.000000000000000, -0.054901960784314, -0.011764705882353, +0.133333333333333,
+0.000000000000000, -0.015686274509804, +0.074509803921569, -0.117647058823529,
-0.003921568627451, +0.207843137254902, +0.105882352941176, -0.113725490196078,
-0.039215686274510, -0.054901960784314, -0.149019607843137, +0.105882352941176,
-0.023529411764706, -0.227450980392157, -0.043137254901961, -0.039215686274510,
-0.223529411764706, -0.035294117647059, +0.066666666666667, -0.145098039215686,
-0.219607843137255, -0.066666666666667, +0.019607843137255, -0.176470588235294,
-0.086274509803922, -0.043137254901961, -0.203921568627451, -0.105882352941176,
-0.062745098039216, -0.180392156862745, -0.145098039215686, -0.058823529411765,
-0.211764705882353, -0.184313725490196, +0.000000000000000, -0.031372549019608,
-0.094117647058824, +0.007843137254902, +0.078431372549020, -0.125490196078431,
-0.200000000000000, +0.011764705882353, -0.125490196078431, -0.149019607843137,
+0.098039215686275, -0.090196078431373, -0.207843137254902, +0.000000000000000,
-0.011764705882353, -0.184313725490196, +0.043137254901961, +0.007843137254902,
-0.270588235294118, -0.082352941176471, +0.015686274509804, -0.168627450980392,
-0.058823529411765, +0.137254901960784, -0.121568627450980, -0.078431372549020,
+0.031372549019608, -0.129411764705882, -0.168627450980392, +0.035294117647059,
-0.019607843137255, -0.129411764705882, +0.031372549019608, +0.031372549019608,
-0.137254901960784, -0.176470588235294, -0.066666666666667, -0.180392156862745,
-0.094117647058824, +0.039215686274510, -0.086274509803922, -0.164705882352941,
-0.027450980392157, +0.101960784313725, -0.011764705882353, -0.133333333333333,
-0.070588235294118, +0.011764705882353, -0.176470588235294, -0.125490196078431,
+0.031372549019608, -0.176470588235294, -0.192156862745098, -0.137254901960784,
-0.149019607843137, -0.180392156862745, -0.058823529411765, +0.011764705882353,
-0.133333333333333, +0.019607843137255, +0.062745098039216, -0.125490196078431,
-0.125490196078431, +0.058823529411765, -0.007843137254902, -0.203921568627451,
-0.054901960784314, +0.007843137254902, -0.105882352941176, +0.062745098039216,
+0.094117647058824, -0.082352941176471, -0.031372549019608, +0.098039215686275,
+0.113725490196078, +0.050980392156863, +0.133333333333333, +0.117647058823529,
-0.011764705882353, +0.090196078431373, +0.133333333333333, +0.000000000000000,
+0.070588235294118, +0.192156862745098, +0.113725490196078, +0.062745098039216,
+0.117647058823529, +0.050980392156863, +0.082352941176471, +0.258823529411765,
+0.125490196078431, -0.019607843137255, +0.074509803921569, +0.117647058823529,
+0.023529411764706, +0.078431372549020, +0.227450980392157, +0.015686274509804,
-0.023529411764706, +0.200000000000000, +0.125490196078431, -0.050980392156863,
+0.137254901960784, +0.082352941176471, -0.133333333333333, +0.152941176470588,
+0.180392156862745, -0.047058823529412, +0.082352941176471, +0.172549019607843,
-0.062745098039216, +0.039215686274510, +0.227450980392157, +0.050980392156863,
-0.007843137254902, +0.121568627450980, +0.168627450980392, +0.007843137254902,
+0.043137254901961, +0.141176470588235, +0.058823529411765, +0.039215686274510,
+0.011764705882353, -0.023529411764706, +0.054901960784314, +0.152941176470588,
-0.101960784313725, -0.082352941176471, +0.129411764705882, -0.082352941176471,
-0.188235294117647, +0.007843137254902, +0.231372549019608, +0.098039215686275,
-0.031372549019608, +0.156862745098039, +0.062745098039216, -0.039215686274510,
+0.164705882352941, -0.015686274509804, -0.160784313725490, +0.094117647058824,
+0.043137254901961, -0.105882352941176, +0.125490196078431, +0.113725490196078,
-0.129411764705882, +0.043137254901961, +0.149019607843137, -0.086274509803922,
+0.011764705882353, +0.129411764705882, -0.133333333333333, -0.007843137254902,
+0.200000000000000, +0.074509803921569, -0.062745098039216, +0.090196078431373,
+0.101960784313725, +0.011764705882353, +0.172549019607843, +0.176470588235294,
+0.086274509803922, +0.094117647058824, +0.152941176470588, +0.039215686274510,
-0.062745098039216, +0.105882352941176, +0.145098039215686, -0.074509803921569,
+0.000000000000000, +0.196078431372549, +0.105882352941176, +0.011764705882353,
+0.117647058823529, +0.203921568627451, -0.062745098039216, -0.031372549019608,
+0.219607843137255, +0.074509803921569, +0.031372549019608, +0.050980392156863,
+0.070588235294118, -0.031372549019608, -0.003921568627451, +0.043137254901961,
-0.196078431372549, -0.035294117647059, +0.094117647058824, -0.105882352941176,
-0.121568627450980, +0.000000000000000, -0.050980392156863, -0.227450980392157,
-0.125490196078431, -0.054901960784314, -0.239215686274510, -0.086274509803922,
-0.007843137254902, -0.231372549019608, -0.156862745098039, -0.023529411764706,
-0.094117647058824, -0.219607843137255, -0.066666666666667, -0.082352941176471,
-0.156862745098039, -0.023529411764706, -0.058823529411765, -0.160784313725490,
-0.121568627450980, +0.058823529411765, -0.074509803921569, -0.117647058823529,
-0.058823529411765, -0.196078431372549, -0.176470588235294, +0.121568627450980,
+0.050980392156863, -0.168627450980392, -0.082352941176471, -0.129411764705882,
-0.211764705882353, -0.156862745098039, +0.043137254901961, -0.074509803921569,
-0.168627450980392, -0.007843137254902, -0.109803921568627, -0.239215686274510,
+0.043137254901961, +0.043137254901961, -0.200000000000000, -0.019607843137255,
+0.039215686274510, -0.129411764705882, -0.023529411764706, +0.082352941176471,
-0.082352941176471, +0.003921568627451, +0.164705882352941, -0.031372549019608,
-0.066666666666667, +0.156862745098039, +0.176470588235294, +0.039215686274510,
+0.050980392156863, +0.062745098039216, +0.074509803921569, +0.133333333333333,
+0.054901960784314, +0.039215686274510, +0.105882352941176, +0.180392156862745,
-0.086274509803922, -0.117647058823529, +0.180392156862745, +0.007843137254902,
-0.235294117647059, -0.094117647058824, +0.098039215686275, -0.043137254901961,
-0.078431372549020, +0.031372549019608, +0.015686274509804, +0.019607843137255,
+0.125490196078431, -0.054901960784314, -0.231372549019608, +0.074509803921569,
+0.003921568627451, -0.117647058823529, +0.070588235294118, -0.031372549019608,
-0.180392156862745, +0.058823529411765, +0.168627450980392, -0.082352941176471,
+0.023529411764706, +0.062745098039216, -0.203921568627451, -0.137254901960784,
+0.039215686274510, -0.027450980392157, -0.117647058823529, +0.011764705882353,
+0.003921568627451, -0.137254901960784, +0.050980392156863, +0.141176470588235,
-0.003921568627451, +0.058823529411765, +0.066666666666667, -0.027450980392157,
-0.180392156862745, -0.164705882352941, +0.105882352941176, -0.062745098039216,
-0.090196078431373, +0.078431372549020, -0.082352941176471, -0.168627450980392,
+0.011764705882353, +0.180392156862745, -0.062745098039216, -0.066666666666667,
+0.149019607843137, +0.031372549019608, +0.011764705882353, +0.113725490196078,
+0.086274509803922, +0.019607843137255, +0.047058823529412, +0.188235294117647,
-0.027450980392157, +0.039215686274510, +0.239215686274510, +0.086274509803922,
+0.082352941176471, +0.184313725490196, +0.137254901960784, +0.074509803921569,
+0.125490196078431, +0.172549019607843, -0.050980392156863, +0.019607843137255,
+0.211764705882353, +0.047058823529412, +0.007843137254902, +0.117647058823529,
+0.054901960784314, -0.141176470588235, +0.023529411764706, +0.082352941176471,
+0.019607843137255, +0.082352941176471, +0.007843137254902, -0.082352941176471,
-0.054901960784314, +0.109803921568627, -0.113725490196078, -0.105882352941176,
+0.094117647058824, -0.050980392156863, -0.156862745098039, +0.074509803921569,
+0.125490196078431, -0.035294117647059, +0.054901960784314, +0.003921568627451,
-0.125490196078431, -0.180392156862745, -0.019607843137255, -0.039215686274510,
-0.007843137254902, +0.113725490196078, -0.070588235294118, -0.172549019607843,
+0.058823529411765, +0.039215686274510, -0.200000000000000, -0.035294117647059,
-0.003921568627451, -0.192156862745098, -0.070588235294118, -0.003921568627451,
-0.200000000000000, -0.015686274509804, +0.145098039215686, -0.125490196078431,
-0.282352941176471, -0.031372549019608, -0.011764705882353, -0.149019607843137,
-0.011764705882353, -0.121568627450980, -0.168627450980392, -0.007843137254902,
-0.109803921568627, -0.192156862745098, -0.082352941176471, +0.023529411764706,
-0.129411764705882, -0.129411764705882, +0.098039215686275, +0.007843137254902,
-0.203921568627451, -0.066666666666667, +0.062745098039216, -0.160784313725490,
-0.152941176470588, -0.003921568627451, -0.105882352941176, -0.066666666666667,
+0.090196078431373, -0.058823529411765, -0.207843137254902, -0.019607843137255,
-0.094117647058824, -0.109803921568627, +0.086274509803922, -0.050980392156863,
-0.211764705882353, -0.007843137254902, +0.086274509803922, -0.105882352941176,
+0.086274509803922, +0.105882352941176, -0.137254901960784, -0.082352941176471,
-0.011764705882353, -0.145098039215686, -0.215686274509804, -0.007843137254902,
+0.019607843137255, -0.243137254901961, -0.121568627450980, +0.070588235294118,
-0.172549019607843, -0.101960784313725, +0.070588235294118, +0.000000000000000,
-0.098039215686275, -0.207843137254902, +0.066666666666667, +0.011764705882353,
-0.070588235294118, +0.082352941176471, -0.086274509803922, -0.207843137254902,
-0.031372549019608, +0.125490196078431, -0.145098039215686, -0.172549019607843,
+0.039215686274510, -0.086274509803922, -0.168627450980392, -0.039215686274510,
-0.160784313725490, -0.168627450980392, -0.168627450980392, -0.039215686274510,
-0.101960784313725, -0.054901960784314, +0.043137254901961, -0.082352941176471,
-0.062745098039216, +0.133333333333333, +0.105882352941176, +0.039215686274510,
+0.101960784313725, +0.215686274509804, -0.011764705882353, -0.062745098039216,
+0.074509803921569, +0.023529411764706, +0.062745098039216, +0.180392156862745,
+0.117647058823529, -0.133333333333333, -0.074509803921569, -0.015686274509804,
+0.054901960784314, +0.200000000000000, +0.156862745098039, +0.039215686274510,
+0.070588235294118, +0.223529411764706, -0.023529411764706, -0.035294117647059,
+0.184313725490196, +0.137254901960784, -0.007843137254902, -0.011764705882353,
+0.094117647058824, +0.109803921568627, +0.180392156862745, +0.231372549019608,
+0.168627450980392, +0.086274509803922, +0.133333333333333, +0.031372549019608,
+0.054901960784314, +0.156862745098039, -0.023529411764706, -0.050980392156863,
+0.196078431372549, +0.188235294117647, -0.023529411764706, +0.117647058823529,
+0.211764705882353, +0.023529411764706, +0.113725490196078, +0.203921568627451,
-0.007843137254902, +0.082352941176471, +0.231372549019608, +0.019607843137255,
-0.243137254901961, -0.058823529411765, +0.094117647058824, -0.031372549019608,
+0.105882352941176, +0.094117647058824, +0.031372549019608, +0.172549019607843,
+0.117647058823529, -0.062745098039216, +0.062745098039216, +0.184313725490196,
-0.066666666666667, -0.043137254901961, +0.215686274509804, +0.062745098039216,
-0.113725490196078, +0.078431372549020, +0.231372549019608, +0.078431372549020,
+0.003921568627451, +0.156862745098039, +0.090196078431373, +0.039215686274510,
+0.219607843137255, +0.090196078431373, -0.019607843137255, +0.145098039215686,
+0.082352941176471, +0.062745098039216, +0.239215686274510, +0.172549019607843,
-0.003921568627451, +0.098039215686275, +0.141176470588235, -0.035294117647059,
+0.149019607843137, +0.231372549019608, +0.062745098039216, +0.149019607843137,
+0.250980392156863, +0.047058823529412, -0.078431372549020, +0.156862745098039,
+0.270588235294118, +0.019607843137255, +0.007843137254902, +0.231372549019608,
+0.058823529411765, +0.023529411764706, +0.250980392156863, +0.215686274509804,
+0.074509803921569, -0.043137254901961, +0.129411764705882, +0.141176470588235,
+0.082352941176471, +0.188235294117647, +0.101960784313725, -0.019607843137255,
+0.078431372549020, +0.250980392156863, +0.047058823529412, -0.031372549019608,
+0.168627450980392, +0.113725490196078, +0.011764705882353, +0.047058823529412,
-0.043137254901961, -0.031372549019608, -0.149019607843137, -0.164705882352941,
-0.223529411764706, -0.200000000000000, +0.000000000000000, -0.007843137254902,
-0.207843137254902, -0.094117647058824, +0.035294117647059, -0.125490196078431,
-0.219607843137255, +0.054901960784314, -0.043137254901961, -0.160784313725490,
-0.047058823529412, -0.152941176470588, -0.137254901960784, +0.094117647058824,
+0.078431372549020, -0.176470588235294, -0.200000000000000, -0.176470588235294,
-0.160784313725490, +0.000000000000000, +0.082352941176471, -0.117647058823529,
-0.152941176470588, +0.101960784313725, +0.027450980392157, -0.125490196078431,
-0.054901960784314, +0.011764705882353, -0.062745098039216, -0.239215686274510,
-0.211764705882353, -0.141176470588235, -0.145098039215686, -0.058823529411765,
-0.050980392156863, -0.086274509803922, +0.031372549019608, -0.031372549019608,
-0.082352941176471, -0.003921568627451, -0.172549019607843, -0.219607843137255,
+0.035294117647059, +0.082352941176471, -0.066666666666667, +0.054901960784314,
+0.117647058823529, -0.082352941176471, -0.094117647058824, +0.011764705882353,
-0.062745098039216, -0.050980392156863, +0.149019607843137, +0.188235294117647,
-0.078431372549020, -0.176470588235294, +0.023529411764706, +0.094117647058824,
+0.058823529411765, +0.011764705882353, -0.035294117647059, +0.074509803921569,
+0.223529411764706, +0.031372549019608, -0.019607843137255, +0.211764705882353,
-0.019607843137255, -0.156862745098039, +0.094117647058824, +0.000000000000000,
-0.168627450980392, -0.031372549019608, +0.074509803921569, -0.031372549019608,
-0.117647058823529, -0.011764705882353, +0.050980392156863, +0.027450980392157,
+0.223529411764706, +0.109803921568627, -0.133333333333333, -0.011764705882353,
+0.027450980392157, -0.062745098039216, +0.039215686274510, +0.043137254901961,
-0.117647058823529, -0.011764705882353, +0.031372549019608, -0.105882352941176,
-0.011764705882353, +0.105882352941176, -0.050980392156863, -0.066666666666667,
+0.152941176470588, +0.121568627450980, -0.039215686274510, +0.039215686274510,
+0.117647058823529, +0.000000000000000, -0.062745098039216, +0.031372549019608,
-0.043137254901961, -0.125490196078431, -0.011764705882353, +0.043137254901961,
-0.043137254901961, -0.062745098039216, +0.054901960784314, +0.031372549019608,
+0.039215686274510, +0.074509803921569, -0.070588235294118, -0.105882352941176,
+0.023529411764706, +0.105882352941176, -0.003921568627451, -0.023529411764706,
+0.015686274509804, -0.074509803921569, -0.070588235294118, -0.082352941176471,
-0.011764705882353, +0.070588235294118, -0.031372549019608, +0.039215686274510,
+0.050980392156863, -0.086274509803922, +0.015686274509804, +0.180392156862745,
-0.043137254901961, -0.109803921568627, +0.168627450980392, +0.168627450980392,
-0.070588235294118, +0.094117647058824, +0.117647058823529, -0.015686274509804,
+0.145098039215686, +0.082352941176471, -0.090196078431373, +0.098039215686275,
+0.250980392156863, +0.145098039215686, +0.082352941176471, +0.101960784313725,
-0.039215686274510, -0.062745098039216, +0.149019607843137, +0.050980392156863,
-0.149019607843137, -0.011764705882353, +0.109803921568627, -0.101960784313725,
-0.117647058823529, +0.086274509803922, +0.098039215686275, +0.000000000000000,
-0.007843137254902, -0.011764705882353, -0.101960784313725, -0.062745098039216,
-0.054901960784314, -0.152941176470588, +0.031372549019608, +0.125490196078431,
+0.062745098039216, +0.129411764705882, +0.007843137254902, -0.196078431372549,
-0.035294117647059, +0.039215686274510, -0.192156862745098, -0.074509803921569,
+0.098039215686275, -0.043137254901961, -0.145098039215686, -0.094117647058824,
-0.094117647058824, -0.196078431372549, -0.031372549019608, +0.200000000000000,
+0.074509803921569, -0.200000000000000, -0.145098039215686, +0.000000000000000,
-0.050980392156863, +0.000000000000000, -0.082352941176471, -0.117647058823529,
+0.125490196078431, +0.086274509803922, -0.180392156862745, -0.003921568627451,
+0.090196078431373, -0.062745098039216, +0.003921568627451, +0.062745098039216,
-0.035294117647059, -0.113725490196078, -0.074509803921569, -0.137254901960784,
-0.219607843137255, -0.156862745098039, -0.074509803921569, -0.113725490196078,
+0.109803921568627, +0.164705882352941, -0.133333333333333, -0.211764705882353,
-0.152941176470588, -0.215686274509804, -0.074509803921569, +0.070588235294118,
-0.145098039215686, -0.164705882352941, -0.062745098039216, -0.149019607843137,
-0.192156862745098, +0.019607843137255, -0.031372549019608, -0.250980392156863,
-0.192156862745098, -0.035294117647059, -0.054901960784314, -0.176470588235294,
-0.078431372549020, -0.058823529411765, -0.149019607843137, -0.105882352941176,
-0.098039215686275, -0.188235294117647, -0.137254901960784, -0.031372549019608,
-0.188235294117647, -0.243137254901961, -0.043137254901961, -0.082352941176471,
-0.113725490196078, +0.031372549019608, -0.117647058823529, -0.243137254901961,
-0.035294117647059, -0.007843137254902, -0.203921568627451, -0.086274509803922,
-0.011764705882353, -0.200000000000000, -0.168627450980392, -0.137254901960784,
-0.035294117647059, +0.050980392156863, -0.070588235294118, -0.039215686274510,
+0.023529411764706, -0.094117647058824, -0.098039215686275, +0.098039215686275,
+0.031372549019608, -0.184313725490196, -0.043137254901961, +0.066666666666667,
-0.035294117647059, +0.066666666666667, +0.113725490196078, -0.082352941176471,
+0.047058823529412, +0.129411764705882, -0.058823529411765, +0.000000000000000,
+0.164705882352941, +0.082352941176471, +0.039215686274510, +0.121568627450980,
-0.015686274509804, -0.113725490196078, +0.125490196078431, +0.215686274509804,
-0.023529411764706, -0.082352941176471, +0.117647058823529, -0.074509803921569,
-0.105882352941176, +0.156862745098039, +0.145098039215686, +0.125490196078431,
+0.219607843137255, +0.200000000000000, +0.070588235294118, +0.121568627450980,
+0.160784313725490, +0.043137254901961, +0.141176470588235, +0.192156862745098,
+0.058823529411765, +0.152941176470588, +0.219607843137255, +0.019607843137255,
+0.019607843137255, +0.164705882352941, -0.105882352941176, -0.129411764705882,
+0.176470588235294, +0.156862745098039, +0.031372549019608, +0.098039215686275,
+0.203921568627451, -0.066666666666667, -0.098039215686275, +0.180392156862745,
+0.196078431372549, +0.007843137254902, +0.019607843137255, +0.074509803921569,
-0.031372549019608, +0.113725490196078, +0.117647058823529, -0.054901960784314,
+0.058823529411765, +0.219607843137255, +0.039215686274510, -0.011764705882353,
+0.172549019607843, +0.117647058823529, -0.023529411764706, +0.090196078431373,
+0.137254901960784, +0.054901960784314, +0.117647058823529, +0.176470588235294,
+0.094117647058824, +0.047058823529412, +0.082352941176471, -0.007843137254902,
+0.090196078431373, +0.243137254901961, +0.180392156862745, +0.117647058823529,
+0.117647058823529, -0.015686274509804, +0.000000000000000, +0.231372549019608,
+0.160784313725490, +0.031372549019608, +0.125490196078431, +0.125490196078431,
-0.031372549019608, +0.129411764705882, +0.243137254901961, +0.062745098039216,
-0.070588235294118, +0.019607843137255, +0.125490196078431, -0.086274509803922,
-0.007843137254902, +0.200000000000000, +0.117647058823529, +0.129411764705882,
+0.200000000000000, +0.007843137254902, -0.023529411764706, +0.168627450980392,
+0.015686274509804, -0.180392156862745, +0.023529411764706, +0.117647058823529,
+0.019607843137255, +0.188235294117647, +0.105882352941176, -0.156862745098039,
+0.043137254901961, +0.188235294117647, -0.094117647058824, +0.003921568627451,
+0.239215686274510, +0.101960784313725, +0.066666666666667, +0.125490196078431,
+0.156862745098039, +0.188235294117647, +0.082352941176471, +0.023529411764706,
+0.062745098039216, -0.011764705882353, -0.035294117647059, +0.074509803921569,
+0.023529411764706, -0.200000000000000, -0.149019607843137, -0.070588235294118,
-0.168627450980392, -0.027450980392157, +0.082352941176471, -0.200000000000000,
-0.129411764705882, +0.094117647058824, -0.090196078431373, -0.141176470588235,
+0.062745098039216, -0.094117647058824, -0.196078431372549, -0.039215686274510,
-0.137254901960784, -0.250980392156863, -0.062745098039216, +0.117647058823529,
-0.058823529411765, -0.129411764705882, +0.070588235294118, -0.101960784313725,
-0.243137254901961, -0.047058823529412, -0.031372549019608, -0.062745098039216,
+0.062745098039216, +0.031372549019608, -0.141176470588235, -0.137254901960784,
-0.125490196078431, -0.133333333333333, +0.043137254901961, +0.039215686274510,
-0.192156862745098, -0.125490196078431, +0.078431372549020, +0.039215686274510,
-0.094117647058824, -0.011764705882353, -0.117647058823529, -0.223529411764706,
+0.039215686274510, +0.078431372549020, -0.105882352941176, -0.023529411764706,
+0.188235294117647, -0.074509803921569, -0.188235294117647, +0.098039215686275,
+0.082352941176471, -0.058823529411765, +0.023529411764706, +0.027450980392157,
-0.149019607843137, -0.082352941176471, +0.062745098039216, -0.086274509803922,
-0.074509803921569, +0.082352941176471, +0.019607843137255, -0.007843137254902,
+0.039215686274510, -0.054901960784314, -0.137254901960784, +0.039215686274510,
+0.054901960784314, -0.082352941176471, +0.007843137254902, +0.137254901960784,
+0.043137254901961, -0.015686274509804, +0.003921568627451, -0.121568627450980,
-0.125490196078431, +0.047058823529412, +0.070588235294118, +0.094117647058824,
+0.113725490196078, -0.078431372549020, -0.156862745098039, +0.062745098039216,
+0.137254901960784, -0.015686274509804, -0.023529411764706, +0.015686274509804,
-0.054901960784314, +0.019607843137255, +0.180392156862745, +0.113725490196078,
-0.035294117647059, -0.007843137254902, +0.113725490196078, -0.082352941176471,
-0.168627450980392, +0.074509803921569, +0.129411764705882, +0.086274509803922,
+0.223529411764706, +0.062745098039216, -0.039215686274510, +0.188235294117647,
+0.105882352941176, -0.113725490196078, -0.007843137254902, +0.125490196078431,
-0.011764705882353, +0.050980392156863, +0.043137254901961, -0.149019607843137,
+0.023529411764706, +0.215686274509804, -0.035294117647059, -0.105882352941176,
+0.133333333333333, +0.113725490196078, +0.062745098039216, +0.137254901960784,
+0.164705882352941, +0.196078431372549, +0.176470588235294, +0.149019607843137,
+0.207843137254902, +0.125490196078431, +0.066666666666667, +0.172549019607843,
+0.207843137254902, +0.070588235294118, +0.070588235294118, +0.105882352941176,
-0.043137254901961, +0.031372549019608, +0.227450980392157, -0.031372549019608,
-0.098039215686275, +0.168627450980392, +0.058823529411765, -0.070588235294118,
+0.176470588235294, +0.086274509803922, -0.039215686274510, +0.149019607843137,
+0.105882352941176, -0.082352941176471, -0.031372549019608, +0.168627450980392,
+0.074509803921569, +0.019607843137255, +0.203921568627451, +0.164705882352941,
-0.058823529411765, -0.086274509803922, -0.129411764705882, -0.129411764705882,
+0.113725490196078, +0.090196078431373, -0.019607843137255, +0.003921568627451,
-0.137254901960784, -0.196078431372549, +0.070588235294118, +0.176470588235294,
-0.074509803921569, -0.129411764705882, -0.050980392156863, -0.035294117647059,
-0.023529411764706, +0.074509803921569, +0.039215686274510, -0.105882352941176,
+0.058823529411765, +0.109803921568627, -0.164705882352941, -0.101960784313725,
+0.156862745098039, -0.019607843137255, -0.262745098039216, -0.043137254901961,
+0.011764705882353, -0.149019607843137, +0.003921568627451, +0.062745098039216,
-0.156862745098039, -0.156862745098039, +0.086274509803922, -0.027450980392157,
-0.035294117647059, +0.027450980392157, -0.125490196078431, +0.011764705882353,
+0.074509803921569, -0.149019607843137, -0.192156862745098, +0.078431372549020,
+0.105882352941176, -0.117647058823529, -0.121568627450980, +0.031372549019608,
-0.113725490196078, -0.137254901960784, +0.050980392156863, -0.070588235294118,
-0.168627450980392, +0.000000000000000, -0.094117647058824, -0.137254901960784,
+0.039215686274510, -0.062745098039216, -0.117647058823529, -0.007843137254902,
-0.011764705882353, -0.156862745098039, -0.105882352941176, -0.105882352941176,
-0.215686274509804, -0.141176470588235, +0.062745098039216, +0.043137254901961,
-0.164705882352941, -0.137254901960784, +0.027450980392157, -0.058823529411765,
-0.247058823529412, -0.137254901960784, -0.043137254901961, -0.125490196078431,
+0.054901960784314, -0.015686274509804, -0.149019607843137, +0.062745098039216,
+0.011764705882353, -0.184313725490196, -0.105882352941176, -0.039215686274510,
-0.133333333333333, -0.043137254901961, -0.023529411764706, -0.196078431372549,
-0.054901960784314, +0.133333333333333, -0.023529411764706, -0.188235294117647,
-0.023529411764706, -0.031372549019608, -0.145098039215686, -0.027450980392157,
-0.050980392156863, -0.023529411764706, -0.094117647058824, -0.094117647058824,
-0.011764705882353, -0.082352941176471, -0.086274509803922, +0.121568627450980,
+0.188235294117647, +0.098039215686275, +0.113725490196078, +0.039215686274510,
-0.149019607843137, -0.082352941176471, +0.133333333333333, +0.074509803921569,
-0.023529411764706, +0.066666666666667, -0.054901960784314, -0.125490196078431,
+0.121568627450980, +0.141176470588235, +0.023529411764706, +0.090196078431373,
+0.113725490196078, -0.039215686274510, -0.031372549019608, +0.188235294117647,
+0.117647058823529, +0.031372549019608, +0.145098039215686, +0.196078431372549,
+0.149019607843137, +0.152941176470588, +0.039215686274510, -0.078431372549020,
+0.164705882352941, +0.094117647058824, +0.019607843137255, +0.180392156862745,
+0.062745098039216, -0.062745098039216, +0.066666666666667, +0.211764705882353,
+0.078431372549020, +0.062745098039216, +0.090196078431373, -0.007843137254902,
+0.050980392156863, +0.168627450980392, +0.058823529411765, -0.070588235294118,
+0.133333333333333, +0.235294117647059, +0.019607843137255, +0.027450980392157,
+0.231372549019608, +0.168627450980392, -0.137254901960784, -0.035294117647059,
+0.152941176470588, +0.011764705882353, +0.109803921568627, +0.258823529411765,
+0.054901960784314, -0.050980392156863, +0.141176470588235, +0.086274509803922,
+0.098039215686275, +0.164705882352941, +0.000000000000000, +0.121568627450980,
+0.290196078431373, +0.074509803921569, -0.101960784313725, +0.105882352941176,
+0.203921568627451, +0.050980392156863, +0.027450980392157, +0.188235294117647,
+0.082352941176471, -0.003921568627451, +0.180392156862745, +0.125490196078431,
+0.000000000000000, +0.160784313725490, +0.105882352941176, +0.019607843137255,
+0.180392156862745, +0.086274509803922, +0.000000000000000, +0.129411764705882,
+0.184313725490196, +0.043137254901961, +0.117647058823529, +0.168627450980392,
-0.082352941176471, -0.149019607843137, +0.094117647058824, +0.231372549019608,
+0.105882352941176, +0.054901960784314, +0.200000000000000, +0.200000000000000,
-0.027450980392157, +0.000000000000000, +0.094117647058824, -0.031372549019608,
+0.094117647058824, +0.015686274509804, -0.137254901960784, +0.090196078431373,
+0.094117647058824, -0.062745098039216, +0.062745098039216, +0.152941176470588,
+0.113725490196078, +0.086274509803922, +0.035294117647059, -0.160784313725490,
-0.094117647058824, +0.176470588235294, +0.145098039215686, -0.015686274509804,
+0.031372549019608, -0.035294117647059, -0.164705882352941, +0.000000000000000,
-0.031372549019608, +0.047058823529412, -0.117647058823529, -0.223529411764706,
-0.082352941176471, -0.090196078431373, -0.168627450980392, -0.011764705882353,
-0.066666666666667, -0.188235294117647, +0.035294117647059, +0.062745098039216,
-0.207843137254902, -0.219607843137255, -0.137254901960784, -0.137254901960784,
-0.043137254901961, +0.090196078431373, -0.141176470588235, -0.250980392156863,
-0.035294117647059, -0.007843137254902, -0.121568627450980, -0.137254901960784,
-0.039215686274510, -0.152941176470588, -0.172549019607843, +0.027450980392157,
-0.090196078431373, -0.152941176470588, -0.007843137254902, -0.062745098039216,
-0.094117647058824, +0.050980392156863, +0.003921568627451, -0.164705882352941,
+0.058823529411765, +0.007843137254902, -0.188235294117647, -0.094117647058824,
-0.145098039215686, -0.192156862745098, -0.086274509803922, -0.019607843137255,
-0.121568627450980, +0.050980392156863, +0.160784313725490, -0.074509803921569,
-0.176470588235294, +0.000000000000000, -0.094117647058824, -0.200000000000000,
+0.007843137254902, +0.043137254901961, -0.082352941176471, -0.105882352941176,
+0.070588235294118, +0.121568627450980, -0.066666666666667, -0.027450980392157,
+0.062745098039216, -0.047058823529412, -0.054901960784314, +0.023529411764706,
-0.023529411764706, -0.070588235294118, -0.062745098039216, -0.137254901960784,
-0.054901960784314, +0.074509803921569, -0.070588235294118, -0.070588235294118,
+0.160784313725490, +0.149019607843137, -0.007843137254902, +0.031372549019608,
+0.078431372549020, -0.101960784313725, -0.125490196078431, +0.070588235294118,
+0.043137254901961, -0.039215686274510, +0.000000000000000, -0.086274509803922,
-0.094117647058824, +0.039215686274510, +0.023529411764706, +0.011764705882353,
+0.098039215686275, -0.019607843137255, -0.125490196078431, +0.070588235294118,
+0.160784313725490, -0.031372549019608, -0.058823529411765, +0.168627450980392,
-0.011764705882353, -0.168627450980392, +0.043137254901961, +0.145098039215686,
+0.078431372549020, +0.031372549019608, +0.117647058823529, +0.090196078431373,
-0.050980392156863, +0.023529411764706, +0.113725490196078, -0.035294117647059,
+0.094117647058824, +0.082352941176471, -0.074509803921569, +0.125490196078431,
+0.117647058823529, -0.007843137254902, +0.090196078431373, +0.149019607843137,
+0.192156862745098, +0.117647058823529, +0.101960784313725, -0.062745098039216,
-0.164705882352941, +0.086274509803922, +0.054901960784314, +0.000000000000000,
+0.141176470588235, +0.066666666666667, -0.094117647058824, +0.000000000000000,
+0.070588235294118, +0.215686274509804, +0.172549019607843, +0.011764705882353,
+0.090196078431373, +0.078431372549020, -0.098039215686275, +0.058823529411765,
+0.062745098039216, -0.137254901960784, +0.066666666666667, +0.188235294117647,
-0.027450980392157, -0.011764705882353, +0.000000000000000, -0.188235294117647,
-0.074509803921569, +0.223529411764706, +0.023529411764706, -0.152941176470588,
-0.035294117647059, -0.011764705882353, -0.105882352941176, -0.074509803921569,
+0.117647058823529, -0.070588235294118, -0.152941176470588, +0.094117647058824,
-0.101960784313725, -0.207843137254902, +0.082352941176471, -0.019607843137255,
-0.160784313725490, +0.007843137254902, +0.019607843137255, -0.137254901960784,
+0.023529411764706, +0.125490196078431, +0.047058823529412, +0.086274509803922,
+0.019607843137255, -0.129411764705882, -0.058823529411765, -0.019607843137255,
-0.258823529411765, -0.129411764705882, +0.070588235294118, -0.047058823529412,
-0.145098039215686, +0.074509803921569, -0.019607843137255, -0.141176470588235,
+0.027450980392157, -0.058823529411765, -0.164705882352941, -0.141176470588235,
+0.062745098039216, +0.015686274509804, -0.149019607843137, -0.070588235294118,
-0.039215686274510, -0.125490196078431, -0.176470588235294, -0.160784313725490,
-0.164705882352941, -0.047058823529412, -0.062745098039216, -0.262745098039216,
-0.219607843137255, +0.066666666666667, -0.039215686274510, -0.231372549019608,
-0.090196078431373, -0.082352941176471, -0.196078431372549, -0.149019607843137,
+0.027450980392157, -0.101960784313725, -0.250980392156863, -0.145098039215686,
-0.156862745098039, -0.172549019607843, -0.062745098039216, -0.164705882352941,
-0.145098039215686, -0.074509803921569, -0.227450980392157, -0.215686274509804,
+0.019607843137255, +0.015686274509804, -0.168627450980392, -0.050980392156863,
+0.003921568627451, -0.133333333333333, -0.196078431372549, +0.086274509803922,
-0.011764705882353, -0.211764705882353, -0.062745098039216, -0.098039215686275,
-0.196078431372549, -0.211764705882353, -0.094117647058824, -0.137254901960784,
-0.215686274509804, -0.113725490196078, -0.039215686274510, -0.192156862745098,
-0.125490196078431, -0.074509803921569, -0.235294117647059, -0.137254901960784,
-0.125490196078431, -0.219607843137255, -0.156862745098039, -0.149019607843137,
-0.050980392156863, -0.050980392156863, +0.011764705882353, -0.023529411764706,
-0.270588235294118, -0.062745098039216, -0.101960784313725, -0.223529411764706,
-0.031372549019608, -0.019607843137255, -0.176470588235294, -0.105882352941176,
-0.031372549019608, +0.050980392156863, +0.007843137254902, +0.015686274509804,
+0.070588235294118, -0.027450980392157, -0.188235294117647, +0.019607843137255,
+0.125490196078431, -0.113725490196078, +0.058823529411765, +0.211764705882353,
-0.031372549019608, +0.000000000000000, +0.188235294117647, -0.054901960784314,
-0.101960784313725, +0.156862745098039, +0.007843137254902, -0.105882352941176,
+0.074509803921569, +0.172549019607843, -0.043137254901961, -0.125490196078431,
+0.180392156862745, +0.007843137254902, -0.160784313725490, +0.192156862745098,
+0.011764705882353, -0.203921568627451, +0.145098039215686, +0.129411764705882,
-0.058823529411765, +0.098039215686275, +0.156862745098039, +0.054901960784314,
+0.086274509803922, +0.078431372549020, +0.043137254901961, +0.160784313725490,
+0.219607843137255, +0.043137254901961, +0.101960784313725, +0.172549019607843,
-0.094117647058824, -0.082352941176471, +0.125490196078431, +0.062745098039216,
-0.023529411764706, +0.196078431372549, +0.074509803921569, -0.082352941176471,
+0.141176470588235, +0.156862745098039, +0.019607843137255, +0.007843137254902,
+0.200000000000000, +0.054901960784314, -0.113725490196078, +0.000000000000000,
+0.105882352941176, +0.113725490196078, +0.050980392156863, +0.062745098039216,
+0.015686274509804, +0.145098039215686, +0.231372549019608, +0.031372549019608,
-0.094117647058824, +0.149019607843137, +0.168627450980392, -0.047058823529412,
+0.043137254901961, +0.043137254901961, -0.133333333333333, -0.098039215686275,
+0.160784313725490, +0.121568627450980, -0.011764705882353, +0.094117647058824,
+0.101960784313725, +0.011764705882353, +0.113725490196078, +0.062745098039216,
+0.054901960784314, +0.168627450980392, +0.003921568627451, -0.105882352941176,
+0.074509803921569, +0.105882352941176, -0.098039215686275, +0.062745098039216,
+0.200000000000000, +0.105882352941176, -0.019607843137255, +0.152941176470588,
+0.125490196078431, -0.027450980392157, +0.113725490196078, +0.027450980392157,
-0.109803921568627, -0.054901960784314, +0.082352941176471, -0.031372549019608,
-0.101960784313725, +0.031372549019608, +0.105882352941176, +0.047058823529412,
+0.035294117647059, +0.050980392156863, -0.062745098039216, -0.054901960784314,
-0.015686274509804, -0.086274509803922, -0.019607843137255, -0.039215686274510,
+0.007843137254902, +0.015686274509804, +0.047058823529412, +0.125490196078431,
-0.082352941176471, +0.082352941176471, +0.137254901960784, -0.113725490196078,
-0.086274509803922, -0.011764705882353, -0.121568627450980, +0.000000000000000,
-0.019607843137255, +0.090196078431373, -0.117647058823529, -0.074509803921569,
+0.054901960784314, -0.031372549019608, -0.125490196078431, +0.011764705882353,
+0.094117647058824, -0.207843137254902, -0.121568627450980, +0.137254901960784,
-0.074509803921569, -0.137254901960784, +0.137254901960784, -0.035294117647059,
-0.082352941176471, +0.145098039215686, -0.011764705882353, -0.172549019607843,
-0.031372549019608, +0.200000000000000, -0.007843137254902, -0.176470588235294,
+0.117647058823529, +0.027450980392157, -0.180392156862745, +0.121568627450980,
+0.007843137254902, -0.231372549019608, +0.023529411764706, +0.023529411764706,
-0.180392156862745, -0.043137254901961, +0.019607843137255, -0.098039215686275,
+0.000000000000000, -0.050980392156863, -0.207843137254902, -0.070588235294118,
+0.043137254901961, -0.086274509803922, +0.054901960784314, +0.231372549019608,
-0.031372549019608, -0.149019607843137, +0.062745098039216, -0.039215686274510,
-0.164705882352941, +0.082352941176471, +0.019607843137255, -0.192156862745098,
-0.039215686274510, +0.050980392156863, -0.101960784313725, -0.050980392156863,
+0.121568627450980, -0.007843137254902, -0.094117647058824, -0.043137254901961,
+0.015686274509804, +0.043137254901961, +0.043137254901961, +0.054901960784314,
-0.070588235294118, +0.000000000000000, +0.152941176470588, +0.050980392156863,
-0.094117647058824, +0.027450980392157, +0.066666666666667, -0.027450980392157,
+0.137254901960784, +0.180392156862745, -0.078431372549020, -0.094117647058824,
+0.117647058823529, +0.027450980392157, -0.007843137254902, +0.125490196078431,
+0.082352941176471, -0.043137254901961, +0.000000000000000, +0.007843137254902,
+0.000000000000000, +0.184313725490196, +0.160784313725490, +0.019607843137255,
+0.035294117647059, +0.019607843137255, -0.133333333333333, -0.031372549019608,
+0.137254901960784, +0.043137254901961, -0.054901960784314, +0.070588235294118,
+0.019607843137255, -0.050980392156863, +0.196078431372549, +0.200000000000000,
-0.043137254901961, +0.054901960784314, +0.200000000000000, -0.039215686274510,
-0.105882352941176, +0.086274509803922, +0.145098039215686, +0.105882352941176,
+0.149019607843137, +0.176470588235294, +0.105882352941176, +0.141176470588235,
+0.094117647058824, -0.027450980392157, +0.125490196078431, +0.105882352941176,
+0.043137254901961, +0.105882352941176, +0.145098039215686, +0.129411764705882,
+0.015686274509804, +0.117647058823529, +0.168627450980392, +0.062745098039216,
+0.113725490196078, +0.156862745098039, +0.011764705882353, +0.090196078431373,
+0.062745098039216, +0.207843137254902, +0.043137254901961, +0.043137254901961,
+0.211764705882353, +0.086274509803922, -0.050980392156863, +0.074509803921569,
+0.270588235294118, +0.043137254901961, -0.074509803921569, +0.152941176470588,
+0.086274509803922, -0.019607843137255, +0.172549019607843, +0.074509803921569,
+0.023529411764706, +0.247058823529412, +0.141176470588235, -0.113725490196078,
-0.078431372549020, +0.200000000000000, +0.047058823529412, -0.086274509803922,
+0.156862745098039, +0.101960784313725, -0.031372549019608, +0.168627450980392,
+0.121568627450980, -0.070588235294118, +0.086274509803922, +0.105882352941176,
-0.105882352941176, -0.015686274509804, +0.125490196078431, -0.086274509803922,
+0.000000000000000, +0.113725490196078, -0.062745098039216, +0.007843137254902,
+0.031372549019608, -0.180392156862745, -0.043137254901961, +0.196078431372549,
+0.066666666666667, -0.086274509803922, +0.082352941176471, +0.019607843137255,
-0.160784313725490, +0.054901960784314, +0.113725490196078, -0.039215686274510,
+0.019607843137255, +0.043137254901961, -0.188235294117647, -0.070588235294118,
+0.062745098039216, -0.113725490196078, -0.149019607843137, -0.113725490196078,
-0.094117647058824, -0.207843137254902, -0.156862745098039, +0.011764705882353,
-0.129411764705882, -0.231372549019608, -0.050980392156863, -0.082352941176471,
-0.231372549019608, -0.109803921568627, -0.113725490196078, -0.180392156862745,
+0.050980392156863, +0.196078431372549, -0.117647058823529, -0.168627450980392,
+0.105882352941176, -0.062745098039216, -0.109803921568627, +0.054901960784314,
-0.043137254901961, -0.200000000000000, -0.113725490196078, -0.082352941176471,
-0.156862745098039, -0.019607843137255, +0.043137254901961, +0.019607843137255,
+0.015686274509804, -0.003921568627451, -0.074509803921569, -0.121568627450980,
+0.019607843137255, -0.054901960784314, -0.180392156862745, +0.011764705882353,
-0.023529411764706, -0.156862745098039, +0.070588235294118, +0.188235294117647,
-0.113725490196078, -0.043137254901961, +0.192156862745098, -0.007843137254902,
-0.203921568627451, -0.105882352941176, +0.054901960784314, -0.019607843137255,
+0.007843137254902, +0.043137254901961, -0.086274509803922, +0.078431372549020,
+0.047058823529412, -0.223529411764706, -0.082352941176471, -0.007843137254902,
-0.145098039215686, -0.070588235294118, +0.094117647058824, -0.054901960784314,
-0.164705882352941, +0.054901960784314, -0.007843137254902, -0.137254901960784,
+0.019607843137255, +0.062745098039216, -0.121568627450980, -0.047058823529412,
+0.003921568627451, +0.086274509803922, -0.043137254901961, -0.098039215686275,
+0.113725490196078, +0.019607843137255, -0.192156862745098, -0.066666666666667,
+0.137254901960784, +0.039215686274510, -0.031372549019608, +0.098039215686275,
-0.007843137254902, -0.074509803921569, +0.074509803921569, +0.007843137254902,
-0.082352941176471, +0.094117647058824, +0.074509803921569, -0.125490196078431,
-0.090196078431373, +0.082352941176471, -0.043137254901961, -0.094117647058824,
+0.054901960784314, -0.066666666666667, -0.078431372549020, +0.129411764705882,
+0.121568627450980, -0.043137254901961, +0.058823529411765, +0.168627450980392,
-0.050980392156863, -0.062745098039216, +0.149019607843137, +0.027450980392157,
+0.039215686274510, +0.105882352941176, -0.027450980392157, +0.117647058823529,
+0.160784313725490, -0.066666666666667, +0.003921568627451, +0.156862745098039,
+0.078431372549020, +0.035294117647059, +0.180392156862745, +0.133333333333333,
-0.074509803921569, +0.058823529411765, +0.149019607843137, +0.094117647058824,
+0.176470588235294, +0.211764705882353, -0.066666666666667, +0.031372549019608,
+0.176470588235294, -0.027450980392157, -0.074509803921569, +0.047058823529412,
+0.180392156862745, -0.027450980392157, -0.050980392156863, +0.188235294117647,
+0.160784313725490, -0.007843137254902, +0.117647058823529, +0.200000000000000,
+0.062745098039216, +0.113725490196078, +0.074509803921569, -0.078431372549020,
+0.035294117647059, +0.278431372549020, +0.043137254901961, -0.137254901960784,
+0.137254901960784, -0.007843137254902, -0.082352941176471, +0.211764705882353,
+0.207843137254902, +0.011764705882353, +0.074509803921569, +0.098039215686275,
-0.039215686274510, -0.003921568627451, +0.062745098039216, +0.133333333333333,
+0.156862745098039, +0.105882352941176, +0.039215686274510, -0.011764705882353,
+0.137254901960784, +0.133333333333333, -0.047058823529412, +0.094117647058824,
+0.109803921568627, -0.043137254901961, +0.070588235294118, +0.200000000000000,
-0.003921568627451, -0.011764705882353, +0.188235294117647, +0.149019607843137,
+0.000000000000000, -0.043137254901961, +0.094117647058824, +0.054901960784314,
+0.078431372549020, +0.105882352941176, -0.070588235294118, +0.098039215686275,
+0.250980392156863, -0.019607843137255, -0.039215686274510, +0.098039215686275,
-0.035294117647059, -0.078431372549020, +0.156862745098039, +0.074509803921569,
-0.090196078431373, +0.145098039215686, +0.113725490196078, -0.156862745098039,
-0.043137254901961, +0.125490196078431, -0.113725490196078, -0.054901960784314,
-0.086274509803922, +0.043137254901961, -0.058823529411765, -0.137254901960784,
+0.050980392156863, +0.015686274509804, -0.180392156862745, -0.043137254901961,
+0.062745098039216, -0.137254901960784, -0.200000000000000, -0.007843137254902,
-0.125490196078431, -0.192156862745098, +0.035294117647059, -0.043137254901961,
-0.164705882352941, -0.023529411764706, -0.039215686274510, -0.129411764705882,
+0.000000000000000, +0.043137254901961, -0.160784313725490, -0.137254901960784,
+0.043137254901961, -0.160784313725490, -0.250980392156863, -0.050980392156863,
-0.019607843137255, -0.180392156862745, -0.168627450980392, +0.031372549019608,
-0.125490196078431, -0.149019607843137, +0.027450980392157, -0.117647058823529,
-0.050980392156863, -0.039215686274510, -0.219607843137255, -0.023529411764706,
+0.129411764705882, -0.070588235294118, -0.043137254901961, +0.054901960784314,
-0.152941176470588, -0.164705882352941, +0.062745098039216, +0.039215686274510,
-0.125490196078431, -0.094117647058824, -0.129411764705882, -0.141176470588235,
+0.058823529411765, +0.125490196078431, -0.109803921568627, +0.000000000000000,
+0.247058823529412, +0.066666666666667, -0.133333333333333, -0.019607843137255,
+0.168627450980392, -0.011764705882353, -0.062745098039216, +0.074509803921569,
+0.031372549019608, -0.015686274509804, +0.094117647058824, +0.129411764705882,
+0.074509803921569, +0.054901960784314, -0.003921568627451, -0.117647058823529,
-0.113725490196078, +0.058823529411765, -0.015686274509804, -0.168627450980392,
+0.074509803921569, -0.035294117647059, -0.211764705882353, +0.070588235294118,
+0.133333333333333, -0.050980392156863, -0.015686274509804, +0.000000000000000,
-0.090196078431373, -0.050980392156863, -0.058823529411765, -0.086274509803922,
-0.011764705882353, +0.050980392156863, -0.074509803921569, -0.094117647058824,
-0.019607843137255, -0.050980392156863, -0.105882352941176, +0.015686274509804,
+0.007843137254902, -0.125490196078431, +0.007843137254902, +0.039215686274510,
-0.121568627450980, -0.113725490196078, -0.011764705882353, -0.050980392156863,
-0.039215686274510, -0.011764705882353, -0.074509803921569, -0.137254901960784,
+0.000000000000000, +0.043137254901961, -0.082352941176471, -0.011764705882353,
+0.101960784313725, +0.015686274509804, -0.062745098039216, -0.031372549019608,
-0.125490196078431, -0.125490196078431, +0.086274509803922, +0.086274509803922,
-0.023529411764706, +0.117647058823529, +0.156862745098039, -0.078431372549020,
-0.062745098039216, +0.152941176470588, -0.074509803921569, -0.031372549019608,
-0.050980392156863, +0.098039215686275, +0.011764705882353, -0.023529411764706,
+0.156862745098039, +0.152941176470588, -0.043137254901961, +0.078431372549020,
+0.231372549019608, +0.027450980392157, -0.180392156862745, +0.062745098039216,
+0.090196078431373, -0.074509803921569, +0.137254901960784, +0.149019607843137,
+0.066666666666667, +0.156862745098039, +0.105882352941176, +0.003921568627451,
+0.145098039215686, +0.239215686274510, +0.031372549019608, -0.019607843137255,
+0.211764705882353, +0.137254901960784, -0.082352941176471, +0.027450980392157,
+0.145098039215686, +0.007843137254902, -0.039215686274510, +0.164705882352941,
+0.054901960784314, +0.019607843137255, +0.156862745098039, -0.031372549019608,
+0.047058823529412, +0.176470588235294, -0.019607843137255, -0.039215686274510,
+0.078431372549020, -0.094117647058824, -0.070588235294118, +0.156862745098039,
-0.050980392156863, -0.133333333333333, +0.098039215686275, +0.070588235294118,
-0.050980392156863, +0.054901960784314, -0.074509803921569, -0.215686274509804,
+0.031372549019608, +0.105882352941176, -0.137254901960784, -0.058823529411765,
+0.243137254901961, +0.149019607843137, -0.125490196078431, -0.043137254901961,
+0.101960784313725, -0.117647058823529, -0.211764705882353, -0.007843137254902,
-0.078431372549020, -0.156862745098039, -0.027450980392157, -0.113725490196078,
-0.149019607843137, -0.090196078431373, -0.098039215686275, -0.188235294117647,
-0.129411764705882, +0.027450980392157, -0.066666666666667, -0.176470588235294,
+0.105882352941176, +0.066666666666667, -0.200000000000000, -0.015686274509804,
+0.062745098039216, -0.137254901960784, -0.058823529411765, -0.019607843137255,
-0.117647058823529, -0.070588235294118, -0.082352941176471, -0.164705882352941,
-0.101960784313725, +0.015686274509804, -0.137254901960784, -0.050980392156863,
+0.066666666666667, -0.094117647058824, -0.192156862745098, -0.003921568627451,
+0.050980392156863, -0.192156862745098, -0.117647058823529, -0.074509803921569,
-0.215686274509804, -0.156862745098039, -0.031372549019608, -0.176470588235294,
-0.243137254901961, -0.015686274509804, -0.011764705882353, -0.239215686274510,
-0.105882352941176, -0.047058823529412, -0.137254901960784, -0.054901960784314,
-0.098039215686275, -0.141176470588235, -0.117647058823529, -0.082352941176471,
-0.160784313725490, -0.137254901960784, -0.043137254901961, -0.141176470588235,
-0.211764705882353, -0.019607843137255, +0.066666666666667, -0.160784313725490,
-0.105882352941176, +0.066666666666667, -0.137254901960784, -0.015686274509804,
-0.039215686274510, +0.003921568627451, -0.039215686274510, -0.105882352941176,
+0.000000000000000, +0.000000000000000, -0.168627450980392, -0.074509803921569,
+0.141176470588235, +0.086274509803922, -0.117647058823529, -0.050980392156863,
-0.035294117647059, -0.156862745098039, +0.000000000000000, +0.015686274509804,
-0.031372549019608, +0.086274509803922, +0.011764705882353, -0.137254901960784,
-0.043137254901961, +0.156862745098039, +0.082352941176471, -0.027450980392157,
+0.086274509803922, +0.066666666666667, -0.043137254901961, +0.054901960784314,
+0.176470588235294, +0.050980392156863, +0.023529411764706, +0.105882352941176,
-0.015686274509804, -0.043137254901961, +0.086274509803922, -0.031372549019608,
-0.011764705882353, +0.133333333333333, -0.011764705882353, -0.023529411764706,
+0.156862745098039, +0.007843137254902, -0.035294117647059, +0.207843137254902,
+0.058823529411765, -0.149019607843137, +0.047058823529412, +0.090196078431373,
-0.015686274509804, +0.145098039215686, +0.050980392156863, -0.227450980392157,
+0.007843137254902, +0.203921568627451, -0.015686274509804, -0.031372549019608,
+0.207843137254902, +0.200000000000000, -0.058823529411765, +0.000000000000000,
+0.211764705882353, +0.094117647058824, -0.125490196078431, +0.058823529411765,
+0.094117647058824, -0.011764705882353, +0.145098039215686, +0.125490196078431,
+0.082352941176471, +0.141176470588235, +0.200000000000000, +0.113725490196078,
+0.101960784313725, +0.231372549019608, +0.094117647058824, -0.086274509803922,
+0.129411764705882, +0.168627450980392, -0.070588235294118, +0.047058823529412,
+0.188235294117647, +0.031372549019608, +0.062745098039216, +0.125490196078431,
+0.054901960784314, +0.125490196078431, +0.129411764705882, +0.031372549019608,
+0.074509803921569, +0.168627450980392, -0.011764705882353, +0.003921568627451,
+0.215686274509804, +0.172549019607843, +0.000000000000000, +0.054901960784314,
+0.196078431372549, +0.074509803921569, +0.066666666666667, +0.160784313725490,
+0.117647058823529, +0.098039215686275, +0.188235294117647, +0.121568627450980,
-0.035294117647059, +0.105882352941176, +0.262745098039216, +0.062745098039216,
+0.066666666666667, +0.117647058823529, +0.062745098039216, +0.180392156862745,
+0.149019607843137, +0.050980392156863, +0.090196078431373, +0.168627450980392,
+0.062745098039216, +0.050980392156863, +0.141176470588235, +0.058823529411765,
-0.090196078431373, +0.039215686274510, +0.137254901960784, -0.109803921568627,
+0.011764705882353, +0.223529411764706, +0.007843137254902, +0.086274509803922,
-0.031372549019608, -0.070588235294118, +0.007843137254902, -0.086274509803922,
-0.027450980392157, -0.070588235294118, -0.172549019607843, -0.082352941176471,
+0.086274509803922, -0.011764705882353, -0.137254901960784, -0.011764705882353,
-0.043137254901961, -0.207843137254902, -0.125490196078431, -0.149019607843137,
-0.250980392156863, -0.039215686274510, -0.007843137254902, -0.231372549019608,
-0.231372549019608, +0.011764705882353, -0.054901960784314, -0.184313725490196,
-0.015686274509804, -0.101960784313725, -0.250980392156863, -0.133333333333333,
-0.023529411764706, -0.180392156862745, -0.133333333333333, -0.054901960784314,
-0.200000000000000, -0.243137254901961, -0.094117647058824, -0.164705882352941,
-0.145098039215686, +0.015686274509804, -0.200000000000000, -0.176470588235294,
+0.074509803921569, -0.074509803921569, -0.180392156862745, -0.023529411764706,
-0.054901960784314, -0.231372549019608, -0.105882352941176, -0.039215686274510,
-0.152941176470588, +0.015686274509804, +0.019607843137255, -0.231372549019608,
-0.192156862745098, -0.031372549019608, -0.137254901960784, -0.129411764705882,
+0.019607843137255, +0.035294117647059, -0.156862745098039, -0.192156862745098,
+0.078431372549020, +0.125490196078431, -0.117647058823529, -0.094117647058824,
-0.066666666666667, -0.156862745098039, -0.039215686274510, +0.035294117647059,
-0.003921568627451, +0.039215686274510, +0.086274509803922, +0.000000000000000,
-0.023529411764706, +0.062745098039216, -0.023529411764706, -0.137254901960784,
+0.023529411764706, +0.015686274509804, -0.180392156862745, -0.117647058823529,
+0.058823529411765, -0.019607843137255, -0.105882352941176, -0.094117647058824,
-0.113725490196078, -0.023529411764706, +0.011764705882353, -0.113725490196078,
-0.027450980392157, +0.082352941176471, -0.086274509803922, -0.192156862745098,
-0.082352941176471, +0.054901960784314, -0.019607843137255, -0.031372549019608,
+0.019607843137255, -0.003921568627451, +0.027450980392157, +0.058823529411765,
+0.015686274509804, +0.035294117647059, +0.090196078431373, +0.007843137254902,
-0.121568627450980, -0.125490196078431, +0.054901960784314, +0.039215686274510,
+0.003921568627451, +0.031372549019608, -0.035294117647059, +0.082352941176471,
+0.145098039215686, +0.000000000000000, -0.003921568627451, +0.054901960784314,
-0.062745098039216, -0.074509803921569, +0.074509803921569, +0.086274509803922,
-0.058823529411765, +0.011764705882353, +0.054901960784314, -0.160784313725490,
-0.078431372549020, +0.094117647058824, -0.066666666666667, -0.019607843137255,
+0.125490196078431, +0.074509803921569, +0.094117647058824, -0.019607843137255,
+0.156862745098039, +0.152941176470588, +0.000000000000000, +0.043137254901961,
+0.231372549019608, +0.145098039215686, -0.070588235294118, +0.105882352941176,
+0.243137254901961, +0.145098039215686, +0.149019607843137, +0.043137254901961,
-0.141176470588235, +0.054901960784314, +0.239215686274510, -0.031372549019608,
-0.184313725490196, +0.098039215686275, +0.105882352941176, -0.047058823529412,
+0.149019607843137, +0.149019607843137, -0.074509803921569, -0.015686274509804,
+0.094117647058824, -0.070588235294118, +0.011764705882353, +0.141176470588235,
+0.047058823529412, -0.043137254901961, +0.062745098039216, -0.078431372549020,
-0.086274509803922, +0.184313725490196, +0.003921568627451, -0.054901960784314,
+0.129411764705882, +0.003921568627451, -0.054901960784314, +0.109803921568627,
+0.152941176470588, +0.023529411764706, +0.054901960784314, +0.078431372549020,
-0.031372549019608, +0.133333333333333, +0.168627450980392, +0.031372549019608,
-0.035294117647059, -0.062745098039216, -0.188235294117647, -0.156862745098039,
-0.070588235294118, -0.074509803921569, -0.113725490196078, -0.188235294117647,
+0.027450980392157, +0.082352941176471, -0.141176470588235, -0.062745098039216,
+0.000000000000000, -0.105882352941176, -0.149019607843137, -0.125490196078431,
-0.215686274509804, -0.105882352941176, +0.015686274509804, -0.168627450980392,
-0.129411764705882, -0.011764705882353, -0.145098039215686, -0.219607843137255,
+0.015686274509804, +0.023529411764706, -0.192156862745098, -0.200000000000000,
-0.023529411764706, -0.050980392156863, -0.105882352941176, -0.031372549019608,
-0.098039215686275, -0.113725490196078, -0.054901960784314, -0.200000000000000,
-0.086274509803922, +0.078431372549020, -0.027450980392157, -0.086274509803922,
-0.094117647058824, +0.000000000000000, -0.098039215686275, -0.054901960784314,
+0.050980392156863, -0.105882352941176, -0.074509803921569, -0.007843137254902,
-0.152941176470588, -0.129411764705882, -0.043137254901961, -0.113725490196078,
-0.152941176470588, -0.215686274509804, -0.070588235294118, -0.043137254901961,
-0.168627450980392, -0.133333333333333, -0.188235294117647, -0.098039215686275,
-0.011764705882353, -0.215686274509804, -0.200000000000000, +0.015686274509804,
-0.035294117647059, -0.200000000000000, -0.117647058823529, -0.047058823529412,
-0.125490196078431, -0.019607843137255, +0.000000000000000, -0.164705882352941,
-0.105882352941176, +0.027450980392157, -0.168627450980392, -0.105882352941176,
+0.062745098039216, +0.101960784313725, +0.070588235294118, -0.149019607843137,
-0.019607843137255, +0.113725490196078, -0.062745098039216, -0.149019607843137,
+0.054901960784314, +0.137254901960784, -0.039215686274510, -0.015686274509804,
+0.031372549019608, +0.054901960784314, +0.192156862745098, +0.078431372549020,
-0.121568627450980, +0.007843137254902, +0.184313725490196, -0.035294117647059,
-0.145098039215686, +0.062745098039216, +0.113725490196078, +0.000000000000000,
+0.105882352941176, +0.227450980392157, +0.094117647058824, +0.129411764705882,
+0.215686274509804, +0.101960784313725, +0.133333333333333, +0.200000000000000,
+0.094117647058824, +0.050980392156863, +0.215686274509804, +0.086274509803922,
+0.003921568627451, +0.172549019607843, +0.082352941176471, +0.039215686274510,
+0.184313725490196, +0.129411764705882, +0.043137254901961, +0.145098039215686,
+0.117647058823529, +0.050980392156863, +0.145098039215686, +0.164705882352941,
+0.058823529411765, +0.137254901960784, +0.168627450980392, +0.125490196078431,
+0.200000000000000, +0.239215686274510, +0.066666666666667, +0.074509803921569,
+0.172549019607843, +0.031372549019608, -0.043137254901961, -0.043137254901961,
+0.141176470588235, +0.243137254901961, +0.062745098039216, +0.117647058823529,
+0.219607843137255, +0.196078431372549, +0.156862745098039, +0.160784313725490,
+0.011764705882353, +0.031372549019608, +0.239215686274510, +0.011764705882353,
+0.007843137254902, +0.239215686274510, +0.039215686274510, -0.133333333333333,
+0.137254901960784, +0.250980392156863, +0.066666666666667, +0.000000000000000,
+0.152941176470588, +0.121568627450980, +0.078431372549020, +0.203921568627451,
+0.160784313725490, +0.149019607843137, +0.207843137254902, +0.023529411764706,
+0.043137254901961, +0.227450980392157, +0.149019607843137, +0.074509803921569,
+0.058823529411765, +0.168627450980392, +0.094117647058824, +0.070588235294118,
+0.188235294117647, +0.066666666666667, +0.109803921568627, +0.211764705882353,
+0.105882352941176, +0.113725490196078, +0.188235294117647, +0.188235294117647,
+0.180392156862745, +0.031372549019608, +0.070588235294118, +0.180392156862745,
+0.117647058823529, +0.098039215686275, +0.031372549019608, +0.086274509803922,
+0.211764705882353, +0.035294117647059, -0.062745098039216, +0.129411764705882,
+0.231372549019608, +0.078431372549020, +0.074509803921569, +0.164705882352941,
+0.137254901960784, +0.200000000000000, +0.192156862745098, +0.062745098039216,
+0.082352941176471, +0.258823529411765, +0.082352941176471, +0.082352941176471,
-0.058823529411765, +0.047058823529412, +0.113725490196078, -0.176470588235294,
-0.219607843137255, +0.058823529411765, +0.043137254901961, -0.168627450980392,
-0.070588235294118, -0.043137254901961, -0.184313725490196, -0.070588235294118,
-0.121568627450980, -0.219607843137255, -0.003921568627451, -0.035294117647059,
-0.200000000000000, -0.011764705882353, +0.180392156862745, -0.105882352941176,
-0.141176470588235, +0.003921568627451, -0.023529411764706, -0.117647058823529,
-0.086274509803922, -0.019607843137255, -0.098039215686275, -0.011764705882353,
-0.031372549019608, -0.152941176470588, -0.149019607843137, -0.043137254901961,
-0.137254901960784, -0.180392156862745, +0.054901960784314, +0.035294117647059,
-0.019607843137255, +0.050980392156863, -0.082352941176471, -0.125490196078431,
-0.015686274509804, -0.054901960784314, -0.133333333333333, -0.043137254901961,
-0.152941176470588, -0.215686274509804, -0.011764705882353, +0.011764705882353,
-0.109803921568627, -0.062745098039216, +0.007843137254902, -0.109803921568627,
-0.047058823529412, +0.074509803921569, -0.062745098039216, -0.031372549019608,
+0.231372549019608, +0.125490196078431, -0.164705882352941, -0.125490196078431,
+0.070588235294118, +0.090196078431373, -0.043137254901961, -0.050980392156863,
-0.023529411764706, +0.019607843137255, +0.117647058823529, +0.133333333333333,
-0.027450980392157, -0.050980392156863, +0.192156862745098, -0.007843137254902,
-0.094117647058824, +0.207843137254902, +0.047058823529412, -0.121568627450980,
+0.090196078431373, +0.090196078431373, -0.043137254901961, -0.003921568627451,
+0.117647058823529, +0.031372549019608, -0.043137254901961, +0.058823529411765,
+0.003921568627451, +0.019607843137255, +0.094117647058824, -0.043137254901961,
-0.137254901960784, +0.000000000000000, -0.058823529411765, -0.125490196078431,
-0.011764705882353, +0.058823529411765, +0.027450980392157, +0.039215686274510,
-0.019607843137255, -0.152941176470588, -0.035294117647059, +0.019607843137255,
-0.047058823529412, -0.027450980392157, +0.066666666666667, +0.086274509803922,
+0.031372549019608, +0.023529411764706, -0.043137254901961, -0.058823529411765,
+0.043137254901961, +0.031372549019608, -0.082352941176471, +0.003921568627451,
+0.188235294117647, +0.050980392156863, -0.098039215686275, +0.000000000000000,
+0.094117647058824, +0.011764705882353, +0.000000000000000, +0.003921568627451,
-0.043137254901961, +0.035294117647059, +0.105882352941176, -0.003921568627451,
-0.109803921568627, +0.019607843137255, +0.031372549019608, +0.054901960784314,
-0.019607843137255, +0.066666666666667, +0.231372549019608, +0.043137254901961,
-0.066666666666667, +0.109803921568627, +0.168627450980392, -0.054901960784314,
+0.050980392156863, +0.094117647058824, -0.137254901960784, +0.039215686274510,
+0.094117647058824, -0.109803921568627, +0.070588235294118, +0.090196078431373,
-0.094117647058824, +0.043137254901961, +0.270588235294118, -0.023529411764706,
-0.133333333333333, +0.094117647058824, +0.066666666666667, -0.062745098039216,
+0.000000000000000, +0.007843137254902, -0.105882352941176, +0.043137254901961,
+0.039215686274510, -0.070588235294118, -0.039215686274510, +0.098039215686275,
+0.058823529411765, -0.062745098039216, +0.015686274509804, -0.062745098039216,
-0.090196078431373, +0.101960784313725, -0.019607843137255, -0.125490196078431,
+0.050980392156863, +0.003921568627451, -0.043137254901961, +0.137254901960784,
+0.019607843137255, -0.223529411764706, -0.101960784313725, -0.031372549019608,
-0.156862745098039, -0.066666666666667, +0.082352941176471, -0.113725490196078,
-0.098039215686275, +0.054901960784314, -0.129411764705882, -0.219607843137255,
+0.074509803921569, +0.219607843137255, -0.086274509803922, -0.160784313725490,
+0.011764705882353, -0.050980392156863, -0.203921568627451, -0.129411764705882,
-0.137254901960784, -0.200000000000000, -0.031372549019608, +0.035294117647059,
-0.129411764705882, -0.125490196078431, +0.074509803921569, -0.125490196078431,
-0.137254901960784, +0.086274509803922, -0.105882352941176, -0.250980392156863,
-0.019607843137255, -0.011764705882353, -0.231372549019608, -0.117647058823529,
+0.019607843137255, -0.137254901960784, -0.184313725490196, -0.019607843137255,
-0.113725490196078, -0.188235294117647, -0.074509803921569, -0.090196078431373,
-0.168627450980392, +0.000000000000000, -0.031372549019608, -0.176470588235294,
-0.094117647058824, -0.149019607843137, -0.211764705882353, -0.070588235294118,
-0.054901960784314, -0.231372549019608, -0.101960784313725, -0.015686274509804,
-0.137254901960784, -0.254901960784314, -0.152941176470588, -0.011764705882353,
-0.133333333333333, +0.011764705882353, +0.086274509803922, -0.149019607843137,
-0.168627450980392, -0.031372549019608, -0.149019607843137, -0.176470588235294,
+0.000000000000000, -0.113725490196078, -0.231372549019608, -0.094117647058824,
-0.039215686274510, -0.184313725490196, -0.149019607843137, -0.117647058823529,
-0.247058823529412, -0.219607843137255, -0.043137254901961, -0.062745098039216,
-0.211764705882353, -0.180392156862745, -0.168627450980392, -0.125490196078431,
+0.027450980392157, +0.043137254901961, +0.160784313725490, +0.003921568627451,
-0.086274509803922, +0.047058823529412, +0.094117647058824, -0.117647058823529,
-0.054901960784314, +0.082352941176471, -0.062745098039216, +0.090196078431373,
+0.180392156862745, -0.094117647058824, +0.031372549019608, +0.125490196078431,
-0.113725490196078, -0.074509803921569, +0.149019607843137, -0.047058823529412,
-0.231372549019608, +0.043137254901961, +0.054901960784314, -0.137254901960784,
+0.031372549019608, +0.074509803921569, -0.098039215686275, +0.062745098039216,
+0.180392156862745, +0.035294117647059, +0.035294117647059, +0.133333333333333,
+0.050980392156863, -0.019607843137255, +0.058823529411765, -0.031372549019608,
-0.152941176470588, +0.082352941176471, +0.074509803921569, -0.094117647058824,
+0.058823529411765, +0.031372549019608, -0.043137254901961, +0.164705882352941,
+0.192156862745098, +0.003921568627451, +0.086274509803922, +0.094117647058824,
-0.094117647058824, -0.027450980392157, +0.160784313725490, +0.019607843137255,
-0.023529411764706, +0.219607843137255, +0.050980392156863, -0.168627450980392,
+0.003921568627451, +0.156862745098039, -0.058823529411765, -0.180392156862745,
+0.074509803921569, +0.172549019607843, +0.000000000000000, +0.062745098039216,
+0.050980392156863, -0.172549019607843, -0.023529411764706, +0.168627450980392,
-0.007843137254902, +0.031372549019608, +0.196078431372549, -0.066666666666667,
-0.023529411764706, +0.200000000000000, +0.027450980392157, -0.235294117647059,
-0.054901960784314, +0.125490196078431, -0.105882352941176, -0.043137254901961,
+0.149019607843137, +0.003921568627451, -0.082352941176471, +0.117647058823529,
+0.101960784313725, +0.023529411764706, +0.125490196078431, +0.137254901960784,
+0.043137254901961, +0.145098039215686, +0.145098039215686, +0.062745098039216,
+0.125490196078431, +0.090196078431373, +0.043137254901961, +0.109803921568627,
+0.168627450980392, -0.066666666666667, -0.039215686274510, +0.203921568627451,
+0.192156862745098, +0.019607843137255, -0.011764705882353, +0.129411764705882,
+0.027450980392157, +0.066666666666667, +0.270588235294118, +0.200000000000000,
+0.070588235294118, +0.176470588235294, +0.105882352941176, +0.015686274509804,
+0.168627450980392, +0.125490196078431, -0.011764705882353, +0.070588235294118,
+0.168627450980392, +0.066666666666667, +0.113725490196078, +0.152941176470588,
+0.015686274509804, -0.035294117647059, +0.145098039215686, +0.223529411764706,
+0.125490196078431, +0.113725490196078, +0.062745098039216, +0.070588235294118,
+0.003921568627451, -0.019607843137255, +0.062745098039216, +0.007843137254902,
-0.176470588235294, -0.023529411764706, +0.168627450980392, +0.062745098039216,
+0.003921568627451, -0.043137254901961, -0.129411764705882, +0.062745098039216,
+0.239215686274510, -0.011764705882353, -0.023529411764706, +0.117647058823529,
-0.113725490196078, -0.109803921568627, +0.137254901960784, +0.043137254901961,
-0.137254901960784, +0.050980392156863, +0.039215686274510, -0.227450980392157,
-0.039215686274510, +0.074509803921569, -0.125490196078431, -0.054901960784314,
+0.101960784313725, -0.129411764705882, -0.168627450980392, +0.003921568627451,
-0.082352941176471, -0.145098039215686, +0.007843137254902, +0.015686274509804,
-0.200000000000000, -0.054901960784314, -0.007843137254902, -0.184313725490196,
-0.113725490196078, -0.160784313725490, -0.215686274509804, -0.019607843137255,
+0.011764705882353, -0.074509803921569, +0.137254901960784, +0.168627450980392,
-0.098039215686275, -0.090196078431373, +0.066666666666667, -0.047058823529412,
-0.164705882352941, +0.125490196078431, +0.062745098039216, -0.223529411764706,
-0.043137254901961, +0.050980392156863, -0.105882352941176, -0.219607843137255,
-0.011764705882353, +0.156862745098039, -0.023529411764706, +0.050980392156863,
+0.094117647058824, -0.196078431372549, -0.184313725490196, -0.019607843137255,
-0.117647058823529, -0.086274509803922, +0.105882352941176, -0.074509803921569,
-0.164705882352941, +0.039215686274510, +0.086274509803922, -0.098039215686275,
-0.043137254901961, +0.058823529411765, -0.137254901960784, -0.152941176470588,
+0.050980392156863, -0.011764705882353, -0.141176470588235, -0.043137254901961,
-0.050980392156863, -0.043137254901961, +0.062745098039216, +0.019607843137255,
-0.078431372549020, -0.019607843137255, -0.050980392156863, -0.062745098039216,
+0.039215686274510, +0.094117647058824, +0.043137254901961, +0.003921568627451,
+0.117647058823529, -0.066666666666667, -0.203921568627451, -0.015686274509804,
+0.019607843137255, -0.007843137254902, +0.000000000000000, -0.035294117647059,
-0.137254901960784, -0.141176470588235, -0.027450980392157, +0.039215686274510,
+0.000000000000000, +0.000000000000000, -0.058823529411765, -0.050980392156863,
+0.117647058823529, +0.078431372549020, -0.074509803921569, +0.015686274509804,
+0.086274509803922, +0.007843137254902, +0.094117647058824, +0.176470588235294,
+0.031372549019608, -0.003921568627451, +0.121568627450980, +0.094117647058824,
+0.003921568627451, +0.031372549019608, +0.086274509803922, +0.121568627450980,
-0.062745098039216, -0.054901960784314, +0.015686274509804, +0.101960784313725,
-0.094117647058824, -0.050980392156863, +0.062745098039216, +0.086274509803922,
+0.219607843137255, +0.113725490196078, -0.058823529411765, +0.019607843137255,
+0.176470588235294, +0.031372549019608, -0.007843137254902, +0.207843137254902,
+0.082352941176471, +0.003921568627451, +0.176470588235294, +0.133333333333333,
-0.023529411764706, +0.133333333333333, +0.141176470588235, -0.117647058823529,
+0.019607843137255, +0.164705882352941, +0.015686274509804, +0.027450980392157,
+0.211764705882353, +0.062745098039216, -0.027450980392157, +0.149019607843137,
+0.125490196078431, -0.007843137254902, +0.082352941176471, +0.149019607843137,
-0.058823529411765, +0.054901960784314, +0.160784313725490, +0.062745098039216,
+0.129411764705882, +0.007843137254902, -0.129411764705882, +0.043137254901961,
-0.023529411764706, -0.223529411764706, -0.015686274509804, +0.207843137254902,
-0.031372549019608, -0.105882352941176, +0.117647058823529, +0.019607843137255,
-0.133333333333333, +0.113725490196078, +0.078431372549020, -0.200000000000000,
+0.039215686274510, +0.231372549019608, +0.149019607843137, -0.003921568627451,
+0.000000000000000, +0.101960784313725, -0.145098039215686, -0.050980392156863,
+0.156862745098039, -0.054901960784314, -0.105882352941176, +0.023529411764706,
+0.019607843137255, -0.039215686274510, +0.062745098039216, +0.043137254901961,
-0.023529411764706, +0.062745098039216, +0.035294117647059, -0.070588235294118,
+0.070588235294118, +0.133333333333333, -0.007843137254902, -0.152941176470588,
-0.023529411764706, +0.007843137254902, -0.117647058823529, -0.062745098039216,
-0.160784313725490, -0.270588235294118, -0.078431372549020, -0.047058823529412,
-0.121568627450980, -0.003921568627451, -0.082352941176471, -0.188235294117647,
-0.219607843137255, -0.109803921568627, -0.082352941176471, -0.113725490196078,
+0.062745098039216, +0.023529411764706, -0.137254901960784, -0.086274509803922,
-0.125490196078431, -0.203921568627451, -0.019607843137255, -0.019607843137255,
-0.203921568627451, -0.196078431372549, -0.125490196078431, -0.152941176470588,
-0.105882352941176, -0.019607843137255, -0.125490196078431, -0.231372549019608,
-0.043137254901961, -0.031372549019608, -0.239215686274510, -0.113725490196078,
-0.062745098039216, -0.223529411764706, -0.211764705882353, -0.039215686274510,
-0.137254901960784, -0.168627450980392, +0.043137254901961, -0.039215686274510,
-0.200000000000000, -0.231372549019608, -0.156862745098039, -0.094117647058824,
-0.070588235294118, -0.086274509803922, -0.031372549019608, -0.054901960784314,
-0.188235294117647, -0.137254901960784, -0.145098039215686, -0.223529411764706,
-0.086274509803922, -0.066666666666667, -0.176470588235294, -0.184313725490196,
-0.015686274509804, -0.098039215686275, -0.149019607843137, +0.000000000000000,
-0.098039215686275, -0.180392156862745, -0.019607843137255, +0.003921568627451,
-0.152941176470588, +0.058823529411765, +0.117647058823529, -0.168627450980392,
-0.090196078431373, +0.031372549019608, -0.066666666666667, -0.043137254901961,
+0.027450980392157, -0.074509803921569, -0.094117647058824, +0.031372549019608,
+0.050980392156863, -0.074509803921569, -0.031372549019608, +0.007843137254902,
-0.133333333333333, -0.023529411764706, +0.054901960784314, -0.007843137254902,
+0.160784313725490, +0.117647058823529, -0.117647058823529, +0.105882352941176,
+0.137254901960784, -0.160784313725490, -0.086274509803922, +0.192156862745098,
+0.074509803921569, -0.062745098039216, +0.172549019607843, +0.058823529411765,
-0.066666666666667, +0.180392156862745, +0.145098039215686, -0.094117647058824,
-0.019607843137255, +0.070588235294118, -0.015686274509804, -0.094117647058824,
-0.031372549019608, +0.180392156862745, -0.047058823529412, -0.117647058823529,
+0.207843137254902, +0.184313725490196, +0.105882352941176, +0.156862745098039,
+0.145098039215686, +0.105882352941176, +0.168627450980392, +0.156862745098039,
+0.137254901960784, +0.262745098039216, +0.184313725490196, +0.000000000000000,
+0.101960784313725, +0.227450980392157, +0.180392156862745, -0.039215686274510,
+0.031372549019608, +0.207843137254902, +0.156862745098039, +0.239215686274510,
+0.117647058823529, -0.109803921568627, +0.050980392156863, +0.133333333333333,
+0.047058823529412, +0.200000000000000, +0.113725490196078, +0.043137254901961,
+0.070588235294118, +0.094117647058824, +0.129411764705882, +0.094117647058824,
+0.180392156862745, +0.180392156862745, +0.113725490196078, +0.156862745098039,
+0.137254901960784, -0.121568627450980, +0.039215686274510, +0.278431372549020,
+0.129411764705882, +0.074509803921569, +0.133333333333333, +0.074509803921569,
+0.070588235294118, +0.196078431372549, +0.168627450980392, -0.015686274509804,
+0.113725490196078, +0.141176470588235, -0.086274509803922, +0.062745098039216,
+0.176470588235294, +0.054901960784314, -0.027450980392157, +0.098039215686275,
-0.054901960784314, -0.188235294117647, +0.137254901960784, +0.215686274509804,
+0.050980392156863, -0.023529411764706, +0.019607843137255, +0.043137254901961,
+0.050980392156863, +0.011764705882353, +0.152941176470588, +0.078431372549020,
+0.027450980392157, +0.058823529411765, +0.019607843137255, -0.125490196078431,
-0.090196078431373, -0.062745098039216, -0.145098039215686, -0.219607843137255,
+0.011764705882353, -0.011764705882353, -0.133333333333333, -0.011764705882353,
-0.125490196078431, -0.254901960784314, -0.054901960784314, +0.039215686274510,
-0.188235294117647, -0.050980392156863, +0.039215686274510, -0.207843137254902,
-0.176470588235294, -0.007843137254902, -0.113725490196078, -0.054901960784314,
-0.058823529411765, -0.243137254901961, -0.258823529411765, -0.121568627450980,
-0.023529411764706, -0.109803921568627, -0.082352941176471, -0.074509803921569,
-0.180392156862745, -0.129411764705882, -0.086274509803922, -0.168627450980392,
-0.039215686274510, +0.035294117647059, -0.180392156862745, +0.007843137254902,
+0.149019607843137, -0.156862745098039, -0.137254901960784, +0.054901960784314,
-0.062745098039216, -0.156862745098039, +0.094117647058824, -0.043137254901961,
-0.176470588235294, +0.054901960784314, +0.000000000000000, -0.145098039215686,
-0.137254901960784, -0.105882352941176, -0.188235294117647, -0.250980392156863,
-0.145098039215686, +0.086274509803922, -0.031372549019608, -0.243137254901961,
-0.125490196078431, -0.050980392156863, -0.050980392156863, -0.074509803921569,
-0.160784313725490, -0.180392156862745, -0.066666666666667, -0.090196078431373,
-0.141176470588235, +0.023529411764706, +0.039215686274510, -0.141176470588235,
-0.129411764705882, +0.054901960784314, +0.070588235294118, -0.082352941176471,
-0.047058823529412, +0.019607843137255, -0.054901960784314, +0.109803921568627,
+0.101960784313725, -0.168627450980392, -0.050980392156863, +0.043137254901961,
-0.019607843137255, +0.192156862745098, +0.086274509803922, -0.082352941176471,
+0.078431372549020, +0.094117647058824, +0.015686274509804, -0.062745098039216,
+0.035294117647059, +0.000000000000000, -0.054901960784314, +0.031372549019608,
+0.137254901960784, -0.019607843137255, -0.070588235294118, +0.145098039215686,
+0.054901960784314, -0.074509803921569, +0.011764705882353, +0.082352941176471,
+0.019607843137255, +0.003921568627451, -0.019607843137255, -0.043137254901961,
+0.137254901960784, +0.141176470588235, -0.054901960784314, +0.070588235294118,
+0.188235294117647, +0.086274509803922, +0.054901960784314, +0.207843137254902,
+0.003921568627451, -0.188235294117647, +0.074509803921569, +0.137254901960784,
-0.031372549019608, -0.011764705882353, +0.043137254901961, -0.031372549019608,
+0.043137254901961, +0.027450980392157, +0.192156862745098, +0.137254901960784,
+0.117647058823529, +0.211764705882353, +0.207843137254902, +0.105882352941176,
+0.180392156862745, +0.176470588235294, +0.058823529411765, -0.027450980392157,
+0.125490196078431, +0.145098039215686, +0.054901960784314, +0.219607843137255,
+0.090196078431373, -0.121568627450980, +0.090196078431373, +0.192156862745098,
+0.003921568627451, +0.062745098039216, +0.168627450980392, -0.043137254901961,
-0.090196078431373, +0.105882352941176, +0.011764705882353, +0.117647058823529,
+0.243137254901961, +0.113725490196078, -0.007843137254902, +0.031372549019608,
+0.180392156862745, +0.172549019607843, +0.180392156862745, +0.137254901960784,
+0.078431372549020, +0.121568627450980, +0.113725490196078, +0.058823529411765,
+0.113725490196078, +0.168627450980392, -0.007843137254902, +0.043137254901961,
+0.184313725490196, -0.035294117647059, -0.019607843137255, +0.145098039215686,
+0.023529411764706, -0.094117647058824, +0.133333333333333, +0.066666666666667,
-0.043137254901961, +0.137254901960784, +0.074509803921569, +0.019607843137255,
+0.101960784313725, +0.203921568627451, +0.094117647058824, -0.113725490196078,
-0.003921568627451, +0.156862745098039, +0.113725490196078, +0.027450980392157,
-0.105882352941176, -0.176470588235294, -0.039215686274510, +0.023529411764706,
-0.058823529411765, -0.129411764705882, -0.058823529411765, -0.101960784313725,
-0.223529411764706, -0.007843137254902, +0.074509803921569, -0.152941176470588,
-0.223529411764706, +0.062745098039216, +0.113725490196078, -0.007843137254902,
+0.015686274509804, -0.031372549019608, -0.133333333333333, +0.035294117647059,
+0.156862745098039, -0.113725490196078, -0.125490196078431, -0.101960784313725,
-0.207843137254902, +0.082352941176471, +0.113725490196078, -0.192156862745098,
-0.105882352941176, +0.027450980392157, -0.101960784313725, -0.176470588235294,
+0.043137254901961, -0.035294117647059, -0.160784313725490, -0.023529411764706,
+0.105882352941176, +0.023529411764706, -0.184313725490196, -0.101960784313725,
-0.086274509803922, -0.231372549019608, -0.149019607843137, -0.011764705882353,
-0.019607843137255, +0.011764705882353, -0.109803921568627, -0.137254901960784,
+0.050980392156863, +0.000000000000000, -0.180392156862745, -0.125490196078431,
+0.015686274509804, -0.082352941176471, -0.070588235294118, +0.168627450980392,
+0.011764705882353, -0.149019607843137, +0.094117647058824, +0.094117647058824,
-0.156862745098039, -0.074509803921569, +0.031372549019608, -0.137254901960784,
-0.156862745098039, -0.215686274509804, -0.043137254901961, -0.023529411764706,
-0.152941176470588, -0.074509803921569, -0.031372549019608, -0.086274509803922,
+0.086274509803922, +0.086274509803922, -0.125490196078431, -0.149019607843137,
-0.070588235294118, -0.062745098039216, -0.035294117647059, +0.211764705882353,
+0.054901960784314, -0.184313725490196, +0.031372549019608, +0.058823529411765,
-0.066666666666667, +0.015686274509804, +0.223529411764706, +0.000000000000000,
-0.109803921568627, +0.109803921568627, -0.023529411764706, -0.011764705882353,
+0.137254901960784, +0.062745098039216, -0.011764705882353, +0.031372549019608,
+0.027450980392157, -0.019607843137255, +0.078431372549020, +0.109803921568627,
+0.019607843137255, +0.039215686274510, -0.007843137254902, -0.098039215686275,
-0.031372549019608, +0.043137254901961, -0.058823529411765, -0.043137254901961,
+0.031372549019608, -0.105882352941176, -0.086274509803922, +0.113725490196078,
+0.105882352941176, -0.003921568627451, +0.062745098039216, -0.019607843137255,
-0.074509803921569, +0.145098039215686, +0.152941176470588, +0.019607843137255,
+0.082352941176471, +0.250980392156863, +0.125490196078431, -0.176470588235294,
-0.039215686274510, +0.160784313725490, +0.094117647058824, +0.145098039215686,
+0.129411764705882, -0.011764705882353, +0.047058823529412, +0.152941176470588,
+0.105882352941176, +0.035294117647059, +0.137254901960784, +0.121568627450980,
-0.082352941176471, +0.043137254901961, +0.223529411764706, +0.015686274509804,
-0.145098039215686, +0.074509803921569, +0.047058823529412, -0.062745098039216,
+0.090196078431373, +0.117647058823529, -0.023529411764706, -0.007843137254902,
+0.145098039215686, +0.039215686274510, +0.058823529411765, +0.043137254901961,
-0.176470588235294, +0.011764705882353, +0.168627450980392, -0.074509803921569,
-0.094117647058824, +0.133333333333333, -0.007843137254902, -0.172549019607843,
+0.113725490196078, +0.176470588235294, -0.027450980392157, -0.031372549019608,
+0.105882352941176, +0.164705882352941, +0.054901960784314, +0.105882352941176,
+0.105882352941176, -0.121568627450980, -0.078431372549020, +0.062745098039216,
+0.023529411764706, +0.137254901960784, +0.082352941176471, -0.007843137254902,
+0.145098039215686, +0.035294117647059, -0.137254901960784, -0.062745098039216,
+0.086274509803922, +0.031372549019608, +0.019607843137255, +0.176470588235294,
+0.011764705882353, -0.098039215686275, +0.156862745098039, +0.227450980392157,
-0.011764705882353, +0.011764705882353, +0.211764705882353, +0.125490196078431,
+0.027450980392157, -0.113725490196078, +0.011764705882353, +0.149019607843137,
+0.058823529411765, +0.043137254901961, -0.094117647058824, -0.176470588235294,
+0.098039215686275, +0.192156862745098, -0.094117647058824, -0.141176470588235,
-0.003921568627451, -0.074509803921569, -0.109803921568627, +0.168627450980392,
+0.074509803921569, -0.180392156862745, +0.011764705882353, +0.078431372549020,
-0.074509803921569, -0.082352941176471, +0.156862745098039, -0.011764705882353,
-0.156862745098039, +0.105882352941176, +0.015686274509804, -0.101960784313725,
+0.043137254901961, -0.078431372549020, -0.180392156862745, +0.027450980392157,
+0.019607843137255, -0.156862745098039, -0.090196078431373, +0.031372549019608,
-0.113725490196078, -0.086274509803922, +0.007843137254902, -0.094117647058824,
-0.074509803921569, -0.011764705882353, -0.117647058823529, -0.105882352941176,
-0.039215686274510, -0.156862745098039, -0.172549019607843, -0.058823529411765,
-0.133333333333333, -0.129411764705882, +0.011764705882353, -0.098039215686275,
-0.262745098039216, -0.086274509803922, +0.035294117647059, -0.200000000000000,
-0.219607843137255, +0.007843137254902, +0.011764705882353, -0.180392156862745,
-0.180392156862745, +0.011764705882353, -0.066666666666667, -0.168627450980392,
-0.086274509803922, -0.117647058823529, -0.133333333333333, -0.058823529411765,
-0.062745098039216, -0.149019607843137, -0.086274509803922, -0.035294117647059,
-0.145098039215686, -0.105882352941176, +0.094117647058824, -0.043137254901961,
-0.239215686274510, -0.121568627450980, -0.121568627450980, -0.207843137254902,
-0.023529411764706, +0.074509803921569, -0.129411764705882, -0.211764705882353,
-0.129411764705882, -0.145098039215686, -0.023529411764706, +0.027450980392157,
-0.156862745098039, -0.074509803921569, +0.019607843137255, -0.117647058823529,
-0.137254901960784, +0.149019607843137, +0.090196078431373, -0.184313725490196,
-0.062745098039216, +0.015686274509804, -0.117647058823529, -0.231372549019608,
-0.078431372549020, +0.039215686274510, +0.054901960784314, +0.160784313725490,
+0.086274509803922, -0.176470588235294, -0.211764705882353, +0.000000000000000,
-0.047058823529412, -0.058823529411765, -0.035294117647059, -0.141176470588235,
+0.137254901960784, +0.125490196078431, -0.164705882352941, -0.086274509803922,
-0.003921568627451, -0.113725490196078, -0.125490196078431, +0.023529411764706,
-0.098039215686275, -0.168627450980392, +0.043137254901961, +0.141176470588235,
-0.023529411764706, -0.078431372549020, +0.078431372549020, +0.058823529411765,
+0.062745098039216, -0.058823529411765, +0.031372549019608, +0.090196078431373,
+0.058823529411765, +0.231372549019608, +0.105882352941176, -0.137254901960784,
+0.043137254901961, +0.254901960784314, +0.050980392156863, -0.035294117647059,
+0.156862745098039, +0.062745098039216, -0.101960784313725, +0.058823529411765,
+0.101960784313725, -0.113725490196078, +0.019607843137255, +0.141176470588235,
+0.019607843137255, +0.011764705882353, +0.203921568627451, +0.117647058823529,
-0.054901960784314, +0.117647058823529, +0.125490196078431, +0.031372549019608,
+0.180392156862745, +0.141176470588235, -0.019607843137255, +0.137254901960784,
+0.219607843137255, +0.027450980392157, +0.003921568627451, +0.164705882352941,
+0.066666666666667, +0.074509803921569, +0.188235294117647, +0.113725490196078,
+0.121568627450980, +0.219607843137255, +0.152941176470588, +0.113725490196078,
+0.200000000000000, -0.003921568627451, -0.074509803921569, +0.141176470588235,
+0.031372549019608, -0.023529411764706, +0.168627450980392, +0.180392156862745,
-0.043137254901961, +0.000000000000000, +0.211764705882353, +0.035294117647059,
-0.094117647058824, +0.050980392156863, +0.129411764705882, +0.066666666666667,
-0.031372549019608, +0.113725490196078, +0.141176470588235, +0.003921568627451,
+0.047058823529412, +0.043137254901961, +0.007843137254902, +0.054901960784314,
+0.109803921568627, +0.054901960784314, +0.019607843137255, +0.007843137254902,
-0.113725490196078, -0.082352941176471, +0.141176470588235, +0.039215686274510,
-0.101960784313725, +0.019607843137255, +0.066666666666667, -0.070588235294118,
-0.011764705882353, +0.145098039215686, +0.035294117647059, +0.007843137254902,
+0.039215686274510, +0.007843137254902, +0.074509803921569, +0.054901960784314,
-0.094117647058824, +0.019607843137255, +0.121568627450980, +0.019607843137255,
-0.050980392156863, +0.137254901960784, +0.172549019607843, +0.015686274509804,
+0.039215686274510, +0.043137254901961, +0.019607843137255, -0.074509803921569,
-0.031372549019608, -0.121568627450980, -0.145098039215686, +0.109803921568627,
+0.141176470588235, +0.035294117647059, -0.141176470588235, -0.031372549019608,
+0.035294117647059, +0.043137254901961, +0.086274509803922, -0.105882352941176,
+0.094117647058824, +0.200000000000000, -0.168627450980392, -0.156862745098039,
+0.019607843137255, -0.074509803921569, -0.098039215686275, +0.078431372549020,
-0.011764705882353, -0.152941176470588, -0.027450980392157, +0.109803921568627,
+0.050980392156863, -0.035294117647059, +0.000000000000000, -0.101960784313725,
-0.050980392156863, -0.160784313725490, +0.000000000000000, -0.043137254901961,
-0.211764705882353, +0.043137254901961, +0.105882352941176, -0.176470588235294,
-0.086274509803922, +0.141176470588235, -0.043137254901961, -0.125490196078431,
+0.054901960784314, -0.023529411764706, -0.125490196078431, -0.066666666666667,
+0.027450980392157, -0.117647058823529, -0.066666666666667, +0.000000000000000,
-0.207843137254902, -0.125490196078431, +0.070588235294118, +0.019607843137255,
-0.137254901960784, -0.062745098039216, -0.074509803921569, -0.105882352941176,
+0.043137254901961, +0.007843137254902, -0.105882352941176, +0.000000000000000,
+0.082352941176471, -0.105882352941176, -0.094117647058824, +0.062745098039216,
-0.027450980392157, +0.000000000000000, +0.031372549019608, -0.090196078431373,
-0.050980392156863, +0.086274509803922, +0.031372549019608, -0.031372549019608,
+0.200000000000000, +0.031372549019608, -0.168627450980392, +0.105882352941176,
+0.090196078431373, -0.011764705882353, +0.105882352941176, +0.156862745098039,
-0.007843137254902, -0.007843137254902, +0.149019607843137, +0.164705882352941,
+0.054901960784314, +0.062745098039216, +0.066666666666667, -0.050980392156863,
-0.058823529411765, +0.117647058823529, +0.168627450980392, +0.094117647058824,
+0.156862745098039, +0.188235294117647, +0.149019607843137, +0.156862745098039,
+0.129411764705882, +0.090196078431373, +0.160784313725490, +0.211764705882353,
+0.007843137254902, -0.015686274509804, +0.231372549019608, +0.184313725490196,
+0.117647058823529, +0.196078431372549, +0.152941176470588, -0.015686274509804,
+0.031372549019608, +0.196078431372549, +0.117647058823529, +0.145098039215686,
+0.188235294117647, +0.149019607843137, +0.219607843137255, +0.219607843137255,
+0.058823529411765, +0.070588235294118, +0.188235294117647, +0.074509803921569,
-0.023529411764706, +0.082352941176471, +0.090196078431373, +0.003921568627451,
+0.098039215686275, +0.180392156862745, +0.156862745098039, +0.113725490196078,
+0.207843137254902, +0.105882352941176, -0.007843137254902, +0.160784313725490,
+0.145098039215686, +0.094117647058824, +0.054901960784314, +0.050980392156863,
+0.070588235294118, +0.125490196078431, +0.180392156862745, +0.054901960784314,
+0.109803921568627, +0.294117647058824, +0.109803921568627, +0.023529411764706,
+0.200000000000000, +0.125490196078431, +0.031372549019608, +0.160784313725490,
+0.125490196078431, -0.027450980392157, -0.027450980392157, +0.062745098039216,
+0.101960784313725, +0.082352941176471, +0.141176470588235, +0.047058823529412,
+0.137254901960784, -0.019607843137255, +0.149019607843137, +0.184313725490196,
-0.043137254901961, +0.050980392156863, +0.196078431372549, +0.000000000000000,
+0.023529411764706, +0.219607843137255, -0.003921568627451, -0.094117647058824,
+0.149019607843137, +0.121568627450980, +0.070588235294118, +0.082352941176471,
+0.121568627450980, +0.027450980392157, +0.050980392156863, +0.176470588235294,
-0.082352941176471, -0.125490196078431, +0.047058823529412, -0.019607843137255,
-0.184313725490196, -0.050980392156863, -0.070588235294118, -0.141176470588235,
+0.035294117647059, -0.043137254901961, -0.250980392156863, -0.117647058823529,
+0.027450980392157, -0.176470588235294, -0.117647058823529, +0.015686274509804,
-0.168627450980392, -0.105882352941176, +0.050980392156863, -0.113725490196078,
-0.137254901960784, +0.011764705882353, -0.070588235294118, -0.168627450980392,
+0.109803921568627, +0.062745098039216, -0.207843137254902, -0.039215686274510,
-0.062745098039216, -0.203921568627451, -0.043137254901961, +0.019607843137255,
-0.192156862745098, -0.188235294117647, -0.105882352941176, -0.062745098039216,
-0.117647058823529, -0.054901960784314, +0.043137254901961, -0.192156862745098,
-0.254901960784314, -0.125490196078431, -0.113725490196078, -0.188235294117647,
-0.113725490196078, -0.062745098039216, -0.094117647058824, -0.039215686274510,
-0.113725490196078, -0.243137254901961, -0.054901960784314, +0.109803921568627,
-0.137254901960784, -0.160784313725490, +0.050980392156863, -0.031372549019608,
-0.149019607843137, -0.043137254901961, +0.011764705882353, -0.160784313725490,
-0.145098039215686, -0.035294117647059, -0.164705882352941, -0.141176470588235,
-0.062745098039216, -0.125490196078431, -0.050980392156863, +0.019607843137255,
-0.101960784313725, -0.180392156862745, -0.047058823529412, -0.105882352941176,
-0.188235294117647, -0.003921568627451, -0.039215686274510, -0.200000000000000,
-0.192156862745098, -0.133333333333333, -0.164705882352941, -0.125490196078431,
-0.047058823529412, -0.098039215686275, -0.168627450980392, +0.015686274509804,
+0.043137254901961, -0.156862745098039, -0.149019607843137, -0.145098039215686,
-0.200000000000000, -0.086274509803922, -0.070588235294118, -0.168627450980392,
-0.070588235294118, +0.043137254901961, -0.047058823529412, -0.121568627450980,
+0.050980392156863, -0.019607843137255, -0.152941176470588, -0.011764705882353,
-0.039215686274510, -0.129411764705882, -0.188235294117647, -0.219607843137255,
-0.156862745098039, -0.050980392156863, -0.007843137254902, -0.101960784313725,
-0.074509803921569, -0.149019607843137, -0.039215686274510, +0.043137254901961,
-0.011764705882353, +0.007843137254902, +0.050980392156863, -0.043137254901961,
-0.066666666666667, +0.129411764705882, -0.019607843137255, -0.168627450980392,
+0.086274509803922, +0.078431372549020, -0.011764705882353, +0.054901960784314,
+0.058823529411765, -0.039215686274510, +0.031372549019608, +0.168627450980392,
+0.015686274509804, -0.082352941176471, +0.137254901960784, +0.129411764705882,
-0.066666666666667, +0.090196078431373, +0.070588235294118, -0.133333333333333,
+0.054901960784314, +0.168627450980392, -0.054901960784314, +0.031372549019608,
+0.188235294117647, -0.113725490196078, -0.074509803921569, +0.196078431372549,
+0.011764705882353, -0.050980392156863, +0.156862745098039, +0.074509803921569,
+0.007843137254902, +0.152941176470588, +0.098039215686275, -0.019607843137255,
+0.168627450980392, +0.180392156862745, -0.027450980392157, +0.094117647058824,
+0.137254901960784, -0.058823529411765, +0.039215686274510, +0.200000000000000,
+0.031372549019608, +0.007843137254902, +0.082352941176471, -0.019607843137255,
-0.137254901960784, +0.023529411764706, +0.266666666666667, +0.074509803921569,
-0.078431372549020, +0.011764705882353, -0.027450980392157, -0.145098039215686,
-0.031372549019608, -0.062745098039216, -0.074509803921569, +0.109803921568627,
+0.062745098039216, -0.145098039215686, -0.011764705882353, +0.184313725490196,
-0.090196078431373, -0.137254901960784, +0.043137254901961, -0.066666666666667,
-0.227450980392157, -0.058823529411765, +0.105882352941176, -0.011764705882353,
+0.019607843137255, +0.058823529411765, -0.125490196078431, -0.109803921568627,
+0.000000000000000, -0.156862745098039, -0.094117647058824, +0.054901960784314,
-0.015686274509804, -0.062745098039216, +0.074509803921569, +0.000000000000000,
-0.156862745098039, +0.086274509803922, +0.117647058823529, -0.031372549019608,
-0.019607843137255, -0.011764705882353, -0.105882352941176, -0.023529411764706,
+0.031372549019608, -0.047058823529412, -0.152941176470588, +0.003921568627451,
+0.192156862745098, -0.043137254901961, -0.098039215686275, -0.011764705882353,
-0.141176470588235, -0.039215686274510, +0.015686274509804, -0.172549019607843,
-0.047058823529412, +0.035294117647059, -0.066666666666667, -0.145098039215686,
+0.039215686274510, +0.039215686274510, -0.125490196078431, +0.000000000000000,
-0.054901960784314, -0.094117647058824, -0.011764705882353, -0.082352941176471,
-0.156862745098039, -0.007843137254902, +0.047058823529412, -0.078431372549020,
-0.125490196078431, -0.164705882352941, -0.066666666666667, -0.109803921568627,
-0.105882352941176, -0.011764705882353, -0.074509803921569, -0.125490196078431,
-0.188235294117647, +0.027450980392157, +0.003921568627451, -0.231372549019608,
-0.015686274509804, -0.019607843137255, -0.203921568627451, -0.129411764705882,
-0.125490196078431, -0.254901960784314, -0.141176470588235, -0.043137254901961,
-0.160784313725490, -0.243137254901961, +0.011764705882353, +0.062745098039216,
-0.160784313725490, -0.019607843137255, +0.011764705882353, -0.207843137254902,
-0.156862745098039, -0.019607843137255, -0.117647058823529, -0.015686274509804,
+0.160784313725490, -0.121568627450980, -0.172549019607843, +0.098039215686275,
-0.031372549019608, -0.141176470588235, -0.043137254901961, -0.101960784313725,
-0.105882352941176, +0.070588235294118, -0.011764705882353, -0.125490196078431,
+0.031372549019608, +0.011764705882353, -0.137254901960784, -0.043137254901961,
+0.105882352941176, -0.007843137254902, -0.011764705882353, +0.094117647058824,
-0.011764705882353, +0.031372549019608, +0.274509803921569, +0.125490196078431,
-0.137254901960784, -0.027450980392157, +0.200000000000000, +0.109803921568627,
-0.023529411764706, +0.180392156862745, +0.101960784313725, -0.105882352941176,
+0.094117647058824, +0.019607843137255, -0.066666666666667, +0.176470588235294,
+0.227450980392157, +0.000000000000000, +0.027450980392157, +0.211764705882353,
-0.003921568627451, -0.109803921568627, +0.105882352941176, +0.054901960784314,
-0.133333333333333, +0.043137254901961, +0.117647058823529, -0.054901960784314,
+0.027450980392157, +0.149019607843137, -0.019607843137255, -0.011764705882353,
+0.145098039215686, -0.101960784313725, -0.086274509803922, +0.137254901960784,
+0.074509803921569, +0.043137254901961, +0.149019607843137, +0.027450980392157,
-0.168627450980392, +0.031372549019608, +0.160784313725490, +0.031372549019608,
+0.094117647058824, +0.176470588235294, -0.019607843137255, +0.011764705882353,
+0.188235294117647, +0.188235294117647, +0.000000000000000, -0.043137254901961,
+0.145098039215686, +0.082352941176471, +0.047058823529412, +0.184313725490196,
+0.082352941176471, +0.101960784313725, +0.164705882352941, -0.019607843137255,
+0.054901960784314, +0.156862745098039, +0.054901960784314, -0.031372549019608,
+0.070588235294118, +0.074509803921569, -0.047058823529412, +0.129411764705882,
+0.098039215686275, -0.031372549019608, +0.149019607843137, +0.188235294117647,
+0.019607843137255, +0.101960784313725, +0.188235294117647, +0.117647058823529,
+0.137254901960784, +0.019607843137255, +0.086274509803922, +0.062745098039216,
+0.011764705882353, +0.137254901960784, +0.149019607843137, +0.109803921568627,
+0.000000000000000, +0.129411764705882, +0.145098039215686, -0.125490196078431,
+0.047058823529412, +0.188235294117647, +0.062745098039216, +0.101960784313725,
+0.113725490196078, -0.054901960784314, +0.011764705882353, +0.109803921568627,
+0.039215686274510, -0.090196078431373, +0.043137254901961, +0.133333333333333,
-0.086274509803922, +0.050980392156863, +0.168627450980392, -0.015686274509804,
-0.027450980392157, +0.003921568627451, -0.200000000000000, -0.129411764705882,
+0.117647058823529, -0.043137254901961, -0.117647058823529, +0.105882352941176,
+0.027450980392157, -0.043137254901961, +0.062745098039216, -0.039215686274510,
-0.172549019607843, +0.019607843137255, -0.003921568627451, -0.117647058823529,
+0.054901960784314, +0.007843137254902, -0.109803921568627, -0.082352941176471,
-0.019607843137255, -0.133333333333333, -0.043137254901961, +0.082352941176471,
-0.156862745098039, -0.172549019607843, +0.164705882352941, +0.109803921568627,
-0.211764705882353, -0.168627450980392, +0.011764705882353, -0.047058823529412,
-0.160784313725490, +0.050980392156863, -0.015686274509804, -0.192156862745098,
+0.000000000000000, -0.058823529411765, -0.239215686274510, -0.109803921568627,
+0.070588235294118, -0.094117647058824, -0.090196078431373, +0.109803921568627,
-0.003921568627451, -0.098039215686275, +0.062745098039216, -0.007843137254902,
-0.223529411764706, -0.031372549019608, +0.043137254901961, -0.152941176470588,
-0.086274509803922, +0.074509803921569, -0.117647058823529, -0.074509803921569,
+0.192156862745098, -0.027450980392157, -0.188235294117647, -0.117647058823529,
-0.176470588235294, -0.094117647058824, +0.098039215686275, +0.035294117647059,
-0.105882352941176, -0.074509803921569, +0.086274509803922, -0.054901960784314,
-0.070588235294118, +0.121568627450980, -0.074509803921569, -0.180392156862745,
-0.023529411764706, +0.062745098039216, -0.054901960784314, -0.121568627450980,
-0.066666666666667, -0.113725490196078, -0.172549019607843, -0.058823529411765,
-0.031372549019608, -0.094117647058824, -0.094117647058824, -0.168627450980392,
-0.050980392156863, +0.039215686274510, -0.133333333333333, -0.207843137254902,
-0.105882352941176, -0.094117647058824, -0.207843137254902, +0.027450980392157,
+0.062745098039216, -0.231372549019608, -0.086274509803922, +0.031372549019608,
-0.129411764705882, -0.031372549019608, +0.039215686274510, -0.043137254901961,
+0.062745098039216, +0.000000000000000, -0.003921568627451, -0.007843137254902,
-0.070588235294118, -0.039215686274510, +0.074509803921569, +0.101960784313725,
+0.031372549019608, +0.133333333333333, +0.141176470588235, -0.062745098039216,
-0.003921568627451, +0.105882352941176, +0.039215686274510, +0.133333333333333,
+0.215686274509804, +0.054901960784314, +0.098039215686275, +0.164705882352941,
+0.062745098039216, +0.019607843137255, +0.113725490196078, +0.156862745098039,
-0.015686274509804, +0.094117647058824, +0.247058823529412, +0.082352941176471,
+0.109803921568627, +0.274509803921569, +0.066666666666667, +0.011764705882353,
+0.188235294117647, +0.007843137254902, -0.074509803921569, +0.152941176470588,
+0.152941176470588, +0.011764705882353, +0.105882352941176, +0.168627450980392,
+0.019607843137255, +0.121568627450980, +0.141176470588235, +0.039215686274510,
+0.192156862745098, +0.149019607843137, +0.000000000000000, +0.062745098039216,
+0.168627450980392, +0.035294117647059, +0.082352941176471, +0.262745098039216,
+0.062745098039216, -0.082352941176471, +0.078431372549020, +0.149019607843137,
-0.003921568627451, -0.003921568627451, +0.031372549019608, -0.031372549019608,
-0.003921568627451, +0.152941176470588, +0.101960784313725, +0.000000000000000,
+0.137254901960784, +0.149019607843137, -0.011764705882353, -0.074509803921569,
+0.094117647058824, +0.007843137254902, -0.019607843137255, +0.164705882352941,
+0.117647058823529, +0.078431372549020, +0.200000000000000, +0.168627450980392,
-0.047058823529412, +0.074509803921569, +0.188235294117647, +0.031372549019608,
+0.058823529411765, +0.211764705882353, +0.043137254901961, -0.003921568627451,
+0.227450980392157, +0.200000000000000, +0.090196078431373, +0.050980392156863,
-0.039215686274510, +0.000000000000000, +0.156862745098039, +0.149019607843137,
+0.082352941176471, +0.031372549019608, +0.109803921568627, +0.031372549019608,
+0.003921568627451, +0.200000000000000, +0.125490196078431, -0.039215686274510,
+0.011764705882353, +0.117647058823529, +0.113725490196078, +0.117647058823529,
+0.105882352941176, -0.039215686274510, -0.149019607843137, -0.019607843137255,
+0.070588235294118, +0.039215686274510, +0.043137254901961, -0.043137254901961,
+0.031372549019608, +0.176470588235294, +0.054901960784314, -0.050980392156863,
+0.058823529411765, +0.117647058823529, -0.062745098039216, +0.050980392156863,
+0.156862745098039, -0.152941176470588, -0.062745098039216, +0.125490196078431,
-0.113725490196078, -0.043137254901961, +0.152941176470588, +0.019607843137255,
-0.094117647058824, -0.090196078431373, +0.000000000000000, -0.062745098039216,
-0.152941176470588, -0.215686274509804, -0.188235294117647, -0.196078431372549,
-0.227450980392157, -0.121568627450980, -0.074509803921569, -0.188235294117647,
-0.184313725490196, -0.105882352941176, -0.207843137254902, -0.105882352941176,
+0.027450980392157, -0.164705882352941, -0.125490196078431, +0.062745098039216,
-0.105882352941176, -0.211764705882353, +0.007843137254902, +0.019607843137255,
-0.188235294117647, -0.070588235294118, +0.078431372549020, -0.101960784313725,
-0.058823529411765, +0.125490196078431, -0.015686274509804, -0.086274509803922,
+0.117647058823529, -0.054901960784314, -0.184313725490196, -0.035294117647059,
-0.023529411764706, -0.098039215686275, -0.074509803921569, -0.043137254901961,
-0.094117647058824, +0.019607843137255, -0.019607843137255, -0.090196078431373,
+0.070588235294118, +0.058823529411765, -0.180392156862745, -0.105882352941176,
+0.094117647058824, -0.011764705882353, -0.043137254901961, +0.070588235294118,
-0.043137254901961, -0.050980392156863, +0.054901960784314, +0.058823529411765,
+0.074509803921569, +0.176470588235294, +0.050980392156863, -0.156862745098039,
-0.070588235294118, +0.113725490196078, +0.094117647058824, +0.011764705882353,
+0.129411764705882, +0.188235294117647, +0.137254901960784, +0.011764705882353,
+0.050980392156863, +0.007843137254902, -0.062745098039216, +0.094117647058824,
+0.011764705882353, -0.062745098039216, +0.074509803921569, +0.129411764705882,
-0.027450980392157, -0.023529411764706, +0.090196078431373, -0.050980392156863,
-0.043137254901961, +0.117647058823529, +0.054901960784314, -0.023529411764706,
+0.039215686274510, +0.062745098039216, +0.117647058823529, +0.129411764705882,
-0.027450980392157, -0.074509803921569, +0.043137254901961, -0.011764705882353,
-0.094117647058824, -0.019607843137255, -0.043137254901961, -0.137254901960784,
-0.090196078431373, +0.047058823529412, -0.003921568627451, -0.078431372549020,
+0.015686274509804, +0.003921568627451, -0.019607843137255, +0.098039215686275,
+0.247058823529412, +0.078431372549020, -0.098039215686275, +0.078431372549020,
+0.117647058823529, +0.109803921568627, +0.129411764705882, +0.007843137254902,
+0.031372549019608, +0.219607843137255, +0.215686274509804, +0.105882352941176,
+0.149019607843137, +0.188235294117647, +0.066666666666667, +0.125490196078431,
+0.247058823529412, +0.000000000000000, -0.011764705882353, +0.250980392156863,
+0.082352941176471, -0.015686274509804, +0.180392156862745, +0.113725490196078,
-0.078431372549020, -0.168627450980392, +0.086274509803922, +0.168627450980392,
+0.086274509803922, +0.066666666666667, +0.054901960784314, -0.019607843137255,
-0.070588235294118, -0.019607843137255, -0.011764705882353, -0.027450980392157,
+0.035294117647059, +0.039215686274510, -0.137254901960784, -0.031372549019608,
+0.105882352941176, -0.047058823529412, -0.121568627450980, +0.121568627450980,
+0.050980392156863, -0.219607843137255, +0.031372549019608, +0.129411764705882,
-0.117647058823529, -0.023529411764706, +0.094117647058824, -0.184313725490196,
-0.113725490196078, +0.125490196078431, -0.082352941176471, -0.231372549019608,
+0.086274509803922, +0.000000000000000, -0.141176470588235, +0.027450980392157,
-0.027450980392157, -0.121568627450980, +0.003921568627451, -0.062745098039216,
-0.192156862745098, +0.019607843137255, -0.039215686274510, -0.200000000000000,
-0.035294117647059, +0.098039215686275, -0.125490196078431, -0.137254901960784,
+0.015686274509804, -0.125490196078431, -0.133333333333333, +0.035294117647059,
-0.141176470588235, -0.125490196078431, +0.039215686274510, -0.109803921568627,
-0.137254901960784, +0.121568627450980, +0.152941176470588, -0.152941176470588,
-0.239215686274510, -0.082352941176471, -0.086274509803922, -0.211764705882353,
-0.074509803921569, -0.078431372549020, -0.098039215686275, -0.133333333333333,
-0.168627450980392, -0.168627450980392, -0.243137254901961, -0.031372549019608,
-0.082352941176471, -0.266666666666667, -0.203921568627451, -0.109803921568627,
-0.145098039215686, -0.211764705882353, -0.074509803921569, -0.168627450980392,
-0.203921568627451, -0.113725490196078, -0.188235294117647, -0.200000000000000,
-0.145098039215686, -0.207843137254902, -0.156862745098039, -0.031372549019608,
-0.160784313725490, -0.254901960784314, -0.125490196078431, -0.145098039215686,
-0.262745098039216, -0.121568627450980, -0.109803921568627, -0.247058823529412,
-0.247058823529412, -0.137254901960784, -0.200000000000000, -0.274509803921569,
-0.105882352941176, -0.137254901960784, -0.266666666666667, -0.235294117647059,
+0.027450980392157, -0.019607843137255, -0.254901960784314, -0.125490196078431,
-0.149019607843137, -0.172549019607843, -0.086274509803922, -0.196078431372549,
-0.219607843137255, -0.113725490196078, -0.129411764705882, -0.180392156862745,
-0.117647058823529, -0.117647058823529, -0.160784313725490, -0.043137254901961,
+0.019607843137255, -0.152941176470588, -0.211764705882353, -0.062745098039216,
-0.101960784313725, -0.117647058823529, -0.078431372549020, -0.137254901960784,
+0.000000000000000, -0.156862745098039, +0.011764705882353, +0.050980392156863,
-0.074509803921569, +0.054901960784314, +0.215686274509804, +0.133333333333333,
+0.074509803921569, +0.203921568627451, +0.082352941176471, -0.019607843137255,
+0.200000000000000, +0.180392156862745, -0.129411764705882, -0.066666666666667,
+0.137254901960784, +0.098039215686275, +0.000000000000000, +0.094117647058824,
+0.062745098039216, -0.109803921568627, +0.066666666666667, +0.105882352941176,
-0.054901960784314, +0.074509803921569, +0.235294117647059, -0.031372549019608,
-0.101960784313725, +0.200000000000000, +0.137254901960784, -0.074509803921569,
+0.133333333333333, +0.109803921568627, -0.003921568627451, +0.180392156862745,
+0.168627450980392, -0.019607843137255, +0.117647058823529, +0.133333333333333,
-0.047058823529412, +0.125490196078431, +0.160784313725490, -0.031372549019608,
+0.039215686274510, +0.192156862745098, +0.000000000000000, -0.011764705882353,
+0.160784313725490, +0.062745098039216, +0.011764705882353, +0.211764705882353,
+0.062745098039216, -0.054901960784314, +0.137254901960784, +0.070588235294118,
-0.066666666666667, +0.082352941176471, +0.254901960784314, +0.050980392156863,
-0.137254901960784, +0.031372549019608, -0.007843137254902, -0.176470588235294,
+0.043137254901961, +0.070588235294118, -0.027450980392157, +0.054901960784314,
+0.082352941176471, +0.027450980392157, -0.062745098039216, +0.121568627450980,
+0.098039215686275, -0.062745098039216, +0.019607843137255, +0.074509803921569,
+0.090196078431373, +0.039215686274510, +0.137254901960784, +0.058823529411765,
-0.007843137254902, +0.050980392156863, -0.031372549019608, +0.007843137254902,
+0.094117647058824, +0.007843137254902, +0.003921568627451, +0.180392156862745,
+0.121568627450980, +0.015686274509804, +0.101960784313725, +0.105882352941176,
-0.054901960784314, +0.035294117647059, +0.176470588235294, +0.082352941176471,
+0.039215686274510, +0.125490196078431, +0.105882352941176, +0.031372549019608,
+0.086274509803922, +0.074509803921569, -0.019607843137255, -0.109803921568627,
+0.047058823529412, +0.050980392156863, -0.137254901960784, -0.023529411764706,
+0.007843137254902, -0.023529411764706, +0.098039215686275, -0.031372549019608,
-0.145098039215686, -0.062745098039216, -0.113725490196078, -0.125490196078431,
+0.007843137254902, -0.066666666666667, -0.203921568627451, -0.027450980392157,
+0.050980392156863, -0.047058823529412, -0.082352941176471, -0.043137254901961,
-0.145098039215686, -0.043137254901961, +0.070588235294118, +0.000000000000000,
+0.050980392156863, -0.090196078431373, -0.007843137254902, -0.070588235294118,
-0.211764705882353, -0.121568627450980, +0.086274509803922, -0.015686274509804,
-0.094117647058824, +0.145098039215686, +0.003921568627451, -0.243137254901961,
+0.007843137254902, +0.203921568627451, -0.023529411764706, -0.192156862745098,
-0.031372549019608, -0.054901960784314, -0.054901960784314, +0.043137254901961,
-0.105882352941176, -0.105882352941176, +0.043137254901961, -0.074509803921569,
-0.239215686274510, -0.137254901960784, +0.011764705882353, -0.090196078431373,
-0.200000000000000, +0.011764705882353, +0.027450980392157, -0.121568627450980,
+0.011764705882353, -0.023529411764706, -0.152941176470588, -0.058823529411765,
+0.015686274509804, -0.043137254901961, -0.031372549019608, -0.070588235294118,
-0.168627450980392, -0.066666666666667, +0.023529411764706, -0.058823529411765,
+0.003921568627451, +0.070588235294118, -0.137254901960784, -0.070588235294118,
+0.101960784313725, -0.015686274509804, -0.031372549019608, +0.141176470588235,
+0.011764705882353, -0.133333333333333, +0.101960784313725, +0.105882352941176,
-0.082352941176471, -0.003921568627451, +0.215686274509804, +0.074509803921569,
-0.137254901960784, +0.125490196078431, +0.082352941176471, -0.196078431372549,
+0.039215686274510, +0.200000000000000, +0.011764705882353, +0.035294117647059,
+0.192156862745098, +0.090196078431373, -0.039215686274510, +0.082352941176471,
+0.023529411764706, -0.019607843137255, +0.188235294117647, +0.129411764705882,
+0.003921568627451, +0.043137254901961, +0.133333333333333, +0.039215686274510,
+0.086274509803922, +0.180392156862745, +0.054901960784314, +0.098039215686275,
+0.164705882352941, +0.062745098039216, +0.043137254901961, +0.180392156862745,
+0.164705882352941, +0.105882352941176, +0.207843137254902, +0.235294117647059,
+0.070588235294118, +0.070588235294118, +0.227450980392157, +0.098039215686275,
+0.074509803921569, +0.196078431372549, +0.133333333333333, +0.137254901960784,
+0.215686274509804, +0.211764705882353, +0.137254901960784, +0.082352941176471,
+0.172549019607843, +0.133333333333333, +0.039215686274510, +0.145098039215686,
+0.180392156862745, +0.121568627450980, +0.227450980392157, +0.152941176470588,
+0.062745098039216, +0.203921568627451, +0.156862745098039, +0.031372549019608,
+0.168627450980392, +0.082352941176471, -0.149019607843137, +0.031372549019608,
+0.156862745098039, +0.086274509803922, +0.094117647058824, +0.215686274509804,
-0.011764705882353, -0.050980392156863, +0.196078431372549, +0.180392156862745,
+0.113725490196078, +0.058823529411765, +0.176470588235294, +0.129411764705882,
+0.027450980392157, +0.011764705882353, +0.145098039215686, +0.039215686274510,
-0.074509803921569, +0.172549019607843, +0.180392156862745, -0.078431372549020,
-0.027450980392157, +0.215686274509804, +0.200000000000000, -0.054901960784314,
+0.035294117647059, +0.019607843137255, +0.015686274509804, +0.137254901960784,
-0.070588235294118, +0.011764705882353, +0.180392156862745, +0.125490196078431,
-0.035294117647059, -0.039215686274510, +0.015686274509804, -0.082352941176471,
-0.137254901960784, +0.043137254901961, -0.011764705882353, -0.188235294117647,
+0.007843137254902, +0.054901960784314, -0.082352941176471, -0.054901960784314,
-0.043137254901961, -0.050980392156863, +0.019607843137255, -0.031372549019608,
-0.086274509803922, -0.031372549019608, +0.011764705882353, -0.152941176470588,
-0.066666666666667, +0.101960784313725, -0.086274509803922, -0.062745098039216,
+0.062745098039216, -0.164705882352941, -0.172549019607843, +0.098039215686275,
-0.031372549019608, -0.215686274509804, +0.011764705882353, +0.019607843137255,
-0.247058823529412, -0.141176470588235, +0.196078431372549, +0.109803921568627,
-0.192156862745098, +0.043137254901961, +0.125490196078431, -0.231372549019608,
-0.105882352941176, +0.164705882352941, -0.043137254901961, -0.223529411764706,
-0.027450980392157, -0.019607843137255, -0.117647058823529, +0.000000000000000,
-0.121568627450980, -0.176470588235294, +0.007843137254902, -0.058823529411765,
-0.243137254901961, -0.125490196078431, +0.039215686274510, -0.113725490196078,
-0.047058823529412, +0.062745098039216, -0.164705882352941, -0.145098039215686,
-0.050980392156863, -0.180392156862745, -0.219607843137255, -0.023529411764706,
-0.023529411764706, -0.156862745098039, -0.015686274509804, +0.019607843137255,
-0.121568627450980, -0.094117647058824, +0.125490196078431, -0.066666666666667,
-0.207843137254902, -0.043137254901961, -0.149019607843137, -0.207843137254902,
-0.054901960784314, +0.078431372549020, -0.050980392156863, -0.031372549019608,
+0.141176470588235, -0.015686274509804, -0.133333333333333, -0.031372549019608,
-0.062745098039216, -0.125490196078431, +0.070588235294118, +0.027450980392157,
-0.094117647058824, +0.082352941176471, +0.094117647058824, -0.125490196078431,
+0.023529411764706, +0.105882352941176, -0.156862745098039, -0.129411764705882,
-0.039215686274510, -0.129411764705882, -0.094117647058824, +0.141176470588235,
-0.054901960784314, -0.184313725490196, +0.031372549019608, -0.043137254901961,
-0.031372549019608, -0.019607843137255, +0.113725490196078, +0.058823529411765,
-0.050980392156863, -0.043137254901961, +0.101960784313725, -0.011764705882353,
-0.211764705882353, +0.000000000000000, +0.117647058823529, -0.074509803921569,
-0.074509803921569, +0.066666666666667, +0.066666666666667, -0.015686274509804,
+0.109803921568627, +0.125490196078431, +0.133333333333333, +0.172549019607843,
-0.023529411764706, +0.011764705882353, +0.168627450980392, +0.172549019607843,
+0.027450980392157, +0.094117647058824, +0.262745098039216, +0.129411764705882,
+0.019607843137255, +0.176470588235294, +0.176470588235294, -0.031372549019608,
+0.074509803921569, +0.164705882352941, +0.050980392156863, +0.098039215686275,
+0.141176470588235, +0.082352941176471, +0.172549019607843, +0.149019607843137,
+0.035294117647059, +0.090196078431373, +0.215686274509804, +0.050980392156863,
+0.003921568627451, +0.152941176470588, -0.039215686274510, -0.047058823529412,
+0.215686274509804, -0.019607843137255, -0.168627450980392, +0.176470588235294,
+0.070588235294118, -0.176470588235294, +0.074509803921569, +0.231372549019608,
-0.031372549019608, -0.125490196078431, +0.121568627450980, +0.160784313725490,
-0.050980392156863, +0.054901960784314, +0.250980392156863, -0.011764705882353,
-0.050980392156863, +0.207843137254902, +0.196078431372549, -0.015686274509804,
+0.062745098039216, +0.133333333333333, +0.094117647058824, +0.196078431372549,
-0.015686274509804, -0.058823529411765, +0.149019607843137, +0.137254901960784,
-0.086274509803922, -0.050980392156863, +0.164705882352941, -0.003921568627451,
-0.011764705882353, +0.168627450980392, -0.047058823529412, -0.074509803921569,
+0.133333333333333, +0.062745098039216, -0.086274509803922, +0.043137254901961,
+0.113725490196078, -0.011764705882353, +0.109803921568627, +0.000000000000000,
-0.196078431372549, -0.090196078431373, +0.196078431372549, +0.074509803921569,
-0.109803921568627, +0.082352941176471, +0.070588235294118, -0.086274509803922,
-0.062745098039216, +0.109803921568627, -0.015686274509804, -0.027450980392157,
+0.180392156862745, +0.023529411764706, -0.200000000000000, -0.047058823529412,
-0.007843137254902, -0.141176470588235, +0.066666666666667, +0.000000000000000,
-0.223529411764706, -0.023529411764706, +0.149019607843137, -0.082352941176471,
-0.011764705882353, +0.200000000000000, +0.047058823529412, +0.043137254901961,
+0.101960784313725, -0.117647058823529, -0.125490196078431, +0.168627450980392,
+0.039215686274510, -0.094117647058824, +0.098039215686275, +0.003921568627451,
-0.101960784313725, -0.160784313725490, -0.062745098039216, +0.019607843137255,
-0.090196078431373, -0.141176470588235, +0.062745098039216, +0.066666666666667,
-0.188235294117647, -0.113725490196078, -0.050980392156863, -0.207843137254902,
-0.098039215686275, +0.062745098039216, -0.125490196078431, -0.145098039215686,
-0.050980392156863, -0.145098039215686, -0.062745098039216, +0.066666666666667,
-0.125490196078431, -0.188235294117647, +0.043137254901961, +0.074509803921569,
-0.172549019607843, -0.156862745098039, +0.074509803921569, -0.035294117647059,
-0.086274509803922, +0.078431372549020, +0.035294117647059, -0.137254901960784,
-0.023529411764706, -0.003921568627451, -0.180392156862745, -0.070588235294118,
+0.023529411764706, -0.113725490196078, -0.019607843137255, +0.003921568627451,
-0.180392156862745, -0.121568627450980, -0.003921568627451, -0.074509803921569,
-0.054901960784314, +0.086274509803922, -0.129411764705882, -0.176470588235294,
+0.156862745098039, -0.007843137254902, -0.192156862745098, +0.137254901960784,
+0.062745098039216, -0.270588235294118, -0.074509803921569, +0.145098039215686,
+0.019607843137255, -0.078431372549020, +0.035294117647059, +0.039215686274510,
-0.105882352941176, -0.011764705882353, +0.188235294117647, +0.050980392156863,
-0.050980392156863, +0.082352941176471, +0.090196078431373, +0.035294117647059,
+0.086274509803922, +0.027450980392157, -0.019607843137255, +0.133333333333333,
-0.023529411764706, -0.125490196078431, +0.105882352941176, +0.192156862745098,
+0.000000000000000, -0.070588235294118, +0.062745098039216, -0.086274509803922,
-0.074509803921569, +0.172549019607843, +0.023529411764706, -0.062745098039216,
+0.129411764705882, +0.125490196078431, +0.027450980392157, +0.031372549019608,
+0.062745098039216, +0.070588235294118, +0.247058823529412, +0.109803921568627,
-0.164705882352941, -0.078431372549020, +0.145098039215686, +0.125490196078431,
-0.031372549019608, +0.101960784313725, +0.098039215686275, -0.047058823529412,
-0.011764705882353, +0.113725490196078, +0.027450980392157, -0.031372549019608,
+0.098039215686275, +0.078431372549020, -0.054901960784314, +0.070588235294118,
+0.129411764705882, -0.043137254901961, +0.137254901960784, +0.086274509803922,
-0.184313725490196, -0.023529411764706, +0.250980392156863, +0.066666666666667,
+0.007843137254902, +0.098039215686275, +0.023529411764706, +0.137254901960784,
+0.250980392156863, +0.019607843137255, -0.105882352941176, +0.129411764705882,
+0.094117647058824, -0.011764705882353, +0.062745098039216, -0.015686274509804,
+0.105882352941176, -0.007843137254902, -0.031372549019608, +0.121568627450980,
+0.007843137254902, -0.113725490196078, +0.086274509803922, +0.200000000000000,
+0.066666666666667, +0.105882352941176, +0.039215686274510, -0.192156862745098,
-0.054901960784314, +0.219607843137255, +0.043137254901961, -0.007843137254902,
+0.086274509803922, -0.105882352941176, -0.109803921568627, +0.141176470588235,
+0.039215686274510, -0.125490196078431, +0.105882352941176, +0.207843137254902,
-0.086274509803922, -0.172549019607843, +0.054901960784314, -0.078431372549020,
-0.129411764705882, +0.113725490196078, +0.062745098039216, -0.121568627450980,
+0.050980392156863, +0.074509803921569, -0.164705882352941, -0.050980392156863,
+0.094117647058824, -0.070588235294118, +0.011764705882353, +0.133333333333333,
-0.031372549019608, +0.011764705882353, +0.047058823529412, -0.113725490196078,
-0.062745098039216, +0.200000000000000, +0.000000000000000, -0.156862745098039,
+0.098039215686275, +0.003921568627451, -0.133333333333333, +0.082352941176471,
+0.074509803921569, -0.188235294117647, -0.105882352941176, -0.031372549019608,
-0.137254901960784, -0.176470588235294, -0.015686274509804, -0.011764705882353,
-0.254901960784314, -0.156862745098039, +0.043137254901961, -0.086274509803922,
-0.231372549019608, -0.129411764705882, -0.180392156862745, -0.125490196078431,
+0.000000000000000, -0.149019607843137, -0.200000000000000, +0.031372549019608,
-0.011764705882353, -0.274509803921569, -0.223529411764706, -0.105882352941176,
-0.156862745098039, -0.145098039215686, +0.011764705882353, -0.156862745098039,
-0.164705882352941, +0.074509803921569, -0.062745098039216, -0.176470588235294,
-0.031372549019608, -0.113725490196078, -0.129411764705882, -0.090196078431373,
-0.219607843137255, -0.219607843137255, +0.031372549019608, +0.062745098039216,
-0.211764705882353, -0.164705882352941, +0.023529411764706, -0.015686274509804,
-0.137254901960784, +0.019607843137255, -0.035294117647059, -0.200000000000000,
-0.050980392156863, -0.027450980392157, -0.160784313725490, -0.168627450980392,
-0.082352941176471, -0.137254901960784, -0.149019607843137, +0.050980392156863,
+0.035294117647059, -0.152941176470588, +0.082352941176471, +0.090196078431373,
-0.196078431372549, -0.082352941176471, +0.094117647058824, -0.054901960784314,
-0.011764705882353, +0.007843137254902, -0.168627450980392, -0.098039215686275,
+0.003921568627451, -0.074509803921569, -0.188235294117647, -0.043137254901961,
-0.074509803921569, -0.152941176470588, -0.121568627450980, -0.188235294117647,
-0.082352941176471, -0.113725490196078, -0.141176470588235, +0.062745098039216,
-0.011764705882353, -0.207843137254902, -0.149019607843137, -0.105882352941176,
-0.062745098039216, +0.176470588235294, +0.184313725490196, -0.168627450980392,
-0.141176470588235, +0.101960784313725, +0.011764705882353, -0.027450980392157,
+0.062745098039216, -0.074509803921569, -0.149019607843137, +0.074509803921569,
+0.074509803921569, -0.125490196078431, +0.035294117647059, +0.223529411764706,
-0.023529411764706, -0.117647058823529, +0.149019607843137, +0.035294117647059,
-0.094117647058824, +0.129411764705882, +0.098039215686275, -0.078431372549020,
+0.113725490196078, +0.129411764705882, -0.109803921568627, -0.015686274509804,
+0.074509803921569, -0.039215686274510, +0.015686274509804, +0.058823529411765,
-0.011764705882353, +0.090196078431373, +0.117647058823529, -0.043137254901961,
-0.050980392156863, +0.192156862745098, +0.101960784313725, -0.031372549019608,
+0.133333333333333, +0.011764705882353, -0.039215686274510, +0.172549019607843,
+0.156862745098039, -0.027450980392157, +0.074509803921569, +0.125490196078431,
-0.043137254901961, -0.172549019607843, +0.039215686274510, +0.211764705882353,
-0.074509803921569, -0.113725490196078, +0.125490196078431, +0.098039215686275,
-0.050980392156863, -0.031372549019608, -0.156862745098039, -0.019607843137255,
+0.223529411764706, +0.058823529411765, -0.050980392156863, +0.156862745098039,
+0.223529411764706, +0.054901960784314, +0.031372549019608, +0.031372549019608,
-0.011764705882353, +0.027450980392157, +0.207843137254902, +0.035294117647059,
-0.062745098039216, +0.145098039215686, +0.058823529411765, -0.062745098039216,
+0.098039215686275, +0.070588235294118, +0.031372549019608, +0.137254901960784,
+0.007843137254902, -0.172549019607843, -0.043137254901961, +0.160784313725490,
-0.023529411764706, -0.066666666666667, +0.113725490196078, +0.050980392156863,
-0.074509803921569, +0.137254901960784, +0.117647058823529, -0.086274509803922,
+0.078431372549020, +0.149019607843137, +0.043137254901961, +0.062745098039216,
+0.145098039215686, -0.003921568627451, -0.156862745098039, +0.043137254901961,
+0.141176470588235, -0.039215686274510, +0.125490196078431, +0.156862745098039,
-0.180392156862745, -0.105882352941176, +0.098039215686275, -0.137254901960784,
-0.050980392156863, +0.152941176470588, +0.043137254901961, +0.094117647058824,
+0.121568627450980, +0.023529411764706, -0.101960784313725, +0.043137254901961,
+0.007843137254902, -0.062745098039216, +0.086274509803922, +0.031372549019608,
-0.054901960784314, -0.078431372549020, -0.109803921568627, +0.090196078431373,
+0.070588235294118, -0.066666666666667, -0.031372549019608, -0.105882352941176,
-0.192156862745098, +0.023529411764706, +0.215686274509804, -0.105882352941176,
-0.223529411764706, -0.015686274509804, -0.050980392156863, -0.074509803921569,
+0.007843137254902, -0.109803921568627, -0.250980392156863, -0.074509803921569,
-0.011764705882353, -0.215686274509804, -0.074509803921569, +0.094117647058824,
-0.137254901960784, -0.180392156862745, +0.031372549019608, -0.090196078431373,
-0.184313725490196, +0.062745098039216, +0.007843137254902, -0.219607843137255,
+0.011764705882353, +0.074509803921569, -0.141176470588235, -0.019607843137255,
-0.031372549019608, -0.207843137254902, -0.066666666666667, -0.074509803921569,
-0.168627450980392, -0.066666666666667, -0.078431372549020, -0.231372549019608,
-0.176470588235294, +0.031372549019608, -0.031372549019608, -0.137254901960784,
+0.023529411764706, -0.062745098039216, -0.176470588235294, +0.070588235294118,
+0.043137254901961, -0.133333333333333, +0.074509803921569, +0.243137254901961,
+0.050980392156863, -0.109803921568627, -0.031372549019608, +0.031372549019608,
-0.125490196078431, -0.211764705882353, -0.007843137254902, +0.137254901960784,
+0.176470588235294, +0.137254901960784, -0.152941176470588, -0.168627450980392,
+0.082352941176471, +0.019607843137255, -0.094117647058824, +0.054901960784314,
+0.054901960784314, +0.050980392156863, +0.211764705882353, +0.164705882352941,
+0.007843137254902, +0.019607843137255, +0.113725490196078, +0.003921568627451,
-0.070588235294118, +0.019607843137255, -0.043137254901961, -0.082352941176471,
+0.066666666666667, +0.050980392156863, -0.031372549019608, +0.047058823529412,
+0.129411764705882, +0.007843137254902, -0.019607843137255, +0.125490196078431,
+0.015686274509804, -0.043137254901961, +0.043137254901961, -0.027450980392157,
-0.078431372549020, +0.086274509803922, +0.062745098039216, -0.074509803921569,
+0.007843137254902, +0.141176470588235, +0.105882352941176, +0.070588235294118,
+0.180392156862745, +0.133333333333333, -0.062745098039216, -0.074509803921569,
-0.031372549019608, -0.050980392156863, +0.101960784313725, +0.188235294117647,
-0.031372549019608, -0.043137254901961, +0.219607843137255, -0.031372549019608,
-0.145098039215686, +0.098039215686275, +0.086274509803922, +0.086274509803922,
+0.149019607843137, +0.094117647058824, -0.019607843137255, +0.164705882352941,
+0.082352941176471, -0.074509803921569, +0.125490196078431, +0.105882352941176,
+0.094117647058824, +0.031372549019608, -0.023529411764706, +0.192156862745098,
+0.188235294117647, +0.109803921568627, +0.219607843137255, +0.200000000000000,
+0.011764705882353, +0.031372549019608, +0.250980392156863, +0.043137254901961,
-0.133333333333333, +0.101960784313725, +0.145098039215686, +0.105882352941176,
+0.235294117647059, +0.184313725490196, -0.031372549019608, +0.074509803921569,
+0.117647058823529, -0.090196078431373, +0.031372549019608, +0.105882352941176,
-0.117647058823529, -0.105882352941176, +0.121568627450980, -0.039215686274510,
-0.200000000000000, +0.094117647058824, +0.180392156862745, -0.035294117647059,
+0.054901960784314, +0.149019607843137, -0.007843137254902, +0.109803921568627,
+0.192156862745098, -0.007843137254902, +0.058823529411765, +0.133333333333333,
+0.043137254901961, +0.121568627450980, +0.098039215686275, -0.117647058823529,
-0.094117647058824, +0.094117647058824, +0.023529411764706, -0.086274509803922,
+0.086274509803922, +0.090196078431373, -0.168627450980392, -0.019607843137255,
+0.066666666666667, -0.105882352941176, +0.039215686274510, +0.211764705882353,
-0.011764705882353, -0.082352941176471, +0.054901960784314, +0.007843137254902,
-0.031372549019608, -0.121568627450980, -0.074509803921569, -0.082352941176471,
+0.023529411764706, +0.188235294117647, +0.007843137254902, -0.145098039215686,
-0.062745098039216, -0.109803921568627, -0.180392156862745, -0.011764705882353,
-0.137254901960784, -0.274509803921569, -0.031372549019608, +0.074509803921569,
-0.180392156862745, -0.113725490196078, +0.019607843137255, -0.113725490196078,
-0.117647058823529, -0.019607843137255, -0.137254901960784, -0.227450980392157,
-0.074509803921569, -0.078431372549020, -0.113725490196078, -0.094117647058824,
-0.035294117647059, -0.094117647058824, -0.078431372549020, -0.043137254901961,
-0.156862745098039, -0.149019607843137, -0.086274509803922, -0.207843137254902,
-0.188235294117647, -0.047058823529412, -0.121568627450980, -0.172549019607843,
-0.200000000000000, -0.074509803921569, -0.078431372549020, -0.200000000000000,
-0.082352941176471, -0.035294117647059, -0.105882352941176, -0.137254901960784,
-0.223529411764706, -0.243137254901961, -0.086274509803922, -0.019607843137255,
-0.125490196078431, -0.160784313725490, +0.019607843137255, -0.105882352941176,
-0.294117647058824, -0.160784313725490, -0.121568627450980, -0.180392156862745,
-0.090196078431373, -0.113725490196078, -0.184313725490196, +0.062745098039216,
+0.000000000000000, -0.211764705882353, -0.062745098039216, +0.003921568627451,
-0.043137254901961, -0.086274509803922, -0.164705882352941, +0.125490196078431,
+0.027450980392157, -0.129411764705882, +0.000000000000000, +0.090196078431373,
-0.039215686274510, -0.023529411764706, +0.168627450980392, +0.039215686274510,
-0.109803921568627, +0.047058823529412, +0.062745098039216, -0.047058823529412,
+0.070588235294118, +0.109803921568627, -0.039215686274510, +0.109803921568627,
+0.117647058823529, -0.074509803921569, +0.094117647058824, +0.156862745098039,
-0.039215686274510, -0.039215686274510, +0.200000000000000, +0.050980392156863,
-0.188235294117647, +0.027450980392157, +0.141176470588235, +0.050980392156863,
+0.070588235294118, +0.125490196078431, +0.035294117647059, +0.058823529411765,
+0.156862745098039, +0.101960784313725, +0.090196078431373, +0.090196078431373,
+0.035294117647059, +0.168627450980392, +0.211764705882353, +0.007843137254902,
-0.039215686274510, +0.156862745098039, +0.129411764705882, -0.050980392156863,
+0.094117647058824, +0.239215686274510, +0.035294117647059, +0.062745098039216,
+0.074509803921569, -0.101960784313725, +0.043137254901961, +0.180392156862745,
-0.043137254901961, -0.050980392156863, +0.203921568627451, +0.192156862745098,
+0.101960784313725, +0.031372549019608, +0.137254901960784, -0.003921568627451,
-0.086274509803922, +0.149019607843137, +0.125490196078431, +0.066666666666667,
+0.188235294117647, +0.101960784313725, +0.019607843137255, +0.180392156862745,
+0.141176470588235, -0.062745098039216, +0.000000000000000, +0.192156862745098,
-0.078431372549020, -0.050980392156863, +0.196078431372549, +0.113725490196078,
+0.066666666666667, +0.192156862745098, +0.070588235294118, -0.105882352941176,
+0.074509803921569, +0.137254901960784, +0.105882352941176, +0.109803921568627,
+0.074509803921569, -0.074509803921569, +0.039215686274510, +0.156862745098039,
+0.082352941176471, +0.098039215686275, +0.176470588235294, -0.011764705882353,
-0.082352941176471, +0.113725490196078, +0.082352941176471, +0.031372549019608,
-0.031372549019608, +0.086274509803922, +0.113725490196078, -0.011764705882353,
+0.070588235294118, +0.113725490196078, +0.094117647058824, +0.133333333333333,
+0.109803921568627, -0.011764705882353, +0.031372549019608, +0.074509803921569,
-0.047058823529412, -0.050980392156863, +0.098039215686275, +0.023529411764706,
-0.137254901960784, -0.031372549019608, +0.074509803921569, -0.003921568627451,
+0.066666666666667, +0.003921568627451, -0.121568627450980, +0.117647058823529,
+0.184313725490196, +0.031372549019608, +0.050980392156863, +0.062745098039216,
+0.066666666666667, +0.043137254901961, -0.125490196078431, +0.109803921568627,
+0.039215686274510, -0.184313725490196, -0.090196078431373, -0.007843137254902,
-0.152941176470588, -0.066666666666667, +0.078431372549020, -0.062745098039216,
-0.149019607843137, +0.007843137254902, +0.000000000000000, -0.164705882352941,
-0.039215686274510, +0.011764705882353, -0.137254901960784, +0.043137254901961,
+0.074509803921569, -0.094117647058824, +0.035294117647059, +0.047058823529412,
-0.160784313725490, -0.188235294117647, +0.109803921568627, +0.062745098039216,
-0.211764705882353, -0.082352941176471, -0.062745098039216, -0.168627450980392,
-0.082352941176471, -0.031372549019608, -0.125490196078431, -0.082352941176471,
-0.043137254901961, -0.145098039215686, -0.062745098039216, +0.000000000000000,
-0.145098039215686, -0.054901960784314, +0.000000000000000, -0.137254901960784,
-0.188235294117647, -0.031372549019608, +0.000000000000000, -0.219607843137255,
-0.105882352941176, +0.023529411764706, -0.105882352941176, +0.019607843137255,
-0.015686274509804, -0.282352941176471, -0.050980392156863, +0.156862745098039,
-0.070588235294118, -0.133333333333333, +0.035294117647059, +0.043137254901961,
-0.125490196078431, -0.149019607843137, +0.152941176470588, +0.082352941176471,
-0.176470588235294, +0.015686274509804, -0.011764705882353, -0.117647058823529,
+0.082352941176471, +0.027450980392157, -0.074509803921569, +0.023529411764706,
+0.043137254901961, -0.047058823529412, +0.027450980392157, +0.200000000000000,
-0.050980392156863, -0.117647058823529, +0.113725490196078, +0.082352941176471,
+0.023529411764706, +0.149019607843137, +0.125490196078431, +0.000000000000000,
+0.066666666666667, +0.074509803921569, +0.047058823529412, +0.113725490196078,
+0.125490196078431, -0.062745098039216, +0.011764705882353, +0.176470588235294,
+0.054901960784314, +0.023529411764706, +0.207843137254902, +0.066666666666667,
-0.105882352941176, +0.109803921568627, +0.211764705882353, +0.152941176470588,
+0.062745098039216, +0.074509803921569, +0.082352941176471, +0.062745098039216,
+0.145098039215686, +0.176470588235294, +0.125490196078431, +0.121568627450980,
+0.200000000000000, +0.152941176470588, +0.145098039215686, +0.188235294117647,
+0.090196078431373, +0.094117647058824, +0.250980392156863, +0.223529411764706,
+0.082352941176471, +0.113725490196078, +0.141176470588235, +0.050980392156863,
+0.168627450980392, +0.200000000000000, +0.047058823529412, +0.129411764705882,
+0.125490196078431, -0.043137254901961, +0.011764705882353, +0.050980392156863,
-0.011764705882353, -0.031372549019608, -0.133333333333333, +0.109803921568627,
+0.176470588235294, -0.023529411764706, +0.039215686274510, +0.043137254901961,
-0.149019607843137, +0.007843137254902, +0.180392156862745, +0.011764705882353,
-0.109803921568627, +0.086274509803922, +0.149019607843137, -0.039215686274510,
+0.050980392156863, +0.050980392156863, -0.168627450980392, -0.050980392156863,
+0.082352941176471, +0.011764705882353, +0.090196078431373, +0.164705882352941,
+0.007843137254902, -0.105882352941176, +0.125490196078431, +0.200000000000000,
-0.039215686274510, +0.039215686274510, +0.086274509803922, -0.125490196078431,
-0.054901960784314, +0.101960784313725, -0.047058823529412, -0.019607843137255,
+0.105882352941176, -0.066666666666667, -0.043137254901961, +0.149019607843137,
+0.031372549019608, +0.050980392156863, +0.109803921568627, +0.000000000000000,
-0.054901960784314, +0.066666666666667, +0.074509803921569, -0.121568627450980,
+0.003921568627451, +0.066666666666667, -0.094117647058824, +0.101960784313725,
+0.200000000000000, -0.152941176470588, -0.074509803921569, +0.168627450980392,
-0.047058823529412, -0.168627450980392, +0.031372549019608, +0.094117647058824,
-0.152941176470588, -0.211764705882353, +0.117647058823529, +0.152941176470588,
-0.156862745098039, +0.019607843137255, +0.015686274509804, -0.219607843137255,
-0.054901960784314, -0.054901960784314, -0.176470588235294, -0.117647058823529,
-0.125490196078431, -0.262745098039216, -0.168627450980392, +0.023529411764706,
-0.113725490196078, -0.180392156862745, -0.027450980392157, -0.098039215686275,
-0.192156862745098, -0.117647058823529, -0.113725490196078, -0.125490196078431,
-0.027450980392157, -0.082352941176471, -0.113725490196078, +0.062745098039216,
+0.054901960784314, -0.188235294117647, -0.137254901960784, +0.082352941176471,
-0.098039215686275, -0.200000000000000, +0.090196078431373, +0.031372549019608,
-0.250980392156863, -0.137254901960784, -0.031372549019608, -0.078431372549020,
-0.027450980392157, -0.082352941176471, -0.207843137254902, -0.113725490196078,
+0.047058823529412, -0.027450980392157, -0.137254901960784, -0.070588235294118,
-0.023529411764706, -0.125490196078431, -0.050980392156863, +0.023529411764706,
-0.149019607843137, -0.121568627450980, +0.054901960784314, -0.007843137254902,
-0.129411764705882, -0.062745098039216, -0.082352941176471, -0.129411764705882,
+0.031372549019608, +0.007843137254902, -0.168627450980392, -0.066666666666667,
-0.003921568627451, -0.168627450980392, -0.129411764705882, -0.113725490196078,
-0.145098039215686, -0.176470588235294, -0.247058823529412, -0.031372549019608,
+0.011764705882353, -0.133333333333333, +0.058823529411765, +0.027450980392157,
-0.243137254901961, -0.082352941176471, +0.149019607843137, -0.019607843137255,
-0.200000000000000, -0.031372549019608, +0.019607843137255, -0.168627450980392,
-0.007843137254902, +0.074509803921569, -0.149019607843137, -0.188235294117647,
-0.125490196078431, -0.137254901960784, -0.086274509803922, +0.121568627450980,
+0.043137254901961, -0.109803921568627, -0.031372549019608, -0.019607843137255,
-0.082352941176471, +0.023529411764706, +0.141176470588235, -0.074509803921569,
-0.105882352941176, +0.105882352941176, -0.011764705882353, -0.098039215686275,
+0.101960784313725, +0.062745098039216, -0.023529411764706, +0.062745098039216,
+0.000000000000000, +0.023529411764706, +0.160784313725490, +0.074509803921569,
-0.043137254901961, +0.145098039215686, +0.074509803921569, -0.145098039215686,
+0.043137254901961, +0.105882352941176, -0.003921568627451, +0.113725490196078,
+0.262745098039216, +0.043137254901961, -0.027450980392157, +0.180392156862745,
+0.043137254901961, -0.121568627450980, +0.105882352941176, +0.274509803921569,
+0.019607843137255, -0.149019607843137, +0.101960784313725, +0.235294117647059,
-0.070588235294118, +0.031372549019608, +0.176470588235294, -0.015686274509804,
+0.094117647058824, +0.168627450980392, +0.098039215686275, +0.137254901960784,
+0.176470588235294, +0.000000000000000, -0.007843137254902, +0.109803921568627,
+0.019607843137255, -0.019607843137255, +0.113725490196078, +0.133333333333333,
+0.031372549019608, -0.003921568627451, -0.043137254901961, -0.062745098039216,
+0.098039215686275, +0.074509803921569, -0.129411764705882, +0.019607843137255,
+0.168627450980392, -0.035294117647059, -0.058823529411765, +0.156862745098039,
-0.011764705882353, -0.109803921568627, +0.156862745098039, +0.149019607843137,
-0.121568627450980, -0.074509803921569, +0.039215686274510, -0.054901960784314,
+0.066666666666667, +0.145098039215686, -0.082352941176471, -0.113725490196078,
+0.113725490196078, +0.078431372549020, -0.121568627450980, -0.019607843137255,
+0.101960784313725, -0.070588235294118, +0.000000000000000, +0.176470588235294,
-0.062745098039216, -0.086274509803922, +0.094117647058824, -0.047058823529412,
-0.160784313725490, +0.058823529411765, +0.000000000000000, -0.160784313725490,
+0.050980392156863, +0.054901960784314, -0.109803921568627, +0.011764705882353,
+0.137254901960784, -0.054901960784314, -0.043137254901961, +0.011764705882353,
+0.125490196078431, +0.094117647058824, -0.070588235294118, +0.050980392156863,
+0.074509803921569, -0.070588235294118, +0.141176470588235, +0.160784313725490,
-0.105882352941176, -0.082352941176471, +0.145098039215686, +0.015686274509804,
-0.192156862745098, -0.003921568627451, +0.094117647058824, -0.113725490196078,
-0.011764705882353, +0.184313725490196, +0.027450980392157, -0.031372549019608,
-0.011764705882353, -0.117647058823529, -0.149019607843137, +0.043137254901961,
-0.039215686274510, -0.168627450980392, -0.019607843137255, -0.125490196078431,
-0.258823529411765, -0.133333333333333, +0.074509803921569, -0.101960784313725,
-0.211764705882353, -0.062745098039216, -0.125490196078431, -0.231372549019608,
-0.066666666666667, -0.070588235294118, -0.125490196078431, -0.078431372549020,
-0.176470588235294, -0.196078431372549, -0.011764705882353, -0.011764705882353,
-0.168627450980392, +0.058823529411765, +0.000000000000000, -0.254901960784314,
-0.109803921568627, -0.094117647058824, -0.215686274509804, -0.050980392156863,
+0.050980392156863, -0.113725490196078, -0.172549019607843, -0.023529411764706,
-0.050980392156863, -0.211764705882353, -0.054901960784314, +0.101960784313725,
-0.062745098039216, -0.227450980392157, -0.109803921568627, +0.109803921568627,
-0.070588235294118, -0.105882352941176, +0.027450980392157, -0.094117647058824,
-0.035294117647059, +0.050980392156863, +0.019607843137255, +0.078431372549020,
+0.105882352941176, -0.031372549019608, -0.039215686274510, -0.003921568627451,
-0.090196078431373, -0.031372549019608, +0.035294117647059, +0.031372549019608,
+0.023529411764706, +0.062745098039216, +0.011764705882353, -0.070588235294118,
+0.058823529411765, +0.074509803921569, -0.129411764705882, -0.117647058823529,
+0.074509803921569, -0.039215686274510, -0.023529411764706, +0.098039215686275,
-0.145098039215686, -0.223529411764706, +0.074509803921569, +0.149019607843137,
-0.043137254901961, -0.011764705882353, +0.035294117647059, -0.117647058823529,
-0.125490196078431, +0.082352941176471, +0.031372549019608, -0.078431372549020,
+0.039215686274510, -0.007843137254902, -0.180392156862745, -0.101960784313725,
+0.086274509803922, -0.031372549019608, -0.007843137254902, +0.117647058823529,
-0.101960784313725, -0.070588235294118, +0.152941176470588, -0.015686274509804,
-0.101960784313725, +0.109803921568627, +0.023529411764706, -0.188235294117647,
+0.035294117647059, +0.113725490196078, -0.090196078431373, +0.000000000000000,
+0.133333333333333, -0.074509803921569, -0.101960784313725, +0.070588235294118,
+0.121568627450980, +0.117647058823529, +0.019607843137255, +0.105882352941176,
+0.129411764705882, +0.039215686274510, +0.117647058823529, +0.184313725490196,
+0.054901960784314, -0.070588235294118, +0.137254901960784, +0.207843137254902,
+0.000000000000000, +0.101960784313725, +0.239215686274510, +0.094117647058824,
+0.031372549019608, +0.219607843137255, +0.129411764705882, +0.086274509803922,
+0.211764705882353, +0.125490196078431, +0.031372549019608, +0.188235294117647,
+0.117647058823529, -0.058823529411765, +0.172549019607843, +0.192156862745098,
-0.015686274509804, -0.019607843137255, +0.207843137254902, +0.070588235294118,
-0.043137254901961, +0.109803921568627, +0.035294117647059, -0.101960784313725,
+0.050980392156863, +0.082352941176471, +0.050980392156863, +0.121568627450980,
-0.035294117647059, -0.105882352941176, +0.086274509803922, +0.121568627450980,
-0.031372549019608, +0.149019607843137, +0.200000000000000, +0.000000000000000,
+0.023529411764706, +0.074509803921569, -0.145098039215686, -0.039215686274510,
+0.125490196078431, -0.015686274509804, -0.019607843137255, +0.054901960784314,
+0.003921568627451, -0.196078431372549, -0.050980392156863, +0.129411764705882,
+0.019607843137255, -0.105882352941176, -0.109803921568627, +0.054901960784314,
-0.019607843137255, -0.094117647058824, -0.050980392156863, -0.098039215686275,
-0.047058823529412, -0.019607843137255, -0.094117647058824, +0.031372549019608,
+0.094117647058824, -0.050980392156863, -0.031372549019608, -0.031372549019608,
-0.239215686274510, -0.149019607843137, +0.027450980392157, -0.023529411764706,
-0.047058823529412, +0.043137254901961, -0.090196078431373, -0.231372549019608,
+0.019607843137255, +0.156862745098039, +0.031372549019608, -0.058823529411765,
+0.027450980392157, -0.094117647058824, +0.000000000000000, +0.200000000000000,
+0.011764705882353, -0.149019607843137, +0.019607843137255, +0.047058823529412,
-0.121568627450980, +0.000000000000000, +0.027450980392157, -0.058823529411765,
-0.113725490196078, -0.043137254901961, -0.074509803921569, -0.062745098039216,
+0.070588235294118, -0.019607843137255, -0.125490196078431, -0.050980392156863,
+0.050980392156863, -0.168627450980392, -0.090196078431373, +0.082352941176471,
-0.168627450980392, -0.054901960784314, +0.164705882352941, -0.121568627450980,
-0.137254901960784, +0.098039215686275, +0.031372549019608, -0.129411764705882,
+0.035294117647059, +0.094117647058824, -0.172549019607843, -0.054901960784314,
+0.160784313725490, +0.000000000000000, -0.133333333333333, +0.000000000000000,
-0.070588235294118, -0.070588235294118, -0.113725490196078, +0.000000000000000,
-0.050980392156863, -0.054901960784314, -0.019607843137255, +0.007843137254902,
+0.000000000000000, -0.074509803921569, +0.011764705882353, +0.070588235294118,
-0.070588235294118, -0.023529411764706, +0.035294117647059, -0.031372549019608,
-0.039215686274510, +0.109803921568627, +0.109803921568627, +0.031372549019608,
+0.031372549019608, -0.035294117647059, -0.050980392156863, +0.149019607843137,
+0.137254901960784, -0.039215686274510, +0.101960784313725, +0.145098039215686,
+0.011764705882353, +0.011764705882353, +0.196078431372549, +0.152941176470588,
+0.086274509803922, +0.231372549019608, +0.078431372549020, -0.074509803921569,
+0.062745098039216, +0.101960784313725, +0.145098039215686, +0.262745098039216,
+0.043137254901961, -0.094117647058824, +0.105882352941176, +0.133333333333333,
+0.031372549019608, +0.117647058823529, +0.160784313725490, +0.035294117647059,
-0.035294117647059, +0.117647058823529, +0.054901960784314, +0.035294117647059,
+0.207843137254902, +0.117647058823529, +0.137254901960784, +0.270588235294118,
+0.235294117647059, +0.019607843137255, +0.047058823529412, +0.239215686274510,
+0.149019607843137, +0.043137254901961, +0.094117647058824, +0.168627450980392,
+0.141176470588235, +0.125490196078431, +0.031372549019608, -0.011764705882353,
+0.109803921568627, +0.125490196078431, +0.027450980392157, +0.121568627450980,
+0.231372549019608, +0.125490196078431, +0.160784313725490, +0.274509803921569,
+0.015686274509804, -0.062745098039216, +0.156862745098039, +0.141176470588235,
+0.094117647058824, +0.227450980392157, +0.156862745098039, -0.062745098039216,
+0.062745098039216, +0.207843137254902, +0.145098039215686, +0.074509803921569,
+0.156862745098039, +0.054901960784314, +0.050980392156863, +0.231372549019608,
+0.168627450980392, +0.027450980392157, +0.125490196078431, +0.152941176470588,
+0.000000000000000, +0.109803921568627, +0.164705882352941, +0.082352941176471,
+0.062745098039216, +0.133333333333333, -0.015686274509804, -0.003921568627451,
+0.223529411764706, +0.125490196078431, +0.011764705882353, +0.098039215686275,
+0.250980392156863, +0.054901960784314, +0.050980392156863, +0.239215686274510,
+0.031372549019608, +0.066666666666667, +0.231372549019608, -0.023529411764706,
-0.050980392156863, +0.196078431372549, +0.156862745098039, +0.015686274509804,
+0.121568627450980, +0.239215686274510, +0.062745098039216, +0.062745098039216,
+0.160784313725490, +0.035294117647059, -0.015686274509804, +0.074509803921569,
+0.054901960784314, +0.058823529411765, -0.039215686274510, +0.058823529411765,
-0.031372549019608, -0.105882352941176, -0.039215686274510, +0.011764705882353,
-0.062745098039216, -0.047058823529412, +0.043137254901961, -0.070588235294118,
-0.196078431372549, -0.031372549019608, +0.035294117647059, -0.113725490196078,
-0.168627450980392, -0.043137254901961, +0.035294117647059, +0.058823529411765,
+0.043137254901961, -0.137254901960784, -0.168627450980392, +0.054901960784314,
+0.039215686274510, -0.125490196078431, +0.007843137254902, +0.011764705882353,
-0.200000000000000, -0.164705882352941, +0.043137254901961, -0.011764705882353,
-0.070588235294118, +0.141176470588235, -0.019607843137255, -0.200000000000000,
-0.070588235294118, -0.145098039215686, -0.066666666666667, +0.152941176470588,
-0.050980392156863, -0.168627450980392, +0.027450980392157, -0.003921568627451,
-0.086274509803922, +0.000000000000000, -0.011764705882353, -0.074509803921569,
-0.094117647058824, -0.058823529411765, -0.094117647058824, -0.078431372549020,
+0.019607843137255, -0.152941176470588, -0.117647058823529, +0.101960784313725,
+0.082352941176471, -0.094117647058824, -0.074509803921569, +0.113725490196078,
-0.031372549019608, -0.188235294117647, -0.023529411764706, +0.031372549019608,
+0.000000000000000, +0.129411764705882, -0.027450980392157, -0.200000000000000,
-0.058823529411765, -0.050980392156863, -0.105882352941176, -0.031372549019608,
+0.050980392156863, -0.094117647058824, -0.094117647058824, +0.168627450980392,
+0.098039215686275, -0.058823529411765, +0.011764705882353, -0.054901960784314,
-0.098039215686275, +0.062745098039216, +0.082352941176471, -0.054901960784314,
-0.039215686274510, +0.015686274509804, -0.027450980392157, -0.023529411764706,
-0.023529411764706, -0.137254901960784, -0.094117647058824, +0.062745098039216,
-0.058823529411765, -0.168627450980392, -0.023529411764706, +0.019607843137255,
-0.098039215686275, -0.078431372549020, +0.000000000000000, -0.101960784313725,
-0.090196078431373, +0.094117647058824, -0.074509803921569, -0.149019607843137,
+0.113725490196078, +0.003921568627451, -0.176470588235294, -0.098039215686275,
-0.007843137254902, -0.054901960784314, -0.082352941176471, +0.000000000000000,
-0.086274509803922, -0.094117647058824, +0.047058823529412, -0.031372549019608,
-0.125490196078431, +0.011764705882353, -0.015686274509804, -0.101960784313725,
-0.058823529411765, -0.066666666666667, -0.125490196078431, -0.011764705882353,
+0.062745098039216, -0.090196078431373, -0.113725490196078, -0.070588235294118,
-0.058823529411765, -0.047058823529412, -0.129411764705882, -0.011764705882353,
+0.062745098039216, -0.113725490196078, -0.023529411764706, +0.066666666666667,
-0.149019607843137, -0.176470588235294, +0.082352941176471, +0.011764705882353,
-0.168627450980392, +0.043137254901961, +0.137254901960784, +0.015686274509804,
-0.086274509803922, -0.074509803921569, -0.113725490196078, -0.105882352941176,
+0.078431372549020, +0.019607843137255, -0.160784313725490, -0.007843137254902,
+0.015686274509804, -0.184313725490196, -0.003921568627451, +0.188235294117647,
+0.003921568627451, -0.125490196078431, +0.003921568627451, -0.074509803921569,
-0.152941176470588, +0.164705882352941, +0.121568627450980, -0.058823529411765,
+0.082352941176471, -0.031372549019608, -0.050980392156863, +0.176470588235294,
-0.019607843137255, -0.149019607843137, +0.047058823529412, -0.050980392156863,
-0.184313725490196, +0.070588235294118, +0.121568627450980, +0.011764705882353,
+0.082352941176471, +0.027450980392157, -0.176470588235294, -0.062745098039216,
+0.117647058823529, -0.164705882352941, -0.219607843137255, +0.050980392156863,
+0.031372549019608, -0.207843137254902, -0.133333333333333, +0.180392156862745,
+0.039215686274510, -0.270588235294118, -0.101960784313725, +0.023529411764706,
-0.113725490196078, +0.078431372549020, +0.105882352941176, -0.066666666666667,
-0.015686274509804, -0.121568627450980, -0.243137254901961, -0.137254901960784,
+0.035294117647059, -0.133333333333333, -0.207843137254902, +0.000000000000000,
-0.062745098039216, -0.160784313725490, -0.023529411764706, -0.043137254901961,
-0.137254901960784, -0.031372549019608, -0.062745098039216, -0.207843137254902,
-0.117647058823529, +0.003921568627451, -0.074509803921569, +0.000000000000000,
+0.031372549019608, -0.168627450980392, -0.105882352941176, +0.125490196078431,
-0.015686274509804, -0.176470588235294, -0.074509803921569, -0.050980392156863,
-0.094117647058824, -0.094117647058824, +0.019607843137255, -0.094117647058824,
-0.105882352941176, +0.054901960784314, -0.105882352941176, -0.200000000000000,
+0.054901960784314, +0.043137254901961, -0.156862745098039, -0.098039215686275,
-0.074509803921569, -0.105882352941176, -0.098039215686275, -0.050980392156863,
-0.109803921568627, -0.180392156862745, -0.086274509803922, -0.011764705882353,
-0.094117647058824, -0.015686274509804, -0.031372549019608, -0.094117647058824,
-0.011764705882353, -0.117647058823529, -0.231372549019608, -0.090196078431373,
+0.074509803921569, -0.031372549019608, -0.086274509803922, -0.129411764705882,
-0.094117647058824, -0.050980392156863, -0.156862745098039, -0.082352941176471,
+0.047058823529412, -0.125490196078431, +0.011764705882353, +0.086274509803922,
-0.168627450980392, -0.207843137254902, +0.070588235294118, +0.023529411764706,
-0.231372549019608, -0.031372549019608, +0.050980392156863, -0.070588235294118,
-0.050980392156863, +0.031372549019608, -0.105882352941176, -0.219607843137255,
-0.109803921568627, -0.050980392156863, -0.196078431372549, -0.043137254901961,
+0.094117647058824, -0.188235294117647, -0.109803921568627, +0.047058823529412,
-0.043137254901961, -0.137254901960784, -0.082352941176471, -0.125490196078431,
-0.168627450980392, +0.070588235294118, +0.054901960784314, -0.094117647058824,
-0.011764705882353, -0.039215686274510, -0.070588235294118, +0.090196078431373,
-0.047058823529412, -0.258823529411765, -0.031372549019608, -0.035294117647059,
-0.188235294117647, +0.062745098039216, +0.066666666666667, -0.133333333333333,
+0.039215686274510, +0.168627450980392, +0.007843137254902, +0.035294117647059,
+0.192156862745098, -0.062745098039216, -0.282352941176471, +0.015686274509804,
+0.149019607843137, -0.105882352941176, -0.090196078431373, +0.192156862745098,
+0.188235294117647, -0.105882352941176, -0.054901960784314, +0.105882352941176,
-0.011764705882353, +0.058823529411765, +0.074509803921569, -0.007843137254902,
+0.145098039215686, +0.094117647058824, -0.117647058823529, -0.043137254901961,
+0.200000000000000, +0.082352941176471, -0.027450980392157, +0.129411764705882,
+0.019607843137255, -0.109803921568627, +0.066666666666667, +0.086274509803922,
-0.047058823529412, +0.082352941176471, +0.113725490196078, -0.113725490196078,
-0.086274509803922, +0.098039215686275, +0.007843137254902, +0.090196078431373,
+0.164705882352941, -0.050980392156863, -0.043137254901961, +0.200000000000000,
+0.137254901960784, +0.003921568627451, +0.137254901960784, +0.160784313725490,
+0.109803921568627, +0.082352941176471, +0.172549019607843, +0.019607843137255,
-0.086274509803922, +0.105882352941176, +0.066666666666667, -0.039215686274510,
+0.129411764705882, +0.176470588235294, -0.007843137254902, +0.039215686274510,
+0.117647058823529, +0.086274509803922, +0.105882352941176, +0.207843137254902,
+0.180392156862745, +0.043137254901961, -0.027450980392157, -0.023529411764706,
-0.023529411764706, +0.117647058823529, +0.145098039215686, +0.113725490196078,
+0.227450980392157, +0.145098039215686, -0.039215686274510, -0.019607843137255,
+0.086274509803922, +0.078431372549020, +0.105882352941176, +0.054901960784314,
+0.000000000000000, +0.082352941176471, -0.101960784313725, -0.062745098039216,
+0.086274509803922, -0.090196078431373, +0.054901960784314, +0.160784313725490,
-0.062745098039216, -0.137254901960784, +0.070588235294118, +0.054901960784314,
-0.176470588235294, +0.019607843137255, +0.113725490196078, -0.082352941176471,
-0.050980392156863, +0.168627450980392, +0.062745098039216, -0.074509803921569,
-0.019607843137255, +0.043137254901961, -0.105882352941176, -0.023529411764706,
+0.137254901960784, -0.129411764705882, -0.062745098039216, +0.086274509803922,
+0.031372549019608, +0.039215686274510, +0.023529411764706, -0.098039215686275,
-0.137254901960784, +0.054901960784314, +0.082352941176471, -0.015686274509804,
-0.023529411764706, -0.027450980392157, -0.007843137254902, +0.105882352941176,
+0.050980392156863, -0.086274509803922, +0.074509803921569, +0.000000000000000,
-0.168627450980392, +0.050980392156863, +0.062745098039216, -0.094117647058824,
+0.007843137254902, +0.031372549019608, -0.070588235294118, +0.035294117647059,
+0.168627450980392, +0.023529411764706, -0.188235294117647, +0.031372549019608,
+0.117647058823529, -0.149019607843137, -0.090196078431373, +0.145098039215686,
+0.125490196078431, -0.098039215686275, -0.078431372549020, +0.000000000000000,
-0.070588235294118, +0.031372549019608, +0.058823529411765, -0.023529411764706,
+0.109803921568627, +0.086274509803922, -0.137254901960784, -0.094117647058824,
+0.078431372549020, -0.019607843137255, -0.054901960784314, +0.125490196078431,
+0.011764705882353, -0.137254901960784, +0.003921568627451, +0.031372549019608,
-0.050980392156863, +0.078431372549020, +0.109803921568627, -0.105882352941176,
-0.105882352941176, +0.062745098039216, -0.031372549019608, +0.039215686274510,
+0.054901960784314, -0.113725490196078, -0.054901960784314, +0.133333333333333,
+0.031372549019608, -0.074509803921569, +0.039215686274510, -0.007843137254902,
-0.062745098039216, -0.035294117647059, +0.109803921568627, +0.101960784313725,
-0.027450980392157, -0.023529411764706, -0.066666666666667, -0.086274509803922,
+0.082352941176471, +0.129411764705882, -0.043137254901961, -0.058823529411765,
+0.000000000000000, -0.047058823529412, -0.007843137254902, +0.086274509803922,
+0.043137254901961, -0.015686274509804, +0.007843137254902, -0.054901960784314,
-0.101960784313725, +0.011764705882353, +0.011764705882353, +0.007843137254902,
+0.109803921568627, +0.050980392156863, -0.105882352941176, -0.070588235294118,
+0.019607843137255, +0.019607843137255, +0.050980392156863, +0.000000000000000
};

static void
prepare (GeglOperation *operation)
{
  const Babl *new_format;
  const Babl *format = gegl_operation_get_source_format (operation, "input");

  gint components = 0;
  if (format != NULL)
    {
      components = babl_format_get_n_components (format);
    }

  if (components == 1)
    {
      new_format = babl_format ("Y' float");
    }
  else if (components == 2 && babl_format_has_alpha (format))
    {
      new_format = babl_format ("Y'A float");
    }
  else if (components == 0 || babl_format_has_alpha (format))
    {
      new_format = babl_format ("R'G'B'A float");
    }
  else
    {
      new_format = babl_format ("R'G'B' float");
    }

  gegl_operation_set_format (operation, "input", new_format);
  gegl_operation_set_format (operation, "output", new_format);
}

#include "opencl/gegl-cl.h"
#include "opencl/texturize-canvas.cl.h"

static GeglClRunData *cl_data = NULL;

static gboolean
cl_process(GeglOperation *op,
           cl_mem in_tex,
           cl_mem out_tex,
           size_t samples,
           const GeglRectangle *roi,
           gint level)
{
   GeglProperties *opt = GEGL_PROPERTIES(op);
   float mult = (float)opt->depth * 0.25;
   const Babl *format = gegl_operation_get_format(op, "input");
   int has_alpha = babl_format_has_alpha(format);
   int components = babl_format_get_n_components(format) - has_alpha;
   int xm, ym, offs;
   size_t global_ws[] = {roi->width, roi->height};

   switch(opt->direction)
   {
      default:
      case GEGL_TEXTURIZE_CANVAS_DIRECTION_TOP_RIGHT:
          xm = 1;
          ym = 128;
          offs = 0;
          break;
      case GEGL_TEXTURIZE_CANVAS_DIRECTION_TOP_LEFT:
          xm = -1;
          ym=128;
          offs = 127;
          break;
      case GEGL_TEXTURIZE_CANVAS_DIRECTION_BOTTOM_LEFT:
          xm = 128;
          ym = 1;
          offs = 0;
          break;
      case GEGL_TEXTURIZE_CANVAS_DIRECTION_BOTTOM_RIGHT:
          xm = 128;
          ym = -1;
          offs = 127;
          break;
   }

   if(!cl_data)
   {
      const char *kernel_name[] = {"cl_texturize_canvas", NULL};
      cl_data = gegl_cl_compile_and_build(texturize_canvas_cl_source, kernel_name);
   }
   if(!cl_data)
      return TRUE;
   else
   {
      cl_int cl_err = 0;

      cl_mem sdata_tex = gegl_clCreateBuffer(gegl_cl_get_context(),
                                             CL_MEM_USE_HOST_PTR|CL_MEM_READ_ONLY,
                                             sizeof(cl_float)*128*128,(void *)sdata , &cl_err);
      CL_CHECK;

      cl_err = gegl_cl_set_kernel_args(cl_data->kernel[0],
                                       sizeof(cl_mem), (void *)&in_tex,
                                       sizeof(cl_mem), (void *)&out_tex,
                                       sizeof(cl_mem), (void *)&sdata_tex,
                                       sizeof(cl_int), (void *)&roi->x,
                                       sizeof(cl_int), (void *)&roi->y,
                                       sizeof(cl_int), (void *)&xm,
                                       sizeof(cl_int), (void *)&ym,
                                       sizeof(cl_int), (void *)&offs,
                                       sizeof(cl_float), (void *)&mult,
                                       sizeof(cl_int), (void *)&components,
                                       sizeof(cl_int), (void *)&has_alpha, NULL);
      CL_CHECK;

      cl_err = gegl_clEnqueueNDRangeKernel(gegl_cl_get_command_queue(),
                                           cl_data->kernel[0], 2, NULL,
                                           global_ws, NULL, 0, NULL, NULL);
      CL_CHECK;

      cl_err = gegl_clFinish(gegl_cl_get_command_queue());
      CL_CHECK;

      cl_err = gegl_clReleaseMemObject(sdata_tex);
      CL_CHECK_ONLY(cl_err);

      return FALSE;

error:
      return TRUE;
   }
}


static gboolean
process (GeglOperation       *operation,
         void                *in_buf,
         void                *out_buf,
         glong                samples,
         const GeglRectangle *roi,
         gint                 level)
{
  GeglProperties *opt = GEGL_PROPERTIES (operation);

  gfloat *src  = in_buf;
  gfloat *dest = out_buf;

  gint    xm, ym, offs;
  gfloat  mult = (gfloat) opt->depth * 0.25;

  gint   row;   /* Row number in rectangle */
  gint   col;   /* Column number in rectangle */

  const Babl *format     = gegl_operation_get_format (operation, "input");
  gboolean    has_alpha  = babl_format_has_alpha (format);
  gint        components = babl_format_get_n_components (format) - has_alpha;

  switch (opt->direction)
    {
      default:
      case GEGL_TEXTURIZE_CANVAS_DIRECTION_TOP_RIGHT:
        xm = 1;
        ym = 128;
        offs = 0;
        break;

      case GEGL_TEXTURIZE_CANVAS_DIRECTION_TOP_LEFT:
        xm = -1;
        ym = 128;
        offs = 127;
        break;

      case GEGL_TEXTURIZE_CANVAS_DIRECTION_BOTTOM_LEFT:
        xm = 128;
        ym = 1;
        offs = 0;
        break;

      case GEGL_TEXTURIZE_CANVAS_DIRECTION_BOTTOM_RIGHT:
        xm = 128;
        ym = -1;
        offs = 127;
        break;
    }

  for (row = 0; row < roi->height; ++row)
    {
      for (col = 0; col < roi->width; ++col)
        {
          gint i;
          for (i = 0; i < components; ++i)
            {
              /*
               * Assuming twos-complement representation, it holds that n & 127
               * is n % 128 for n >= 0 and (n % 128) + 128 for n < 0.
               */
              gint   index = ((roi->x + col) & 127) * xm +
                             ((roi->y + row) & 127) * ym +
                             offs;
              gfloat color = mult * sdata [index] + *src++;
              *dest++ = CLAMP (color, 0.0, 1.0);
            }

          if (has_alpha)
            {
              *dest++ = *src++;
            }
        }
    }
  return TRUE;
}

static void
gegl_op_class_init (GeglOpClass *klass)
{
  GeglOperationClass            *operation_class;
  GeglOperationPointFilterClass *point_filter_class;

  operation_class    = GEGL_OPERATION_CLASS (klass);
  point_filter_class = GEGL_OPERATION_POINT_FILTER_CLASS (klass);

  point_filter_class->process = process;
  point_filter_class->cl_process = cl_process;
  operation_class->prepare = prepare;

  operation_class->opencl_support = TRUE;
  gegl_operation_class_set_keys (operation_class,
    "name",               "gegl:texturize-canvas",
    "title",              _("Texturize Canvas"),
    "categories",         "artistic",
    "license",            "GPL3+",
    "position-dependent", "true",
    "reference-hash",     "de335c86aa55cd0ecdb33a8d998c9041",
    "description",_("Textures the image as if it were an artist's canvas."),
    NULL);
}

#endif

