/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QFile>
#include <QThread>
#include <QVariant>

#include "src/datovka_shared/isds/error.h"
#include "src/datovka_shared/isds/type_description.h"
#include "src/datovka_shared/isds/types.h"
#include "src/datovka_shared/log/log.h"
#include "src/global.h"
#include "src/settings/prefs_specific.h"
#include "src/io/isds_sessions.h"
#include "src/isds/message_functions.h"
#include "src/isds/services.h"
#include "src/isds/session.h"
#include "src/worker/message_emitter.h"
#include "src/worker/task_authenticate_message.h"

TaskAuthenticateMessage::TaskAuthenticateMessage(const AcntId &acntId,
    const QString &transactId, const QString &fileName,
    enum MsgSizeType sizeType)
    : QObject(),
    m_result(AUTH_ERR),
    m_isdsError(),
    m_isdsLongError(),
    m_acntId(acntId),
    m_transactId(transactId),
    m_data(),
    m_sizeType(sizeType),
    m_reqQuit(false)
{
	Q_ASSERT(m_acntId.isValid());

	QFile file(fileName);

	if (file.exists()) {
		if (file.open(QIODevice::ReadOnly)) {
			m_data = file.readAll();
			file.close();
		} else {
			logErrorNL("Couldn't open file '%s'.",
			    fileName.toUtf8().constData());
		}
	}
}

TaskAuthenticateMessage::TaskAuthenticateMessage(const AcntId &acntId,
    const QString &transactId, const QByteArray &data,
    enum MsgSizeType sizeType)
    : QObject(),
    m_result(AUTH_ERR),
    m_isdsError(),
    m_isdsLongError(),
    m_acntId(acntId),
    m_transactId(transactId),
    m_data(data),
    m_sizeType(sizeType),
    m_reqQuit(false)
{
	Q_ASSERT(m_acntId.isValid());
}

void TaskAuthenticateMessage::run(void)
{
	if (Q_UNLIKELY(!m_acntId.isValid())) {
		Q_ASSERT(0);
		return;
	}

	logDebugLv0NL("Starting authenticate message task in thread '%p'",
	    (void *) QThread::currentThreadId());

	/* ### Worker task begin. ### */

	Isds::Session *session =
	    GlobInstcs::isdsSessionsPtr->session(m_acntId.username());
	if (Q_NULLPTR != session) {
		session->setAbortXfer(&m_reqQuit);
		connect(session, SIGNAL(progress(qint64, qint64, qint64, qint64)),
		    this, SLOT(watchProgress(qint64, qint64, qint64, qint64)),
		    Qt::DirectConnection); /* Qt::DirectConnection because it must be executed immediatelly. */
	}

	m_result = authenticateMessage(m_acntId, m_data, m_sizeType,
	    m_isdsError, m_isdsLongError);

	if (Q_NULLPTR != session) {
		session->setAbortXfer(Q_NULLPTR);
		session->disconnect(Q_NULLPTR, this, Q_NULLPTR);
	}

	/*
	 * Prefering to retrun only long error string as most of the errors
	 * aren't actually errors.
	 * TODO -- libdatovka must return status messaga data as receibed from
	 * server.
	 */
	Q_EMIT GlobInstcs::msgProcEmitterPtr->downloadProgressFinished(
	    AcntId(m_acntId.username(), m_acntId.testing()), m_transactId,
	    MsgId(), m_result,
	    m_isdsLongError.isEmpty() ? (m_isdsError + " " + m_isdsLongError) : m_isdsLongError);

	Q_EMIT GlobInstcs::msgProcEmitterPtr->progressChange(PL_IDLE, 0);

	/* ### Worker task end. ### */

	logDebugLv0NL("Authenticate message task finished in thread '%p'",
	    (void *) QThread::currentThreadId());
}

void TaskAuthenticateMessage::requestQuit(void)
{
	m_reqQuit = true;
}

/*
 * Typically, the raw size of 26.7 MB indicates a VoDZ.
 * Both, AuthenticateMessage and AuthenticateBigMessage can be used for
 * messages around this threshold.
 *
 * When no archive timestamps have been added.
 *
 * 26.7 MB / 20 MB = 1.335
 */
#define MB_RATIO 1.335

enum TaskAuthenticateMessage::Result TaskAuthenticateMessage::authenticateMessage(
    const AcntId &acntId, const QByteArray &data, enum MsgSizeType sizeType,
    QString &error, QString &longError)
{
	Q_ASSERT(acntId.isValid());

	if (data.isEmpty()) {
		return AUTH_DATA_ERROR;
	}

	Isds::Session *session = GlobInstcs::isdsSessionsPtr->session(acntId.username());
	if (Q_UNLIKELY(Q_NULLPTR == session)) {
		logErrorNL("Missing active session for username '%s'.",
		    acntId.username().toUtf8().constData());
		return AUTH_ERR;
	}

	if (MSG_VODZ_SIZE_DEPENDENT == sizeType) {
		const qint64 threshold = 1024 * 1024 *
		    PrefsSpecific::maxAttachmentMBytes(*GlobInstcs::prefsPtr);
		qint64 attachTotal = 0;
		Isds::guessDataRawTypeAttachTotal(data, attachTotal);
		sizeType = (attachTotal < threshold) ? MSG_BASIC : MSG_VODZ;
	}

	logInfoNL("%s", (MSG_BASIC == sizeType) ?
	    "Authenticating message." : "Authenticating big message.");

	Isds::Error err = (MSG_BASIC == sizeType) ?
	    Isds::Service::authenticateMessage(session, data) :
	    Isds::Service::authenticateBigMessage(session, data);
	if (err.code() == Isds::Type::ERR_NOTEQUAL) {
		return AUTH_NOT_EQUAL;
	} else if (err.code() != Isds::Type::ERR_SUCCESS) {
		error = Isds::Description::descrError(err.code());
		longError = err.longDescr();
		logErrorNL("%s", "Error authenticating message.");
		return AUTH_ISDS_ERROR;
	}

	return AUTH_SUCCESS;
}

void TaskAuthenticateMessage::watchProgress(qint64 uploadTotal,
    qint64 uploadCurrent, qint64 downloadTotal, qint64 downloadCurrent)
{
	Q_UNUSED(uploadTotal);
	Q_UNUSED(uploadCurrent);

	static qint64 lastUploadCurrent = 0;
	static qint64 lastDownloadCurrent = 0;
	bool downloadStarted = false;

	if (!downloadStarted) {
		if (uploadCurrent > lastUploadCurrent) {
			lastUploadCurrent = uploadCurrent;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->uploadProgress(
			    AcntId(m_acntId.username(), m_acntId.testing()),
			    m_transactId, uploadTotal, uploadCurrent);
		}

		if (downloadCurrent > lastDownloadCurrent) {
			downloadStarted = true;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->uploadProgressFinished(
			    AcntId(m_acntId.username(), m_acntId.testing()),
			    m_transactId, AUTH_SUCCESS, QString(),
			    QVariant());
		}
	}

	if (downloadStarted) {
		if (downloadCurrent > lastDownloadCurrent) {
			lastDownloadCurrent = downloadCurrent;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->downloadProgress(
			    AcntId(m_acntId.username(), m_acntId.testing()),
			     m_transactId, MsgId(), downloadTotal, downloadCurrent);
		}
	}
}
